package de.chiflux.utils;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Limit the number of requests to a configurable limit.
 */
public class ApiRateLimiter {

    private static final Logger LOGGER = Logger.getLogger(ApiRateLimiter.class.getName());

    private final AtomicReference<Instant> lastUsed = new AtomicReference<>(Instant.now());

    private final AtomicLong counter = new AtomicLong(0);

    private volatile int maxRequestsPerDuration;
    private volatile Duration duration = Duration.ofSeconds(5);
    private volatile Duration minWaitTimeBetweenRequests = Duration.ofSeconds(2);

    /**
     * Create a new ApiRateLimiter
     * @param maxRequestsPerDuration The max number of request per duration. See {@link #duration}
     * @param duration See {@link #maxRequestsPerDuration}
     * @param minWaitTimeBetweenRequests wait at least this duration between 2 requests
     */
    public ApiRateLimiter(int maxRequestsPerDuration, Duration duration, Duration minWaitTimeBetweenRequests) {
        setParameters(maxRequestsPerDuration, duration, minWaitTimeBetweenRequests);
    }

    /**
     * Try to get a permit.
     * @return false if there's no permit available.
     */
    synchronized boolean acquire() {
        long newCounter = counter.incrementAndGet();
        try {
            Thread.sleep(minWaitTimeBetweenRequests.toMillis());
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        if (lastUsed.get().isBefore(Instant.now().minus(duration)) && newCounter>maxRequestsPerDuration) {
            counter.set(0);
            lastUsed.set(Instant.now());
            //System.out.println("reset");
            return false;
        } else {
            return newCounter <= maxRequestsPerDuration;
        }
    }

    /**
     * Try to get a permit and wait until there is free permit.
     */
    public synchronized void acquireWait() {
        while (!acquire()) {
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Slow down the allowed rate by a specific percentage
     * @param percent the percentage to slow down the allowed rate
     */
    public synchronized void slowdown(int percent) {
        Duration newDuration = ((duration.multipliedBy(100 + percent).dividedBy(100)));
        Duration newMinWaitTimeBetweenRequests = ((minWaitTimeBetweenRequests.multipliedBy(100 + percent).dividedBy(100)));
        setParameters(maxRequestsPerDuration, newDuration, newMinWaitTimeBetweenRequests);
    }

    /**
     * Set parameters
     * @param maxRequestsPerDuration The max number of request per duration. See {@link #duration}
     * @param duration See {@link #maxRequestsPerDuration}
     * @param minWaitTimeBetweenRequests wait at least this duration between 2 requests
     */
    public synchronized void setParameters(int maxRequestsPerDuration, Duration duration, Duration minWaitTimeBetweenRequests) {
        this.maxRequestsPerDuration = maxRequestsPerDuration;
        this.duration = duration;
        this.minWaitTimeBetweenRequests = minWaitTimeBetweenRequests;
        LOGGER.log(Level.INFO, "Api Rate Limiter Params: " + maxRequestsPerDuration + " requests per " + duration + ". Min Wait time between requests: " + minWaitTimeBetweenRequests);
    }



}
