/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.collect.Iterables;
import de.cau.cs.kieler.klighd.kgraph.EMapPropertyHolder;
import de.cau.cs.kieler.klighd.kgraph.KGraphData;
import de.cau.cs.kieler.klighd.kgraph.KGraphElement;
import de.cau.cs.kieler.klighd.kgraph.KLabel;
import de.cau.cs.kieler.klighd.kgraph.KNode;
import de.cau.cs.kieler.klighd.kgraph.KPort;
import de.cau.cs.kieler.klighd.krendering.Colors;
import de.cau.cs.kieler.klighd.krendering.HorizontalAlignment;
import de.cau.cs.kieler.klighd.krendering.KAction;
import de.cau.cs.kieler.klighd.krendering.KArc;
import de.cau.cs.kieler.klighd.krendering.KAreaPlacementData;
import de.cau.cs.kieler.klighd.krendering.KBackground;
import de.cau.cs.kieler.klighd.krendering.KColor;
import de.cau.cs.kieler.klighd.krendering.KColoring;
import de.cau.cs.kieler.klighd.krendering.KContainerRendering;
import de.cau.cs.kieler.klighd.krendering.KDecoratorPlacementData;
import de.cau.cs.kieler.klighd.krendering.KEllipse;
import de.cau.cs.kieler.klighd.krendering.KFontBold;
import de.cau.cs.kieler.klighd.krendering.KFontItalic;
import de.cau.cs.kieler.klighd.krendering.KFontName;
import de.cau.cs.kieler.klighd.krendering.KFontSize;
import de.cau.cs.kieler.klighd.krendering.KForeground;
import de.cau.cs.kieler.klighd.krendering.KGridPlacement;
import de.cau.cs.kieler.klighd.krendering.KGridPlacementData;
import de.cau.cs.kieler.klighd.krendering.KHorizontalAlignment;
import de.cau.cs.kieler.klighd.krendering.KInvisibility;
import de.cau.cs.kieler.klighd.krendering.KLineCap;
import de.cau.cs.kieler.klighd.krendering.KLineJoin;
import de.cau.cs.kieler.klighd.krendering.KLineStyle;
import de.cau.cs.kieler.klighd.krendering.KLineWidth;
import de.cau.cs.kieler.klighd.krendering.KPlacementData;
import de.cau.cs.kieler.klighd.krendering.KPointPlacementData;
import de.cau.cs.kieler.klighd.krendering.KPolygon;
import de.cau.cs.kieler.klighd.krendering.KPolyline;
import de.cau.cs.kieler.klighd.krendering.KPosition;
import de.cau.cs.kieler.klighd.krendering.KRectangle;
import de.cau.cs.kieler.klighd.krendering.KRendering;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KRotation;
import de.cau.cs.kieler.klighd.krendering.KRoundedRectangle;
import de.cau.cs.kieler.klighd.krendering.KShadow;
import de.cau.cs.kieler.klighd.krendering.KStyle;
import de.cau.cs.kieler.klighd.krendering.KStyleHolder;
import de.cau.cs.kieler.klighd.krendering.KStyleRef;
import de.cau.cs.kieler.klighd.krendering.KText;
import de.cau.cs.kieler.klighd.krendering.KTextStrikeout;
import de.cau.cs.kieler.klighd.krendering.KTextUnderline;
import de.cau.cs.kieler.klighd.krendering.KVerticalAlignment;
import de.cau.cs.kieler.klighd.krendering.KXPosition;
import de.cau.cs.kieler.klighd.krendering.KYPosition;
import de.cau.cs.kieler.klighd.krendering.LineCap;
import de.cau.cs.kieler.klighd.krendering.LineJoin;
import de.cau.cs.kieler.klighd.krendering.LineStyle;
import de.cau.cs.kieler.klighd.krendering.ModifierState;
import de.cau.cs.kieler.klighd.krendering.Trigger;
import de.cau.cs.kieler.klighd.krendering.Underline;
import de.cau.cs.kieler.klighd.krendering.VerticalAlignment;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * This class contains lots of convenient helper functions for configuring KRendering-based view models,
 * and it does not claim to be complete ;-).<br>
 * In order to be consistent with the further extension classes the extension methods are non-static
 * ones requiring this class to be instantiated. Since this class doesn't declare any fields (i.e.
 * required memory) the instantiation should not be a problem. The instantiation may be done directly
 * by calling 'new KRenderingExtensions()' or by delegating that to a dependency injection framework.<br>
 * <br>
 * NOTE: Do NOT introduce <i>create extensions</i> or other continuous memory in that class!
 * 
 * @containsExtensions
 */
@SuppressWarnings("all")
public class KRenderingExtensions {
  @Extension
  private KRenderingFactory _kRenderingFactory = KRenderingFactory.eINSTANCE;

  /**
   * Reveals the KRendering of a KGraphElement, e.g. for the purpose of highlighting.
   */
  public KRendering getKRendering(final KGraphElement kge) {
    return kge.<KRendering>getData(KRendering.class);
  }

  public KContainerRendering getKContainerRendering(final KGraphElement kge) {
    return kge.<KContainerRendering>getData(KContainerRendering.class);
  }

  protected KContainerRendering _addInvisibleContainerRendering(final KNode node) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      this.setInvisible(it, true);
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  protected KContainerRendering _addInvisibleContainerRendering(final KPort port) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      this.setInvisible(it, true);
      EList<KGraphData> _data = port.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  protected KContainerRendering _addInvisibleContainerRendering(final KLabel label) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      this.setInvisible(it, true);
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  protected KEllipse _addEllipse(final KNode node) {
    KEllipse _createKEllipse = this._kRenderingFactory.createKEllipse();
    final Procedure1<KEllipse> _function = (KEllipse it) -> {
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KEllipse>operator_doubleArrow(_createKEllipse, _function);
  }

  protected KEllipse _addEllipse(final KPort port) {
    KEllipse _createKEllipse = this._kRenderingFactory.createKEllipse();
    final Procedure1<KEllipse> _function = (KEllipse it) -> {
      EList<KGraphData> _data = port.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KEllipse>operator_doubleArrow(_createKEllipse, _function);
  }

  protected KEllipse _addEllipse(final KLabel label) {
    KEllipse _createKEllipse = this._kRenderingFactory.createKEllipse();
    final Procedure1<KEllipse> _function = (KEllipse it) -> {
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KEllipse>operator_doubleArrow(_createKEllipse, _function);
  }

  protected KArc _addArc(final KNode node) {
    KArc _createKArc = this._kRenderingFactory.createKArc();
    final Procedure1<KArc> _function = (KArc it) -> {
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KArc>operator_doubleArrow(_createKArc, _function);
  }

  protected KArc _addArc(final KPort port) {
    KArc _createKArc = this._kRenderingFactory.createKArc();
    final Procedure1<KArc> _function = (KArc it) -> {
      EList<KGraphData> _data = port.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KArc>operator_doubleArrow(_createKArc, _function);
  }

  protected KArc _addArc(final KLabel label) {
    KArc _createKArc = this._kRenderingFactory.createKArc();
    final Procedure1<KArc> _function = (KArc it) -> {
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KArc>operator_doubleArrow(_createKArc, _function);
  }

  protected KPolygon _addPolygon(final KNode node) {
    KPolygon _createKPolygon = this._kRenderingFactory.createKPolygon();
    final Procedure1<KPolygon> _function = (KPolygon it) -> {
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KPolygon>operator_doubleArrow(_createKPolygon, _function);
  }

  protected KPolygon _addPolygon(final KPort port) {
    KPolygon _createKPolygon = this._kRenderingFactory.createKPolygon();
    final Procedure1<KPolygon> _function = (KPolygon it) -> {
      EList<KGraphData> _data = port.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KPolygon>operator_doubleArrow(_createKPolygon, _function);
  }

  /**
   * Adds a rectangle to the node.
   * @example
   * node.addRectangle =&gt; [
   *   it.background = "green".color
   * ]
   * 
   * @extensionCategory composition
   */
  protected KRectangle _addRectangle(final KNode node) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  protected KRectangle _addRectangle(final KPort port) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      EList<KGraphData> _data = port.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  protected KRectangle _addRectangle(final KLabel label) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  /**
   * @extensionCategory composition
   */
  protected KRoundedRectangle _addRoundedRectangle(final KNode node, final float cWidth, final float cHeight) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  protected KRoundedRectangle _addRoundedRectangle(final KLabel label, final float cWidth, final float cHeight) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  protected KRoundedRectangle _addRoundedRectangle(final KNode node, final float cWidth, final float cHeight, final float lineWidth) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
      this.<KRoundedRectangle>setLineWidth(it, lineWidth);
      EList<KGraphData> _data = node.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  protected KRoundedRectangle _addRoundedRectangle(final KLabel label, final float cWidth, final float cHeight, final float lineWidth) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
      this.<KRoundedRectangle>setLineWidth(it, lineWidth);
      EList<KGraphData> _data = label.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  public KRoundedRectangle setCornerSize(final KRoundedRectangle rect, final float cWidth, final float cHeight) {
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(rect, _function);
  }

  public KText addText(final KNode node, final String text) {
    KText _createKText = this._kRenderingFactory.createKText();
    final Procedure1<KText> _function = (KText it) -> {
      EList<KGraphData> _data = node.getData();
      _data.add(it);
      it.setText(text);
    };
    return ObjectExtensions.<KText>operator_doubleArrow(_createKText, _function);
  }

  public <T extends KRendering> T addAction(final T rendering, final Trigger trigger, final String actionId) {
    EList<KAction> _actions = rendering.getActions();
    KAction _createKAction = this._kRenderingFactory.createKAction();
    final Procedure1<KAction> _function = (KAction it) -> {
      it.setTrigger(trigger);
      it.setActionId(actionId);
    };
    KAction _doubleArrow = ObjectExtensions.<KAction>operator_doubleArrow(_createKAction, _function);
    _actions.add(_doubleArrow);
    return rendering;
  }

  /**
   * @deprecated Use {@link #addAction(KRendering, Trigger, String, ModifierState, ModifierState, ModifierState)} instead.
   */
  public <T extends KRendering> T addAction(final T rendering, final Trigger trigger, final String actionId, final boolean altPressed, final boolean ctrlCmdPressed, final boolean shiftPressed) {
    EList<KAction> _actions = rendering.getActions();
    KAction _createKAction = this._kRenderingFactory.createKAction();
    final Procedure1<KAction> _function = (KAction it) -> {
      it.setTrigger(trigger);
      it.setActionId(actionId);
      if (altPressed) {
        it.setAltPressed(ModifierState.PRESSED);
      } else {
        it.setAltPressed(ModifierState.DONT_CARE);
      }
      if (ctrlCmdPressed) {
        it.setCtrlCmdPressed(ModifierState.PRESSED);
      } else {
        it.setCtrlCmdPressed(ModifierState.DONT_CARE);
      }
      if (shiftPressed) {
        it.setShiftPressed(ModifierState.PRESSED);
      } else {
        it.setShiftPressed(ModifierState.DONT_CARE);
      }
    };
    KAction _doubleArrow = ObjectExtensions.<KAction>operator_doubleArrow(_createKAction, _function);
    _actions.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addAction(final T rendering, final Trigger trigger, final String actionId, final ModifierState altPressed, final ModifierState ctrlCmdPressed, final ModifierState shiftPressed) {
    EList<KAction> _actions = rendering.getActions();
    KAction _createKAction = this._kRenderingFactory.createKAction();
    final Procedure1<KAction> _function = (KAction it) -> {
      it.setTrigger(trigger);
      it.setActionId(actionId);
      it.setAltPressed(altPressed);
      it.setCtrlCmdPressed(ctrlCmdPressed);
      it.setShiftPressed(shiftPressed);
    };
    KAction _doubleArrow = ObjectExtensions.<KAction>operator_doubleArrow(_createKAction, _function);
    _actions.add(_doubleArrow);
    return rendering;
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single (!) click on <code>rendering</code>. The action triggered with a delay if and
   * only if no subsequent click occurs within the system wide double click time.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addSingleClickAction(final T rendering, final String actionId) {
    return this.<T>addAction(rendering, Trigger.SINGLECLICK, actionId);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single (!) click on <code>rendering</code>. The action triggered with a delay if and
   * only if no subsequent click occurs within the system wide double click time, and if the
   * required modifier keys are pressed. If the modifiers here are false, it is ignored if they are pressed.
   * 
   * @extensionCategory actions
   * @deprecated Use {@link #addSingleClickAction(KRendering, String, ModifierState, ModifierState, ModifierState)}
   * instead.
   */
  public <T extends KRendering> T addSingleClickAction(final T rendering, final String actionId, final boolean altPressed, final boolean ctrlCmdPressed, final boolean shiftPressed) {
    return this.<T>addAction(rendering, Trigger.SINGLECLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single (!) click on <code>rendering</code>. The action triggered with a delay if and
   * only if no subsequent click occurs within the system wide double click time, and if the
   * modifier keys are in the given state.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addSingleClickAction(final T rendering, final String actionId, final ModifierState altPressed, final ModifierState ctrlCmdPressed, final ModifierState shiftPressed) {
    return this.<T>addAction(rendering, Trigger.SINGLECLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a double (or triple, ...) click on <code>rendering</code>.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addDoubleClickAction(final T rendering, final String actionId) {
    return this.<T>addAction(rendering, Trigger.DOUBLECLICK, actionId);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a double (or triple, ...) click on <code>rendering</code> if the required modifier
   * keys are pressed. If the modifiers here are false, it is ignored if they are pressed.
   * 
   * @extensionCategory actions
   * @deprecated Use {@link #addDoubleClickAction(KRendering, String, ModifierState, ModifierState, ModifierState)}
   * instead.
   */
  public <T extends KRendering> T addDoubleClickAction(final T rendering, final String actionId, final boolean altPressed, final boolean ctrlCmdPressed, final boolean shiftPressed) {
    return this.<T>addAction(rendering, Trigger.DOUBLECLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a double (or triple, ...) click on <code>rendering</code> if the modifier keys are in the given state.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addDoubleClickAction(final T rendering, final String actionId, final ModifierState altPressed, final ModifierState ctrlCmdPressed, final ModifierState shiftPressed) {
    return this.<T>addAction(rendering, Trigger.DOUBLECLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single click on <code>rendering</code> regardless of whether more clicks follow
   * within the system wide double click time. In contrast to #addSingleClickAction, there's
   * no delay between the event receipt and the action execution.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addSingleOrMultiClickAction(final T rendering, final String actionId) {
    return this.<T>addAction(rendering, Trigger.SINGLE_OR_MULTICLICK, actionId);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single click on <code>rendering</code> regardless of whether more clicks follow
   * within the system wide double click time, if the required modifier keys are pressed.
   * In contrast to #addSingleClickAction, there's no delay between the event receipt and the
   * action execution.
   * 
   * @extensionCategory actions
   * @deprecated Use {@link #addSingleOrMultiClickAction(KRendering, String, ModifierState, ModifierState, ModifierState)}
   * instead.
   */
  public <T extends KRendering> T addSingleOrMultiClickAction(final T rendering, final String actionId, final boolean altPressed, final boolean ctrlCmdPressed, final boolean shiftPressed) {
    return this.<T>addAction(rendering, Trigger.SINGLE_OR_MULTICLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  /**
   * Configures the action being registered in KLighD with <code>actionId</code> to be executed
   * after a single click on <code>rendering</code> regardless of whether more clicks follow
   * within the system wide double click time, if the modifier keys are in the given state.
   * In contrast to #addSingleClickAction, there's no delay between the event receipt and the
   * action execution.
   * 
   * @extensionCategory actions
   */
  public <T extends KRendering> T addSingleOrMultiClickAction(final T rendering, final String actionId, final ModifierState altPressed, final ModifierState ctrlCmdPressed, final ModifierState shiftPressed) {
    return this.<T>addAction(rendering, Trigger.SINGLE_OR_MULTICLICK, actionId, altPressed, ctrlCmdPressed, shiftPressed);
  }

  public <T extends KRendering> T with(final T rendering, final KPlacementData pd) {
    rendering.setPlacementData(pd);
    return rendering;
  }

  public <T extends KRendering> T with(final T rendering, final KStyle style) {
    EList<KStyle> _styles = rendering.getStyles();
    _styles.add(style);
    return rendering;
  }

  public <T extends KRendering> T withCopyOf(final T rendering, final KStyle style) {
    EList<KStyle> _styles = rendering.getStyles();
    KStyle _copy = EcoreUtil.<KStyle>copy(style);
    _styles.add(_copy);
    return rendering;
  }

  private static final Function1<? super KStyle, ? extends Boolean> IS_SELECTION = ((Function1<KStyle, Boolean>) (KStyle it) -> {
    return Boolean.valueOf(it.isSelection());
  });

  private static final Procedure1<? super KStyle> SET_SELECTION = ((Procedure1<KStyle>) (KStyle it) -> {
    it.setSelection(true);
  });

  public <T extends KRendering> T setStyleRef(final T rendering, final KStyleHolder styleHolder) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(rendering.getStyles(), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionStyleRef(final T rendering, final KStyleHolder styleHolder) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setStyleRef(final T rendering, final KStyleHolder styleHolder, final Class<KStyle>... styleTypes) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(rendering.getStyles(), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<? extends Class<KStyle>>)Conversions.doWrapArray(styleTypes)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionStyleRef(final T rendering, final KStyleHolder styleHolder, final Class<KStyle>... styleTypes) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<? extends Class<KStyle>>)Conversions.doWrapArray(styleTypes)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setStyleRef(final T rendering, final KStyleHolder styleHolder, final EClass... styleTypes) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(rendering.getStyles(), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      final Function1<EClass, Class<?>> _function_1 = (EClass it_1) -> {
        return it_1.getInstanceClass();
      };
      final Function1<Class<?>, Boolean> _function_2 = (Class<?> it_1) -> {
        return Boolean.valueOf(KStyle.class.isAssignableFrom(it_1));
      };
      Iterable<Class<?>> _filter = IterableExtensions.<Class<?>>filter(ListExtensions.<EClass, Class<?>>map(((List<EClass>)Conversions.doWrapArray(styleTypes)), _function_1), _function_2);
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<Class<KStyle>>) ((Iterable<?>) _filter)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionStyleRef(final T rendering, final KStyleHolder styleHolder, final EClass... styleTypes) {
    rendering.getStyles().removeAll(IterableExtensions.<KStyleRef>toList(Iterables.<KStyleRef>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KStyleRef.class)));
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      final Function1<EClass, Class<?>> _function_1 = (EClass it_1) -> {
        return it_1.getInstanceClass();
      };
      final Function1<Class<?>, Boolean> _function_2 = (Class<?> it_1) -> {
        return Boolean.valueOf(KStyle.class.isAssignableFrom(it_1));
      };
      Iterable<Class<?>> _filter = IterableExtensions.<Class<?>>filter(ListExtensions.<EClass, Class<?>>map(((List<EClass>)Conversions.doWrapArray(styleTypes)), _function_1), _function_2);
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<Class<KStyle>>) ((Iterable<?>) _filter)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addStyleRef(final T rendering, final KStyleHolder styleHolder) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addSelectionStyleRef(final T rendering, final KStyleHolder styleHolder) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addStyleRef(final T rendering, final KStyleHolder styleHolder, final Class<KStyle>... styleTypes) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<? extends Class<KStyle>>)Conversions.doWrapArray(styleTypes)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addSelectionStyleRef(final T rendering, final KStyleHolder styleHolder, final Class<KStyle>... styleTypes) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<? extends Class<KStyle>>)Conversions.doWrapArray(styleTypes)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addStyleRef(final T rendering, final KStyleHolder styleHolder, final EClass... styleTypes) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      final Function1<EClass, Class<?>> _function_1 = (EClass it_1) -> {
        return it_1.getInstanceClass();
      };
      final Function1<Class<?>, Boolean> _function_2 = (Class<?> it_1) -> {
        return Boolean.valueOf(KStyle.class.isAssignableFrom(it_1));
      };
      Iterable<Class<?>> _filter = IterableExtensions.<Class<?>>filter(ListExtensions.<EClass, Class<?>>map(((List<EClass>)Conversions.doWrapArray(styleTypes)), _function_1), _function_2);
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<Class<KStyle>>) ((Iterable<?>) _filter)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T addSelectionStyleRef(final T rendering, final KStyleHolder styleHolder, final EClass... styleTypes) {
    if ((styleHolder == null)) {
      return rendering;
    }
    EList<KStyle> _styles = rendering.getStyles();
    KStyleRef _createKStyleRef = this._kRenderingFactory.createKStyleRef();
    final Procedure1<KStyleRef> _function = (KStyleRef it) -> {
      it.setSelection(true);
      it.setStyleHolder(styleHolder);
      EList<Class<KStyle>> _referencedTypes = it.getReferencedTypes();
      final Function1<EClass, Class<?>> _function_1 = (EClass it_1) -> {
        return it_1.getInstanceClass();
      };
      final Function1<Class<?>, Boolean> _function_2 = (Class<?> it_1) -> {
        return Boolean.valueOf(KStyle.class.isAssignableFrom(it_1));
      };
      Iterable<Class<?>> _filter = IterableExtensions.<Class<?>>filter(ListExtensions.<EClass, Class<?>>map(((List<EClass>)Conversions.doWrapArray(styleTypes)), _function_1), _function_2);
      Iterables.<Class<KStyle>>addAll(_referencedTypes, ((Iterable<Class<KStyle>>) ((Iterable<?>) _filter)));
    };
    KStyleRef _doubleArrow = ObjectExtensions.<KStyleRef>operator_doubleArrow(_createKStyleRef, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KInvisibility getInvisible(final KRendering rendering) {
    KInvisibility _elvis = null;
    KInvisibility _last = IterableExtensions.<KInvisibility>last(Iterables.<KInvisibility>filter(rendering.getStyles(), KInvisibility.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KInvisibility _createKInvisibility = this._kRenderingFactory.createKInvisibility();
      _elvis = _createKInvisibility;
    }
    return _elvis;
  }

  public boolean getInvisibleValue(final KRendering rendering) {
    KInvisibility _elvis = null;
    KInvisibility _last = IterableExtensions.<KInvisibility>last(Iterables.<KInvisibility>filter(rendering.getStyles(), KInvisibility.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KInvisibility _createKInvisibility = this._kRenderingFactory.createKInvisibility();
      _elvis = _createKInvisibility;
    }
    return _elvis.isInvisible();
  }

  public <T extends KRendering> T setInvisible(final T rendering, final boolean invisible) {
    rendering.getStyles().removeAll(IterableExtensions.<KInvisibility>toList(Iterables.<KInvisibility>filter(rendering.getStyles(), KInvisibility.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KInvisibility _createKInvisibility = this._kRenderingFactory.createKInvisibility();
    final Procedure1<KInvisibility> _function = (KInvisibility it) -> {
      it.setInvisible(invisible);
    };
    KInvisibility _doubleArrow = ObjectExtensions.<KInvisibility>operator_doubleArrow(_createKInvisibility, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionInvisible(final T rendering, final boolean invisible) {
    rendering.getStyles().removeAll(IterableExtensions.<KInvisibility>toList(Iterables.<KInvisibility>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KInvisibility.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KInvisibility _createKInvisibility = this._kRenderingFactory.createKInvisibility();
    final Procedure1<KInvisibility> _function = (KInvisibility it) -> {
      it.setSelection(true);
      it.setInvisible(invisible);
    };
    KInvisibility _doubleArrow = ObjectExtensions.<KInvisibility>operator_doubleArrow(_createKInvisibility, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KLineWidth getLineWidth(final KRendering rendering) {
    KLineWidth _elvis = null;
    KLineWidth _last = IterableExtensions.<KLineWidth>last(Iterables.<KLineWidth>filter(rendering.getStyles(), KLineWidth.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineWidth _createKLineWidth = this._kRenderingFactory.createKLineWidth();
      final Procedure1<KLineWidth> _function = (KLineWidth it) -> {
        it.setLineWidth(1f);
      };
      KLineWidth _doubleArrow = ObjectExtensions.<KLineWidth>operator_doubleArrow(_createKLineWidth, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public float getLineWidthValue(final KRendering rendering) {
    KLineWidth _elvis = null;
    KLineWidth _last = IterableExtensions.<KLineWidth>last(Iterables.<KLineWidth>filter(rendering.getStyles(), KLineWidth.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineWidth _createKLineWidth = this._kRenderingFactory.createKLineWidth();
      final Procedure1<KLineWidth> _function = (KLineWidth it) -> {
        it.setLineWidth(1f);
      };
      KLineWidth _doubleArrow = ObjectExtensions.<KLineWidth>operator_doubleArrow(_createKLineWidth, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getLineWidth();
  }

  /**
   * @extensionCategory style
   */
  public <T extends KRendering> T setLineWidth(final T rendering, final float width) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineWidth>toList(Iterables.<KLineWidth>filter(rendering.getStyles(), KLineWidth.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineWidth _createKLineWidth = this._kRenderingFactory.createKLineWidth();
    final Procedure1<KLineWidth> _function = (KLineWidth it) -> {
      it.setLineWidth(width);
    };
    KLineWidth _doubleArrow = ObjectExtensions.<KLineWidth>operator_doubleArrow(_createKLineWidth, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionLineWidth(final T rendering, final float width) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineWidth>toList(Iterables.<KLineWidth>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KLineWidth.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineWidth _createKLineWidth = this._kRenderingFactory.createKLineWidth();
    final Procedure1<KLineWidth> _function = (KLineWidth it) -> {
      it.setSelection(true);
      it.setLineWidth(width);
    };
    KLineWidth _doubleArrow = ObjectExtensions.<KLineWidth>operator_doubleArrow(_createKLineWidth, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  /**
   * @extensionCategory style
   */
  public KLineStyle getLineStyle(final KRendering rendering) {
    KLineStyle _elvis = null;
    KLineStyle _last = IterableExtensions.<KLineStyle>last(Iterables.<KLineStyle>filter(rendering.getStyles(), KLineStyle.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineStyle _createKLineStyle = this._kRenderingFactory.createKLineStyle();
      final Procedure1<KLineStyle> _function = (KLineStyle it) -> {
        it.setLineStyle(LineStyle.SOLID);
      };
      KLineStyle _doubleArrow = ObjectExtensions.<KLineStyle>operator_doubleArrow(_createKLineStyle, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public LineStyle getLineStyleValue(final KRendering rendering) {
    KLineStyle _elvis = null;
    KLineStyle _last = IterableExtensions.<KLineStyle>last(Iterables.<KLineStyle>filter(rendering.getStyles(), KLineStyle.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineStyle _createKLineStyle = this._kRenderingFactory.createKLineStyle();
      final Procedure1<KLineStyle> _function = (KLineStyle it) -> {
        it.setLineStyle(LineStyle.SOLID);
      };
      KLineStyle _doubleArrow = ObjectExtensions.<KLineStyle>operator_doubleArrow(_createKLineStyle, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getLineStyle();
  }

  public <T extends KRendering> T setLineStyle(final T rendering, final LineStyle style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineStyle>toList(Iterables.<KLineStyle>filter(rendering.getStyles(), KLineStyle.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineStyle _createKLineStyle = this._kRenderingFactory.createKLineStyle();
    final Procedure1<KLineStyle> _function = (KLineStyle it) -> {
      it.setLineStyle(style);
    };
    KLineStyle _doubleArrow = ObjectExtensions.<KLineStyle>operator_doubleArrow(_createKLineStyle, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionLineStyle(final T rendering, final LineStyle style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineStyle>toList(Iterables.<KLineStyle>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KLineStyle.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineStyle _createKLineStyle = this._kRenderingFactory.createKLineStyle();
    final Procedure1<KLineStyle> _function = (KLineStyle it) -> {
      it.setSelection(true);
      it.setLineStyle(style);
    };
    KLineStyle _doubleArrow = ObjectExtensions.<KLineStyle>operator_doubleArrow(_createKLineStyle, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KLineCap getLineCap(final KRendering rendering) {
    KLineCap _elvis = null;
    KLineCap _last = IterableExtensions.<KLineCap>last(Iterables.<KLineCap>filter(rendering.getStyles(), KLineCap.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineCap _createKLineCap = this._kRenderingFactory.createKLineCap();
      final Procedure1<KLineCap> _function = (KLineCap it) -> {
        it.setLineCap(LineCap.CAP_FLAT);
      };
      KLineCap _doubleArrow = ObjectExtensions.<KLineCap>operator_doubleArrow(_createKLineCap, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public LineCap getLineCapValue(final KRendering rendering) {
    KLineCap _elvis = null;
    KLineCap _last = IterableExtensions.<KLineCap>last(Iterables.<KLineCap>filter(rendering.getStyles(), KLineCap.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineCap _createKLineCap = this._kRenderingFactory.createKLineCap();
      final Procedure1<KLineCap> _function = (KLineCap it) -> {
        it.setLineCap(LineCap.CAP_FLAT);
      };
      KLineCap _doubleArrow = ObjectExtensions.<KLineCap>operator_doubleArrow(_createKLineCap, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getLineCap();
  }

  public <T extends KRendering> T setLineCap(final T rendering, final LineCap style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineCap>toList(Iterables.<KLineCap>filter(rendering.getStyles(), KLineCap.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineCap _createKLineCap = this._kRenderingFactory.createKLineCap();
    final Procedure1<KLineCap> _function = (KLineCap it) -> {
      it.setLineCap(style);
    };
    KLineCap _doubleArrow = ObjectExtensions.<KLineCap>operator_doubleArrow(_createKLineCap, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionLineCap(final T rendering, final LineCap style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineCap>toList(Iterables.<KLineCap>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KLineCap.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineCap _createKLineCap = this._kRenderingFactory.createKLineCap();
    final Procedure1<KLineCap> _function = (KLineCap it) -> {
      it.setSelection(true);
      it.setLineCap(style);
    };
    KLineCap _doubleArrow = ObjectExtensions.<KLineCap>operator_doubleArrow(_createKLineCap, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KLineJoin getLineJoin(final KRendering rendering) {
    KLineJoin _elvis = null;
    KLineJoin _last = IterableExtensions.<KLineJoin>last(Iterables.<KLineJoin>filter(rendering.getStyles(), KLineJoin.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
      final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
        it.setLineJoin(LineJoin.JOIN_MITER);
      };
      KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public LineJoin getLineJoinValue(final KRendering rendering) {
    KLineJoin _elvis = null;
    KLineJoin _last = IterableExtensions.<KLineJoin>last(Iterables.<KLineJoin>filter(rendering.getStyles(), KLineJoin.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
      final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
        it.setLineJoin(LineJoin.JOIN_MITER);
      };
      KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getLineJoin();
  }

  public <T extends KRendering> T setLineJoin(final T rendering, final LineJoin style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineJoin>toList(Iterables.<KLineJoin>filter(rendering.getStyles(), KLineJoin.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
    final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
      it.setLineJoin(style);
    };
    KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionLineJoin(final T rendering, final LineJoin style) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineJoin>toList(Iterables.<KLineJoin>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KLineJoin.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
    final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
      it.setSelection(true);
      it.setLineJoin(style);
    };
    KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setLineJoin(final T rendering, final LineJoin style, final float miterLimit) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineJoin>toList(Iterables.<KLineJoin>filter(rendering.getStyles(), KLineJoin.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
    final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
      it.setLineJoin(style);
      it.setMiterLimit(miterLimit);
    };
    KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionLineJoin(final T rendering, final LineJoin style, final float miterLimit) {
    rendering.getStyles().removeAll(IterableExtensions.<KLineJoin>toList(Iterables.<KLineJoin>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KLineJoin.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KLineJoin _createKLineJoin = this._kRenderingFactory.createKLineJoin();
    final Procedure1<KLineJoin> _function = (KLineJoin it) -> {
      it.setSelection(true);
      it.setLineJoin(style);
      it.setMiterLimit(miterLimit);
    };
    KLineJoin _doubleArrow = ObjectExtensions.<KLineJoin>operator_doubleArrow(_createKLineJoin, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KRotation getRotation(final KRendering rendering) {
    KRotation _elvis = null;
    KRotation _last = IterableExtensions.<KRotation>last(Iterables.<KRotation>filter(rendering.getStyles(), KRotation.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KRotation _createKRotation = this._kRenderingFactory.createKRotation();
      _elvis = _createKRotation;
    }
    return _elvis;
  }

  public float getRotationValue(final KRendering rendering) {
    KRotation _elvis = null;
    KRotation _last = IterableExtensions.<KRotation>last(Iterables.<KRotation>filter(rendering.getStyles(), KRotation.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KRotation _createKRotation = this._kRenderingFactory.createKRotation();
      _elvis = _createKRotation;
    }
    return _elvis.getRotation();
  }

  public <T extends KRendering> T setRotation(final T rendering, final Float rotation) {
    rendering.getStyles().removeAll(IterableExtensions.<KRotation>toList(Iterables.<KRotation>filter(rendering.getStyles(), KRotation.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KRotation _createKRotation = this._kRenderingFactory.createKRotation();
    final Procedure1<KRotation> _function = (KRotation it) -> {
      it.setRotation((rotation).floatValue());
    };
    KRotation _doubleArrow = ObjectExtensions.<KRotation>operator_doubleArrow(_createKRotation, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionRotation(final T rendering, final Float rotation) {
    rendering.getStyles().removeAll(IterableExtensions.<KRotation>toList(Iterables.<KRotation>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KRotation.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KRotation _createKRotation = this._kRenderingFactory.createKRotation();
    final Procedure1<KRotation> _function = (KRotation it) -> {
      it.setSelection(true);
      it.setRotation((rotation).floatValue());
    };
    KRotation _doubleArrow = ObjectExtensions.<KRotation>operator_doubleArrow(_createKRotation, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KBackground getBackground(final KRendering rendering) {
    KBackground _elvis = null;
    EList<KStyle> _styles = rendering.getStyles();
    Iterable<KBackground> _filter = null;
    if (_styles!=null) {
      _filter=Iterables.<KBackground>filter(_styles, KBackground.class);
    }
    KBackground _last = IterableExtensions.<KBackground>last(_filter);
    if (_last != null) {
      _elvis = _last;
    } else {
      KBackground _createKBackground = this._kRenderingFactory.createKBackground();
      final Procedure1<KBackground> _function = (KBackground it) -> {
        it.setColor(this._kRenderingFactory.createKColor());
      };
      KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_createKBackground, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public <T extends KRendering> T setBackground(final T rendering, final KColoring<?> coloring) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setGradientAngle2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(coloring.getColor()), coloring.getAlpha()).setTargetColor2(EcoreUtil.<KColor>copy(coloring.getTargetColor()), coloring.getTargetAlpha()).setGradientAngle2(coloring.getGradientAngle());
    final Procedure1<KBackground> _function = (KBackground it) -> {
      it.setModifierId(coloring.getModifierId());
      it.setPropagateToChildren(coloring.isPropagateToChildren());
    };
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setGradientAngle2, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackground(final T rendering, final KColoring<?> coloring) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setGradientAngle2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(coloring.getColor()), coloring.getAlpha()).setTargetColor2(EcoreUtil.<KColor>copy(coloring.getTargetColor()), coloring.getTargetAlpha()).setGradientAngle2(coloring.getGradientAngle());
    final Procedure1<KBackground> _function = (KBackground it) -> {
      it.setModifierId(coloring.getModifierId());
      it.setPropagateToChildren(coloring.isPropagateToChildren());
      it.setSelection(true);
    };
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setGradientAngle2, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackground(final T rendering, final KColor color) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(color));
    _styles.add(_setColor2);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackground(final T rendering, final KColor color) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(color));
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor2, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackground(final T rendering, final KColor color, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(color), alpha);
    _styles.add(_setColor2);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackground(final T rendering, final KColor color, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor2 = this._kRenderingFactory.createKBackground().setColor2(EcoreUtil.<KColor>copy(color), alpha);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor2, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackground(final T rendering, final Colors color) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(color);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackground(final T rendering, final Colors color) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(color);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackground(final T rendering, final Colors color, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(color, alpha);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackground(final T rendering, final Colors color, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(color, alpha);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundColor(final T rendering, final int red, final int green, final int blue) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(red, green, blue);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundColor(final T rendering, final int red, final int green, final int blue) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(red, green, blue);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundColor(final T rendering, final int red, final int green, final int blue, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(red, green, blue, alpha);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundColor(final T rendering, final int red, final int green, final int blue, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColor = this._kRenderingFactory.createKBackground().setColor(red, green, blue, alpha);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundGradient(final T rendering, final KColor color1, final KColor color2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, color2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundGradient(final T rendering, final KColor color1, final KColor color2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, color2, angle);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundGradient(final T rendering, final KColor color1, final int alpha1, final KColor color2, final int alpha2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, alpha1, color2, alpha2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundGradient(final T rendering, final KColor color1, final int alpha1, final KColor color2, final int alpha2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, alpha1, color2, alpha2, angle);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundGradient(final T rendering, final Colors color1, final Colors color2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, color2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundGradient(final T rendering, final Colors color1, final Colors color2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, color2, angle);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundGradient(final T rendering, final Colors color1, final int alpha1, final Colors color2, final int alpha2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KBackground>toList(Iterables.<KBackground>filter(rendering.getStyles(), KBackground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, alpha1, color2, alpha2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionBackgroundGradient(final T rendering, final Colors color1, final int alpha1, final Colors color2, final int alpha2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KBackground _setColors = this._kRenderingFactory.createKBackground().setColors(color1, alpha1, color2, alpha2, angle);
    KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setBackgroundInvisible(final T rendering, final boolean invisible) {
    final Procedure1<T> _function = (T it) -> {
      KBackground _elvis = null;
      EList<KStyle> _styles = it.getStyles();
      Iterable<KBackground> _filter = null;
      if (_styles!=null) {
        _filter=Iterables.<KBackground>filter(_styles, KBackground.class);
      }
      KBackground _last = IterableExtensions.<KBackground>last(_filter);
      if (_last != null) {
        _elvis = _last;
      } else {
        KBackground _createKBackground = this._kRenderingFactory.createKBackground();
        final Procedure1<KBackground> _function_1 = (KBackground it_1) -> {
          EList<KStyle> _styles_1 = rendering.getStyles();
          _styles_1.add(it_1);
          it_1.setColor(this._kRenderingFactory.createKColor());
        };
        KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_createKBackground, _function_1);
        _elvis = _doubleArrow;
      }
      final Procedure1<KBackground> _function_2 = (KBackground it_1) -> {
        Integer _xifexpression = null;
        if (invisible) {
          _xifexpression = Integer.valueOf(0);
        } else {
          Object _defaultValue = this._kRenderingFactory.getKRenderingPackage().getKColoring_Alpha().getDefaultValue();
          _xifexpression = ((Integer) _defaultValue);
        }
        it_1.setAlpha((_xifexpression).intValue());
      };
      ObjectExtensions.<KBackground>operator_doubleArrow(_elvis, _function_2);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  public KForeground getForeground(final KRendering rendering) {
    KForeground _elvis = null;
    KForeground _last = IterableExtensions.<KForeground>last(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KForeground _createKForeground = this._kRenderingFactory.createKForeground();
      final Procedure1<KForeground> _function = (KForeground it) -> {
        it.setColor(this._kRenderingFactory.createKColor());
      };
      KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_createKForeground, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public <T extends KRendering> T setForeground(final T rendering, final KColoring<?> coloring) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(
      EcoreUtil.<KColor>copy(coloring.getColor()), coloring.getAlpha(), EcoreUtil.<KColor>copy(coloring.getTargetColor()), coloring.getTargetAlpha(), coloring.getGradientAngle());
    final Procedure1<KForeground> _function = (KForeground it) -> {
      it.setModifierId(coloring.getModifierId());
      it.setPropagateToChildren(coloring.isPropagateToChildren());
    };
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColors, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForeground(final T rendering, final KColor color) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor2 = this._kRenderingFactory.createKForeground().setColor2(color);
    _styles.add(_setColor2);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForeground(final T rendering, final KColor color) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor2 = this._kRenderingFactory.createKForeground().setColor2(color);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor2, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForeground(final T rendering, final KColor color, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor2 = this._kRenderingFactory.createKForeground().setColor2(color, alpha);
    _styles.add(_setColor2);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForeground(final T rendering, final KColor color, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor2 = this._kRenderingFactory.createKForeground().setColor2(color, alpha);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor2, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForeground(final T rendering, final Colors color) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(color);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForeground(final T rendering, final Colors color) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(color);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForeground(final T rendering, final Colors color, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(color, alpha);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForeground(final T rendering, final Colors color, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(color, alpha);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundColor(final T rendering, final int red, final int green, final int blue) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(red, green, blue);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundColor(final T rendering, final int red, final int green, final int blue) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(red, green, blue);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundColor(final T rendering, final int red, final int green, final int blue, final int alpha) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(red, green, blue, alpha);
    _styles.add(_setColor);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundColor(final T rendering, final int red, final int green, final int blue, final int alpha) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColor = this._kRenderingFactory.createKForeground().setColor(red, green, blue, alpha);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColor, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundGradient(final T rendering, final KColor color1, final KColor color2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, color2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundGradient(final T rendering, final KColor color1, final KColor color2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, color2, angle);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundGradient(final T rendering, final KColor color1, final int alpha1, final KColor color2, final int alpha2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, alpha1, color2, alpha2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundGradient(final T rendering, final KColor color1, final int alpha1, final KColor color2, final int alpha2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, alpha1, color2, alpha2, angle);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundGradient(final T rendering, final Colors color1, final Colors color2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, color2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundGradient(final T rendering, final Colors color1, final Colors color2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, color2, angle);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundGradient(final T rendering, final Colors color1, final int alpha1, final Colors color2, final int alpha2, final float angle) {
    rendering.getStyles().removeAll(IterableExtensions.<KForeground>toList(Iterables.<KForeground>filter(rendering.getStyles(), KForeground.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, alpha1, color2, alpha2, angle);
    _styles.add(_setColors);
    return rendering;
  }

  public <T extends KRendering> T setSelectionForegroundGradient(final T rendering, final Colors color1, final int alpha1, final Colors color2, final int alpha2, final float angle) {
    EList<KStyle> _styles = rendering.getStyles();
    KForeground _setColors = this._kRenderingFactory.createKForeground().setColors(color1, alpha1, color2, alpha2, angle);
    KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_setColors, KRenderingExtensions.SET_SELECTION);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setForegroundInvisible(final T rendering, final boolean invisible) {
    final Procedure1<T> _function = (T it) -> {
      KForeground _elvis = null;
      EList<KStyle> _styles = it.getStyles();
      Iterable<KForeground> _filter = null;
      if (_styles!=null) {
        _filter=Iterables.<KForeground>filter(_styles, KForeground.class);
      }
      KForeground _last = IterableExtensions.<KForeground>last(_filter);
      if (_last != null) {
        _elvis = _last;
      } else {
        KForeground _createKForeground = this._kRenderingFactory.createKForeground();
        final Procedure1<KForeground> _function_1 = (KForeground it_1) -> {
          EList<KStyle> _styles_1 = rendering.getStyles();
          _styles_1.add(it_1);
          it_1.setColor(this._kRenderingFactory.createKColor());
        };
        KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_createKForeground, _function_1);
        _elvis = _doubleArrow;
      }
      final Procedure1<KForeground> _function_2 = (KForeground it_1) -> {
        Integer _xifexpression = null;
        if (invisible) {
          _xifexpression = Integer.valueOf(0);
        } else {
          Object _defaultValue = this._kRenderingFactory.getKRenderingPackage().getKColoring_Alpha().getDefaultValue();
          _xifexpression = ((Integer) _defaultValue);
        }
        it_1.setAlpha((_xifexpression).intValue());
      };
      ObjectExtensions.<KForeground>operator_doubleArrow(_elvis, _function_2);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  private final int defaultShadowSize = 4;

  public KShadow getShadow(final KRendering rendering) {
    KShadow _elvis = null;
    KShadow _last = IterableExtensions.<KShadow>last(Iterables.<KShadow>filter(rendering.getStyles(), KShadow.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      _elvis = null;
    }
    return _elvis;
  }

  public <T extends KRendering> T setShadow(final T rendering, final Colors color) {
    return this.<T>setShadow(rendering, this._kRenderingFactory.createKColor().setColor(color));
  }

  public <T extends KRendering> T setShadow(final T rendering, final Colors color, final float size) {
    return this.<T>setShadow(rendering, this._kRenderingFactory.createKColor().setColor(color), size);
  }

  public <T extends KRendering> T setShadow(final T rendering, final Colors color, final float xOffset, final float yOffset) {
    return this.<T>setShadow(rendering, this._kRenderingFactory.createKColor().setColor(color), xOffset, yOffset);
  }

  public <T extends KRendering> T setShadow(final T rendering, final KColor color) {
    return this.<T>setShadow(rendering, color, this.defaultShadowSize);
  }

  public <T extends KRendering> T setShadow(final T rendering, final KColor color, final float size) {
    return this.<T>setShadow(rendering, color, size, size);
  }

  public <T extends KRendering> T setShadow(final T rendering, final KColor color, final float xOffset, final float yOffset) {
    rendering.getStyles().removeAll(IterableExtensions.<KShadow>toList(Iterables.<KShadow>filter(rendering.getStyles(), KShadow.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KShadow _createKShadow = this._kRenderingFactory.createKShadow();
    final Procedure1<KShadow> _function = (KShadow it) -> {
      EList<KStyle> _styles_1 = rendering.getStyles();
      _styles_1.add(it);
      it.setColor(color);
      it.setXOffset(xOffset);
      it.setYOffset(yOffset);
    };
    KShadow _doubleArrow = ObjectExtensions.<KShadow>operator_doubleArrow(_createKShadow, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KFontSize getFontSize(final KRendering rendering) {
    KFontSize _elvis = null;
    KFontSize _last = IterableExtensions.<KFontSize>last(Iterables.<KFontSize>filter(rendering.getStyles(), KFontSize.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontSize _createKFontSize = this._kRenderingFactory.createKFontSize();
      final Procedure1<KFontSize> _function = (KFontSize it) -> {
        it.setSize(10);
      };
      KFontSize _doubleArrow = ObjectExtensions.<KFontSize>operator_doubleArrow(_createKFontSize, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public int getFontSizeValue(final KRendering rendering) {
    KFontSize _elvis = null;
    KFontSize _last = IterableExtensions.<KFontSize>last(Iterables.<KFontSize>filter(rendering.getStyles(), KFontSize.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontSize _createKFontSize = this._kRenderingFactory.createKFontSize();
      final Procedure1<KFontSize> _function = (KFontSize it) -> {
        it.setSize(10);
      };
      KFontSize _doubleArrow = ObjectExtensions.<KFontSize>operator_doubleArrow(_createKFontSize, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getSize();
  }

  public <T extends KRendering> T setFontSize(final T rendering, final int size) {
    rendering.getStyles().removeAll(IterableExtensions.<KFontSize>toList(Iterables.<KFontSize>filter(rendering.getStyles(), KFontSize.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KFontSize _createKFontSize = this._kRenderingFactory.createKFontSize();
    final Procedure1<KFontSize> _function = (KFontSize it) -> {
      it.setSize(size);
    };
    KFontSize _doubleArrow = ObjectExtensions.<KFontSize>operator_doubleArrow(_createKFontSize, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KFontName getFontName(final KRendering rendering) {
    KFontName _elvis = null;
    KFontName _last = IterableExtensions.<KFontName>last(Iterables.<KFontName>filter(rendering.getStyles(), KFontName.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontName _createKFontName = this._kRenderingFactory.createKFontName();
      final Procedure1<KFontName> _function = (KFontName it) -> {
        it.setName("Arial");
      };
      KFontName _doubleArrow = ObjectExtensions.<KFontName>operator_doubleArrow(_createKFontName, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public String getFontNameValue(final KRendering rendering) {
    KFontName _elvis = null;
    KFontName _last = IterableExtensions.<KFontName>last(Iterables.<KFontName>filter(rendering.getStyles(), KFontName.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontName _createKFontName = this._kRenderingFactory.createKFontName();
      final Procedure1<KFontName> _function = (KFontName it) -> {
        it.setName("Arial");
      };
      KFontName _doubleArrow = ObjectExtensions.<KFontName>operator_doubleArrow(_createKFontName, _function);
      _elvis = _doubleArrow;
    }
    return _elvis.getName();
  }

  public <T extends KRendering> T setFontName(final T rendering, final String name) {
    rendering.getStyles().removeAll(IterableExtensions.<KFontName>toList(Iterables.<KFontName>filter(rendering.getStyles(), KFontName.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KFontName _createKFontName = this._kRenderingFactory.createKFontName();
    final Procedure1<KFontName> _function = (KFontName it) -> {
      it.setName(name);
    };
    KFontName _doubleArrow = ObjectExtensions.<KFontName>operator_doubleArrow(_createKFontName, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KFontBold getFontBold(final KRendering rendering) {
    KFontBold _elvis = null;
    KFontBold _last = IterableExtensions.<KFontBold>last(Iterables.<KFontBold>filter(rendering.getStyles(), KFontBold.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontBold _createKFontBold = this._kRenderingFactory.createKFontBold();
      _elvis = _createKFontBold;
    }
    return _elvis;
  }

  public boolean getFontBoldValue(final KRendering rendering) {
    KFontBold _elvis = null;
    KFontBold _last = IterableExtensions.<KFontBold>last(Iterables.<KFontBold>filter(rendering.getStyles(), KFontBold.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontBold _createKFontBold = this._kRenderingFactory.createKFontBold();
      _elvis = _createKFontBold;
    }
    return _elvis.isBold();
  }

  public <T extends KRendering> T setFontBold(final T rendering, final boolean bold) {
    rendering.getStyles().removeAll(IterableExtensions.<KFontBold>toList(Iterables.<KFontBold>filter(rendering.getStyles(), KFontBold.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KFontBold _createKFontBold = this._kRenderingFactory.createKFontBold();
    final Procedure1<KFontBold> _function = (KFontBold it) -> {
      it.setBold(bold);
    };
    KFontBold _doubleArrow = ObjectExtensions.<KFontBold>operator_doubleArrow(_createKFontBold, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionFontBold(final T rendering, final boolean bold) {
    rendering.getStyles().removeAll(IterableExtensions.<KFontBold>toList(Iterables.<KFontBold>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KFontBold.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KFontBold _createKFontBold = this._kRenderingFactory.createKFontBold();
    final Procedure1<KFontBold> _function = (KFontBold it) -> {
      it.setSelection(true);
      it.setBold(bold);
    };
    KFontBold _doubleArrow = ObjectExtensions.<KFontBold>operator_doubleArrow(_createKFontBold, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KFontItalic getFontItalic(final KRendering rendering) {
    KFontItalic _elvis = null;
    KFontItalic _last = IterableExtensions.<KFontItalic>last(Iterables.<KFontItalic>filter(rendering.getStyles(), KFontItalic.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontItalic _createKFontItalic = this._kRenderingFactory.createKFontItalic();
      _elvis = _createKFontItalic;
    }
    return _elvis;
  }

  public boolean getFontItalicValue(final KRendering rendering) {
    KFontItalic _elvis = null;
    KFontItalic _last = IterableExtensions.<KFontItalic>last(Iterables.<KFontItalic>filter(rendering.getStyles(), KFontItalic.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KFontItalic _createKFontItalic = this._kRenderingFactory.createKFontItalic();
      _elvis = _createKFontItalic;
    }
    return _elvis.isItalic();
  }

  public <T extends KRendering> T setFontItalic(final T rendering, final boolean italic) {
    rendering.getStyles().removeAll(IterableExtensions.<KFontItalic>toList(Iterables.<KFontItalic>filter(rendering.getStyles(), KFontItalic.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KFontItalic _createKFontItalic = this._kRenderingFactory.createKFontItalic();
    final Procedure1<KFontItalic> _function = (KFontItalic it) -> {
      it.setItalic(italic);
    };
    KFontItalic _doubleArrow = ObjectExtensions.<KFontItalic>operator_doubleArrow(_createKFontItalic, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KTextUnderline getTextUnderline(final KRendering rendering) {
    KTextUnderline _elvis = null;
    KTextUnderline _last = IterableExtensions.<KTextUnderline>last(Iterables.<KTextUnderline>filter(rendering.getStyles(), KTextUnderline.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
      _elvis = _createKTextUnderline;
    }
    return _elvis;
  }

  public Underline getTextUnderlineValue(final KRendering rendering) {
    KTextUnderline _elvis = null;
    KTextUnderline _last = IterableExtensions.<KTextUnderline>last(Iterables.<KTextUnderline>filter(rendering.getStyles(), KTextUnderline.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
      _elvis = _createKTextUnderline;
    }
    return _elvis.getUnderline();
  }

  public KColor getTextUnderlineColorValue(final KRendering rendering) {
    KTextUnderline _elvis = null;
    KTextUnderline _last = IterableExtensions.<KTextUnderline>last(Iterables.<KTextUnderline>filter(rendering.getStyles(), KTextUnderline.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
      _elvis = _createKTextUnderline;
    }
    return _elvis.getColor();
  }

  public <T extends KRendering> T setTextUnderline(final T rendering, final Underline underline) {
    rendering.getStyles().removeAll(IterableExtensions.<KTextUnderline>toList(Iterables.<KTextUnderline>filter(rendering.getStyles(), KTextUnderline.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
    final Procedure1<KTextUnderline> _function = (KTextUnderline it) -> {
      it.setUnderline(underline);
    };
    KTextUnderline _doubleArrow = ObjectExtensions.<KTextUnderline>operator_doubleArrow(_createKTextUnderline, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionTextUnderline(final T rendering, final Underline underline) {
    rendering.getStyles().removeAll(IterableExtensions.<KTextUnderline>toList(Iterables.<KTextUnderline>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KTextUnderline.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
    final Procedure1<KTextUnderline> _function = (KTextUnderline it) -> {
      it.setSelection(true);
      it.setUnderline(underline);
    };
    KTextUnderline _doubleArrow = ObjectExtensions.<KTextUnderline>operator_doubleArrow(_createKTextUnderline, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setTextUnderlineColor(final T rendering, final KColor color) {
    final Procedure1<T> _function = (T it) -> {
      KTextUnderline _elvis = null;
      KTextUnderline _last = IterableExtensions.<KTextUnderline>last(Iterables.<KTextUnderline>filter(rendering.getStyles(), KTextUnderline.class));
      if (_last != null) {
        _elvis = _last;
      } else {
        KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
        _elvis = _createKTextUnderline;
      }
      final Procedure1<KTextUnderline> _function_1 = (KTextUnderline it_1) -> {
        it_1.setColor(color);
      };
      ObjectExtensions.<KTextUnderline>operator_doubleArrow(_elvis, _function_1);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  public <T extends KRendering> T setSelectionTextUnderlineColor(final T rendering, final KColor color) {
    final Procedure1<T> _function = (T it) -> {
      KTextUnderline _elvis = null;
      KTextUnderline _last = IterableExtensions.<KTextUnderline>last(Iterables.<KTextUnderline>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KTextUnderline.class));
      if (_last != null) {
        _elvis = _last;
      } else {
        KTextUnderline _createKTextUnderline = this._kRenderingFactory.createKTextUnderline();
        _elvis = _createKTextUnderline;
      }
      final Procedure1<KTextUnderline> _function_1 = (KTextUnderline it_1) -> {
        it_1.setSelection(true);
        it_1.setColor(color);
      };
      ObjectExtensions.<KTextUnderline>operator_doubleArrow(_elvis, _function_1);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  public KTextStrikeout getTextStrikeout(final KRendering rendering) {
    KTextStrikeout _elvis = null;
    KTextStrikeout _last = IterableExtensions.<KTextStrikeout>last(Iterables.<KTextStrikeout>filter(rendering.getStyles(), KTextStrikeout.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
      _elvis = _createKTextStrikeout;
    }
    return _elvis;
  }

  public boolean getTextStrikeoutValue(final KRendering rendering) {
    KTextStrikeout _elvis = null;
    KTextStrikeout _last = IterableExtensions.<KTextStrikeout>last(Iterables.<KTextStrikeout>filter(rendering.getStyles(), KTextStrikeout.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
      _elvis = _createKTextStrikeout;
    }
    return (_elvis.getStruckOut()).booleanValue();
  }

  public KColor getTextStrikeoutColorValue(final KRendering rendering) {
    KTextStrikeout _elvis = null;
    KTextStrikeout _last = IterableExtensions.<KTextStrikeout>last(Iterables.<KTextStrikeout>filter(rendering.getStyles(), KTextStrikeout.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
      _elvis = _createKTextStrikeout;
    }
    return _elvis.getColor();
  }

  public <T extends KRendering> T setTextStrikeout(final T rendering, final boolean struckOut) {
    rendering.getStyles().removeAll(IterableExtensions.<KTextStrikeout>toList(Iterables.<KTextStrikeout>filter(rendering.getStyles(), KTextStrikeout.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
    final Procedure1<KTextStrikeout> _function = (KTextStrikeout it) -> {
      it.setStruckOut(Boolean.valueOf(struckOut));
    };
    KTextStrikeout _doubleArrow = ObjectExtensions.<KTextStrikeout>operator_doubleArrow(_createKTextStrikeout, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setSelectionTextStrikeout(final T rendering, final boolean struckOut) {
    rendering.getStyles().removeAll(IterableExtensions.<KTextStrikeout>toList(Iterables.<KTextStrikeout>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KTextStrikeout.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
    final Procedure1<KTextStrikeout> _function = (KTextStrikeout it) -> {
      it.setSelection(true);
      it.setStruckOut(Boolean.valueOf(struckOut));
    };
    KTextStrikeout _doubleArrow = ObjectExtensions.<KTextStrikeout>operator_doubleArrow(_createKTextStrikeout, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setTextStrikeoutColor(final T rendering, final KColor color) {
    final Procedure1<T> _function = (T it) -> {
      KTextStrikeout _elvis = null;
      KTextStrikeout _last = IterableExtensions.<KTextStrikeout>last(Iterables.<KTextStrikeout>filter(rendering.getStyles(), KTextStrikeout.class));
      if (_last != null) {
        _elvis = _last;
      } else {
        KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
        _elvis = _createKTextStrikeout;
      }
      final Procedure1<KTextStrikeout> _function_1 = (KTextStrikeout it_1) -> {
        it_1.setColor(color);
      };
      ObjectExtensions.<KTextStrikeout>operator_doubleArrow(_elvis, _function_1);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  public <T extends KRendering> T setSelectionTextStrikeoutColor(final T rendering, final KColor color) {
    final Procedure1<T> _function = (T it) -> {
      KTextStrikeout _elvis = null;
      KTextStrikeout _last = IterableExtensions.<KTextStrikeout>last(Iterables.<KTextStrikeout>filter(IterableExtensions.<KStyle>filter(rendering.getStyles(), ((Function1<? super KStyle, Boolean>)KRenderingExtensions.IS_SELECTION)), KTextStrikeout.class));
      if (_last != null) {
        _elvis = _last;
      } else {
        KTextStrikeout _createKTextStrikeout = this._kRenderingFactory.createKTextStrikeout();
        _elvis = _createKTextStrikeout;
      }
      final Procedure1<KTextStrikeout> _function_1 = (KTextStrikeout it_1) -> {
        it_1.setColor(color);
      };
      ObjectExtensions.<KTextStrikeout>operator_doubleArrow(_elvis, _function_1);
    };
    return ObjectExtensions.<T>operator_doubleArrow(rendering, _function);
  }

  public final HorizontalAlignment H_LEFT = HorizontalAlignment.LEFT;

  public final HorizontalAlignment H_CENTRAL = HorizontalAlignment.CENTER;

  public final HorizontalAlignment H_RIGHT = HorizontalAlignment.RIGHT;

  public KHorizontalAlignment getHorizontalAlignment(final KRendering rendering) {
    KHorizontalAlignment _elvis = null;
    KHorizontalAlignment _last = IterableExtensions.<KHorizontalAlignment>last(Iterables.<KHorizontalAlignment>filter(rendering.getStyles(), KHorizontalAlignment.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KHorizontalAlignment _createKHorizontalAlignment = this._kRenderingFactory.createKHorizontalAlignment();
      final Procedure1<KHorizontalAlignment> _function = (KHorizontalAlignment it) -> {
        it.setHorizontalAlignment(this.H_CENTRAL);
      };
      KHorizontalAlignment _doubleArrow = ObjectExtensions.<KHorizontalAlignment>operator_doubleArrow(_createKHorizontalAlignment, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public <T extends KRendering> T setHorizontalAlignment(final T rendering, final HorizontalAlignment ha) {
    rendering.getStyles().removeAll(IterableExtensions.<KHorizontalAlignment>toList(Iterables.<KHorizontalAlignment>filter(rendering.getStyles(), KHorizontalAlignment.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KHorizontalAlignment _createKHorizontalAlignment = this._kRenderingFactory.createKHorizontalAlignment();
    final Procedure1<KHorizontalAlignment> _function = (KHorizontalAlignment it) -> {
      it.setHorizontalAlignment(ha);
    };
    KHorizontalAlignment _doubleArrow = ObjectExtensions.<KHorizontalAlignment>operator_doubleArrow(_createKHorizontalAlignment, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public final VerticalAlignment V_TOP = VerticalAlignment.TOP;

  public final VerticalAlignment V_CENTRAL = VerticalAlignment.CENTER;

  public final VerticalAlignment V_BOTTOM = VerticalAlignment.BOTTOM;

  public KVerticalAlignment getVerticalAlignment(final KRendering rendering) {
    KVerticalAlignment _elvis = null;
    KVerticalAlignment _last = IterableExtensions.<KVerticalAlignment>last(Iterables.<KVerticalAlignment>filter(rendering.getStyles(), KVerticalAlignment.class));
    if (_last != null) {
      _elvis = _last;
    } else {
      KVerticalAlignment _createKVerticalAlignment = this._kRenderingFactory.createKVerticalAlignment();
      final Procedure1<KVerticalAlignment> _function = (KVerticalAlignment it) -> {
        it.setVerticalAlignment(this.V_CENTRAL);
      };
      KVerticalAlignment _doubleArrow = ObjectExtensions.<KVerticalAlignment>operator_doubleArrow(_createKVerticalAlignment, _function);
      _elvis = _doubleArrow;
    }
    return _elvis;
  }

  public <T extends KRendering> T setVerticalAlignment(final T rendering, final VerticalAlignment va) {
    rendering.getStyles().removeAll(IterableExtensions.<KVerticalAlignment>toList(Iterables.<KVerticalAlignment>filter(rendering.getStyles(), KVerticalAlignment.class)));
    EList<KStyle> _styles = rendering.getStyles();
    KVerticalAlignment _createKVerticalAlignment = this._kRenderingFactory.createKVerticalAlignment();
    final Procedure1<KVerticalAlignment> _function = (KVerticalAlignment it) -> {
      it.setVerticalAlignment(va);
    };
    KVerticalAlignment _doubleArrow = ObjectExtensions.<KVerticalAlignment>operator_doubleArrow(_createKVerticalAlignment, _function);
    _styles.add(_doubleArrow);
    return rendering;
  }

  public KPolyline from(final KPolyline polyline, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB);
      _points.add(_createKPosition);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(polyline, _function);
  }

  public KPolyline to(final KPolyline polyline, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB);
      _points.add(_createKPosition);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(polyline, _function);
  }

  public <T extends KRendering> T setAreaPlacementData(final T rendering, final KPosition topLeft, final KPosition bottomRight) {
    KAreaPlacementData _createKAreaPlacementData = this._kRenderingFactory.createKAreaPlacementData();
    final Procedure1<KAreaPlacementData> _function = (KAreaPlacementData it) -> {
      it.setTopLeft(topLeft);
      it.setBottomRight(bottomRight);
    };
    KAreaPlacementData _doubleArrow = ObjectExtensions.<KAreaPlacementData>operator_doubleArrow(_createKAreaPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  public KAreaPlacementData setAreaPlacementData(final KRendering rendering) {
    KAreaPlacementData _createKAreaPlacementData = this._kRenderingFactory.createKAreaPlacementData();
    final Procedure1<KAreaPlacementData> _function = (KAreaPlacementData it) -> {
      rendering.setPlacementData(it);
    };
    return ObjectExtensions.<KAreaPlacementData>operator_doubleArrow(_createKAreaPlacementData, _function);
  }

  public KAreaPlacementData from(final KAreaPlacementData placementData, final KPosition topLeft) {
    final Procedure1<KAreaPlacementData> _function = (KAreaPlacementData it) -> {
      it.setTopLeft(topLeft);
    };
    return ObjectExtensions.<KAreaPlacementData>operator_doubleArrow(placementData, _function);
  }

  public KAreaPlacementData from(final KAreaPlacementData placementData, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    return this.from(placementData, 
      this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
  }

  public KRendering to(final KAreaPlacementData placementData, final KPosition bottomRight) {
    placementData.setBottomRight(bottomRight);
    EObject _eContainer = placementData.eContainer();
    return ((KRendering) _eContainer);
  }

  public KRendering to(final KAreaPlacementData placementData, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    return this.to(placementData, 
      this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
  }

  public <T extends KRendering> T setSurroundingSpace(final T rendering, final float abs, final float rel) {
    return this.<T>setAreaPlacementData(rendering, 
      this.createKPosition(this.LEFT, abs, rel, this.TOP, abs, rel), 
      this.createKPosition(this.RIGHT, abs, rel, this.BOTTOM, abs, rel));
  }

  public <T extends KRendering> T setSurroundingSpace(final T rendering, final float hAbs, final float hRel, final float vAbs, final float vRel) {
    return this.<T>setAreaPlacementData(rendering, 
      this.createKPosition(this.LEFT, hAbs, hRel, this.TOP, vAbs, vRel), 
      this.createKPosition(this.RIGHT, hAbs, hRel, this.BOTTOM, vAbs, vRel));
  }

  /**
   * Adds a grid placement to the rendering element with the specified spacing.
   */
  public <T extends KRendering> T setGridPlacementData(final T rendering, final float minCellWidth, final float minCellHeight, final KPosition topLeft, final KPosition bottomRight) {
    KGridPlacementData _createKGridPlacementData = this._kRenderingFactory.createKGridPlacementData();
    final Procedure1<KGridPlacementData> _function = (KGridPlacementData it) -> {
      it.setMinCellWidth(minCellWidth);
      it.setMinCellHeight(minCellHeight);
      it.setTopLeft(topLeft);
      it.setBottomRight(bottomRight);
    };
    KGridPlacementData _doubleArrow = ObjectExtensions.<KGridPlacementData>operator_doubleArrow(_createKGridPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  /**
   * Adds a grid placement to the rendering element with the specified spacing.
   * 
   * @example
   * rectangle.setGridPlacementData =&gt; [ cell |
   *  cell.from(LEFT, padding, 0, TOP, padding - 2, 0).to(RIGHT, padding, 0, BOTTOM, 0, 0)
   *  cell.minCellHeight = propBarHeight
   *  cell.flexibleHeight = false
   * ]
   */
  public KGridPlacementData setGridPlacementData(final KRendering rendering) {
    KGridPlacementData _createKGridPlacementData = this._kRenderingFactory.createKGridPlacementData();
    final Procedure1<KGridPlacementData> _function = (KGridPlacementData it) -> {
      rendering.setPlacementData(it);
    };
    return ObjectExtensions.<KGridPlacementData>operator_doubleArrow(_createKGridPlacementData, _function);
  }

  /**
   * Adds a grid placement to the rendering element with the specified spacing.
   */
  public KGridPlacementData setGridPlacementData(final KRendering rendering, final float minCellWidth, final float minCellHeight) {
    KGridPlacementData _createKGridPlacementData = this._kRenderingFactory.createKGridPlacementData();
    final Procedure1<KGridPlacementData> _function = (KGridPlacementData it) -> {
      rendering.setPlacementData(it);
      it.setMinCellWidth(minCellWidth);
      it.setMinCellHeight(minCellHeight);
    };
    return ObjectExtensions.<KGridPlacementData>operator_doubleArrow(_createKGridPlacementData, _function);
  }

  public <T extends KRendering> T setSurroundingSpaceGrid(final T rendering, final float abs, final float rel) {
    return this.<T>setGridPlacementData(rendering, 0f, 0f, 
      this.createKPosition(this.LEFT, abs, rel, this.TOP, abs, rel), 
      this.createKPosition(this.RIGHT, abs, rel, this.BOTTOM, abs, rel));
  }

  public <T extends KRendering> T setSurroundingSpaceGrid(final T rendering, final float hAbs, final float hRel, final float vAbs, final float vRel) {
    return this.<T>setGridPlacementData(rendering, 0f, 0f, 
      this.createKPosition(this.LEFT, hAbs, hRel, this.TOP, vAbs, vRel), 
      this.createKPosition(this.RIGHT, hAbs, hRel, this.BOTTOM, vAbs, vRel));
  }

  public KGridPlacement from(final KGridPlacement placement, final KPosition topLeft) {
    placement.setTopLeft(topLeft);
    return placement;
  }

  public KGridPlacement from(final KGridPlacement placement, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    this.from(placement, 
      this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
    return placement;
  }

  public KContainerRendering to(final KGridPlacement placement, final KPosition bottomRight) {
    placement.setBottomRight(bottomRight);
    final EObject cont = placement.eContainer();
    KContainerRendering _xifexpression = null;
    if ((cont instanceof KContainerRendering)) {
      _xifexpression = ((KContainerRendering) cont);
    } else {
      _xifexpression = null;
    }
    return _xifexpression;
  }

  public KContainerRendering to(final KGridPlacement placement, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    this.to(placement, 
      this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
    final EObject cont = placement.eContainer();
    KContainerRendering _xifexpression = null;
    if ((cont instanceof KContainerRendering)) {
      _xifexpression = ((KContainerRendering) cont);
    } else {
      _xifexpression = null;
    }
    return _xifexpression;
  }

  public <T extends KRendering> T setLeftTopAlignedPointPlacementData(final T rendering, final float leftMargin, final float topMargin, final float rightMargin, final float bottomMargin) {
    return this.<T>setLeftTopAlignedPointPlacementData(rendering, leftMargin, 0, topMargin, 0, rightMargin, bottomMargin, 0, 0);
  }

  public <T extends KRendering> T setLeftTopAlignedPointPlacementData(final T rendering, final float absoluteLR, final float relativeLR, final float absoluteTB, final float relativeTB, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    return this.<T>setPointPlacementData(rendering, 
      this.LEFT, absoluteLR, relativeLR, this.TOP, absoluteTB, relativeTB, this.H_LEFT, this.V_TOP, 0, 0, 0, 0);
  }

  public <T extends KRendering> T setLeftBottomAlignedPointPlacementData(final T rendering, final float leftMargin, final float topMargin, final float rightMargin, final float bottomMargin) {
    return this.<T>setLeftTopAlignedPointPlacementData(rendering, leftMargin, 0, bottomMargin, 0, rightMargin, topMargin, 0, 0);
  }

  public <T extends KRendering> T setLeftBottomAlignedPointPlacementData(final T rendering, final float absoluteLR, final float relativeLR, final float absoluteTB, final float relativeTB, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    return this.<T>setPointPlacementData(rendering, 
      this.LEFT, absoluteLR, relativeLR, this.BOTTOM, absoluteTB, relativeTB, this.H_LEFT, this.V_BOTTOM, 0, 0, 0, 0);
  }

  public <T extends KRendering> T setRightTopAlignedPointPlacementData(final T rendering, final float leftMargin, final float topMargin, final float rightMargin, final float bottomMargin) {
    return this.<T>setLeftTopAlignedPointPlacementData(rendering, rightMargin, 0, topMargin, 0, leftMargin, bottomMargin, 0, 0);
  }

  public <T extends KRendering> T setRightTopAlignedPointPlacementData(final T rendering, final float absoluteLR, final float relativeLR, final float absoluteTB, final float relativeTB, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    return this.<T>setPointPlacementData(rendering, 
      this.RIGHT, absoluteLR, relativeLR, this.TOP, absoluteTB, relativeTB, this.H_RIGHT, this.V_TOP, 0, 0, 0, 0);
  }

  public <T extends KRendering> T setRightBottomAlignedPointPlacementData(final T rendering, final float leftMargin, final float topMargin, final float rightMargin, final float bottomMargin) {
    return this.<T>setLeftTopAlignedPointPlacementData(rendering, rightMargin, 0, bottomMargin, 0, leftMargin, topMargin, 0, 0);
  }

  public <T extends KRendering> T setRightBottomAlignedPointPlacementData(final T rendering, final float absoluteLR, final float relativeLR, final float absoluteTB, final float relativeTB, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    return this.<T>setPointPlacementData(rendering, 
      this.RIGHT, absoluteLR, relativeLR, this.BOTTOM, absoluteTB, relativeTB, this.H_RIGHT, this.V_BOTTOM, 0, 0, 0, 0);
  }

  public <T extends KRendering> T setPointPlacementData(final T rendering, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    KPointPlacementData _createKPointPlacementData = this._kRenderingFactory.createKPointPlacementData();
    final Procedure1<KPointPlacementData> _function = (KPointPlacementData it) -> {
      it.setReferencePoint(this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
    };
    KPointPlacementData _doubleArrow = ObjectExtensions.<KPointPlacementData>operator_doubleArrow(_createKPointPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> KPointPlacementData setPointPlacementData(final T rendering) {
    KPointPlacementData _createKPointPlacementData = this._kRenderingFactory.createKPointPlacementData();
    final Procedure1<KPointPlacementData> _function = (KPointPlacementData it) -> {
      rendering.setPlacementData(it);
    };
    return ObjectExtensions.<KPointPlacementData>operator_doubleArrow(_createKPointPlacementData, _function);
  }

  public KPointPlacementData at(final KPointPlacementData ppd, final KPosition pos) {
    ppd.setReferencePoint(pos);
    return ppd;
  }

  public KPointPlacementData at(final KPointPlacementData ppd, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    ppd.setReferencePoint(this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
    return ppd;
  }

  public KPointPlacementData setAlignment(final KPointPlacementData ppd, final HorizontalAlignment horizontalAlignment, final VerticalAlignment verticalAlignment) {
    ppd.setHorizontalAlignment(horizontalAlignment);
    ppd.setVerticalAlignment(verticalAlignment);
    return ppd;
  }

  public KPointPlacementData setMargins(final KPointPlacementData ppd, final float horizontalMargin, final float verticalMargin) {
    ppd.setHorizontalMargin(horizontalMargin);
    ppd.setVerticalMargin(verticalMargin);
    return ppd;
  }

  public KPointPlacementData setMinimalSize(final KPointPlacementData ppd, final float minWidth, final float minHeight) {
    ppd.setMinWidth(minWidth);
    ppd.setMinHeight(minHeight);
    return ppd;
  }

  public KRendering finish(final KPointPlacementData ppd) {
    EObject _eContainer = ppd.eContainer();
    return ((KRendering) _eContainer);
  }

  public <T extends KRendering> T setPointPlacementData(final T rendering, final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB, final HorizontalAlignment horizontalAlignment, final VerticalAlignment verticalAlignment, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    KPointPlacementData _createKPointPlacementData = this._kRenderingFactory.createKPointPlacementData();
    final Procedure1<KPointPlacementData> _function = (KPointPlacementData it) -> {
      it.setReferencePoint(this.createKPosition(px, absoluteLR, relativeLR, py, absoluteTB, relativeTB));
      it.setHorizontalAlignment(horizontalAlignment);
      it.setVerticalAlignment(verticalAlignment);
      it.setHorizontalMargin(horizontalMargin);
      it.setVerticalMargin(verticalMargin);
      it.setMinWidth(minWidth);
      it.setMinHeight(minHeight);
    };
    KPointPlacementData _doubleArrow = ObjectExtensions.<KPointPlacementData>operator_doubleArrow(_createKPointPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setPointPlacementData(final T rendering, final KPosition referencePoint, final HorizontalAlignment horizontalAlignment, final VerticalAlignment verticalAlignment, final float horizontalMargin, final float verticalMargin, final float minWidth, final float minHeight) {
    KPointPlacementData _createKPointPlacementData = this._kRenderingFactory.createKPointPlacementData();
    final Procedure1<KPointPlacementData> _function = (KPointPlacementData it) -> {
      it.setReferencePoint(referencePoint);
      it.setHorizontalAlignment(horizontalAlignment);
      it.setVerticalAlignment(verticalAlignment);
      it.setHorizontalMargin(horizontalMargin);
      it.setVerticalMargin(verticalMargin);
      it.setMinWidth(minWidth);
      it.setMinHeight(minHeight);
    };
    KPointPlacementData _doubleArrow = ObjectExtensions.<KPointPlacementData>operator_doubleArrow(_createKPointPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  public <T extends KRendering> T setDecoratorPlacementData(final T rendering, final float width, final float height, final float posAbsolute, final float posRelative, final boolean rotateWithLine) {
    KDecoratorPlacementData _createKDecoratorPlacementData = this._kRenderingFactory.createKDecoratorPlacementData();
    final Procedure1<KDecoratorPlacementData> _function = (KDecoratorPlacementData it) -> {
      it.setWidth(width);
      it.setHeight(height);
      it.setAbsolute(posAbsolute);
      it.setRelative(posRelative);
      it.setRotateWithLine(rotateWithLine);
      it.setXOffset(((-width) / 2));
      it.setYOffset(((-height) / 2));
    };
    KDecoratorPlacementData _doubleArrow = ObjectExtensions.<KDecoratorPlacementData>operator_doubleArrow(_createKDecoratorPlacementData, _function);
    rendering.setPlacementData(_doubleArrow);
    return rendering;
  }

  public final PositionReferenceX LEFT = PositionReferenceX.LEFT;

  public final PositionReferenceX RIGHT = PositionReferenceX.RIGHT;

  public final PositionReferenceY TOP = PositionReferenceY.TOP;

  public final PositionReferenceY BOTTOM = PositionReferenceY.BOTTOM;

  public KPosition createKPosition(final PositionReferenceX px, final float absoluteLR, final float relativeLR, final PositionReferenceY py, final float absoluteTB, final float relativeTB) {
    KPosition _createKPosition = this._kRenderingFactory.createKPosition();
    final Procedure1<KPosition> _function = (KPosition it) -> {
      KXPosition<?> _switchResult = null;
      if (px != null) {
        switch (px) {
          case LEFT:
            _switchResult = this._kRenderingFactory.createKLeftPosition();
            break;
          case RIGHT:
            _switchResult = this._kRenderingFactory.createKRightPosition();
            break;
          default:
            break;
        }
      }
      final Procedure1<KXPosition<?>> _function_1 = (KXPosition<?> it_1) -> {
        it_1.setAbsolute(absoluteLR);
        it_1.setRelative(relativeLR);
      };
      KXPosition<?> _doubleArrow = ObjectExtensions.<KXPosition<?>>operator_doubleArrow(_switchResult, _function_1);
      it.setX(_doubleArrow);
      KYPosition<?> _switchResult_1 = null;
      if (py != null) {
        switch (py) {
          case TOP:
            _switchResult_1 = this._kRenderingFactory.createKTopPosition();
            break;
          case BOTTOM:
            _switchResult_1 = this._kRenderingFactory.createKBottomPosition();
            break;
          default:
            break;
        }
      }
      final Procedure1<KYPosition<?>> _function_2 = (KYPosition<?> it_1) -> {
        it_1.setAbsolute(absoluteTB);
        it_1.setRelative(relativeTB);
      };
      KYPosition<?> _doubleArrow_1 = ObjectExtensions.<KYPosition<?>>operator_doubleArrow(_switchResult_1, _function_2);
      it.setY(_doubleArrow_1);
    };
    return ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition, _function);
  }

  public KContainerRendering addInvisibleContainerRendering(final EMapPropertyHolder label) {
    if (label instanceof KLabel) {
      return _addInvisibleContainerRendering((KLabel)label);
    } else if (label instanceof KNode) {
      return _addInvisibleContainerRendering((KNode)label);
    } else if (label instanceof KPort) {
      return _addInvisibleContainerRendering((KPort)label);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label).toString());
    }
  }

  public KEllipse addEllipse(final EMapPropertyHolder label) {
    if (label instanceof KLabel) {
      return _addEllipse((KLabel)label);
    } else if (label instanceof KNode) {
      return _addEllipse((KNode)label);
    } else if (label instanceof KPort) {
      return _addEllipse((KPort)label);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label).toString());
    }
  }

  public KArc addArc(final EMapPropertyHolder label) {
    if (label instanceof KLabel) {
      return _addArc((KLabel)label);
    } else if (label instanceof KNode) {
      return _addArc((KNode)label);
    } else if (label instanceof KPort) {
      return _addArc((KPort)label);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label).toString());
    }
  }

  public KPolygon addPolygon(final EMapPropertyHolder node) {
    if (node instanceof KNode) {
      return _addPolygon((KNode)node);
    } else if (node instanceof KPort) {
      return _addPolygon((KPort)node);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(node).toString());
    }
  }

  public KRectangle addRectangle(final EMapPropertyHolder label) {
    if (label instanceof KLabel) {
      return _addRectangle((KLabel)label);
    } else if (label instanceof KNode) {
      return _addRectangle((KNode)label);
    } else if (label instanceof KPort) {
      return _addRectangle((KPort)label);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label).toString());
    }
  }

  public KRoundedRectangle addRoundedRectangle(final EMapPropertyHolder label, final float cWidth, final float cHeight) {
    if (label instanceof KLabel) {
      return _addRoundedRectangle((KLabel)label, cWidth, cHeight);
    } else if (label instanceof KNode) {
      return _addRoundedRectangle((KNode)label, cWidth, cHeight);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label, cWidth, cHeight).toString());
    }
  }

  public KRoundedRectangle addRoundedRectangle(final EMapPropertyHolder label, final float cWidth, final float cHeight, final float lineWidth) {
    if (label instanceof KLabel) {
      return _addRoundedRectangle((KLabel)label, cWidth, cHeight, lineWidth);
    } else if (label instanceof KNode) {
      return _addRoundedRectangle((KNode)label, cWidth, cHeight, lineWidth);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(label, cWidth, cHeight, lineWidth).toString());
    }
  }
}
