/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Scope;
import de.cau.cs.kieler.klighd.kgraph.KGraphData;
import de.cau.cs.kieler.klighd.kgraph.KNode;
import de.cau.cs.kieler.klighd.kgraph.KPort;
import de.cau.cs.kieler.klighd.kgraph.KShapeLayout;
import de.cau.cs.kieler.klighd.kgraph.util.KGraphUtil;
import de.cau.cs.kieler.klighd.krendering.HorizontalAlignment;
import de.cau.cs.kieler.klighd.krendering.KAreaPlacementData;
import de.cau.cs.kieler.klighd.krendering.KBackground;
import de.cau.cs.kieler.klighd.krendering.KFontSize;
import de.cau.cs.kieler.klighd.krendering.KForeground;
import de.cau.cs.kieler.klighd.krendering.KLeftPosition;
import de.cau.cs.kieler.klighd.krendering.KPosition;
import de.cau.cs.kieler.klighd.krendering.KRectangle;
import de.cau.cs.kieler.klighd.krendering.KRendering;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KRightPosition;
import de.cau.cs.kieler.klighd.krendering.KText;
import de.cau.cs.kieler.klighd.krendering.KTopPosition;
import de.cau.cs.kieler.klighd.krendering.VerticalAlignment;
import de.cau.cs.kieler.klighd.krendering.ViewSynthesisShared;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.options.PortSide;
import org.eclipse.elk.core.util.Maybe;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Provides some helpful extension methods for simplifying the composition of KGraph/KRendering-based view models.<br>
 * <br>
 * In order to employ them beyond KLighD diagram syntheses you best declare a field of type
 * {@link KNodeExtensions} in your class and annotate it with {@link Inject Inject}.<br>
 * <br>
 * Make sure to bind the {@link ViewSynthesisShared} annotation in the employed
 * {@link Injector Injector} to a {@link Scope}, e.g. by calling
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createSingletonScopeBindingModule());} or
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createNoScopeBindingModule());}.<br>
 * <br>
 * By means of that {@link Injector Injector} you may get a new instance of your class,
 * or you may inject the above mentioned attribute within instances of your class, e.g. by calling
 * {@code injector.injectMembers(this)} in the constructor of your class.
 * 
 * @author chsch
 * @author nre
 * 
 * @containsExtensions
 */
@ViewSynthesisShared
@SuppressWarnings("all")
public class KPortExtensions {
  /**
   * A constant determining the vertical margin between the top
   * of a node and the first port on the left or right.
   */
  private static final Float PORT_TOP_OFFSET = Float.valueOf(30f);

  /**
   * A constant determining the vertical margin between two ports
   * of a node on the left or right.
   */
  private static final Float VERTICAL_PORT_SPACING = Float.valueOf(15f);

  /**
   * A constant determining the edge length of the ports' figures that are formed
   * by a black square (if they are created by the extensions of this class).
   */
  private static final Float PORT_FIGURE_EDGE_LENGTH = Float.valueOf(7f);

  /**
   * A constant determining the font size of the ports' text labels
   * (if they are created by the extensions of this class).
   */
  private static final Integer PORT_LABEL_FONT_SIZE = Integer.valueOf(7);

  @Extension
  private KRenderingFactory _kRenderingFactory = KRenderingFactory.eINSTANCE;

  @Inject
  @Extension
  private KRenderingExtensions _kRenderingExtensions;

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  private KPort internalCreatePort(final Object... oc) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(oc);
    final KPort _result;
    synchronized (_createCache_internalCreatePort) {
      if (_createCache_internalCreatePort.containsKey(_cacheKey)) {
        return _createCache_internalCreatePort.get(_cacheKey);
      }
      KPort _createInitializedPort = KGraphUtil.createInitializedPort();
      _result = _createInitializedPort;
      _createCache_internalCreatePort.put(_cacheKey, _result);
    }
    _init_internalCreatePort(_result, oc);
    return _result;
  }

  private final HashMap<ArrayList<?>, KPort> _createCache_internalCreatePort = CollectionLiterals.newHashMap();

  private void _init_internalCreatePort(final KPort port, final Object oc) {
  }

  /**
   * The Xtend-generated internal create map for {@link #internalCreatePort} with a more accessible name.
   */
  private HashMap<ArrayList<?>, KPort> getInternalPortMap() {
    return this._createCache_internalCreatePort;
  }

  /**
   * A convenient test method to check whether or not a specific port exists in the create extension
   */
  public boolean portExists(final Object... os) {
    return this.getInternalPortMap().containsKey(CollectionLiterals.<Object>newArrayList(os));
  }

  /**
   * A convenient method to register a port that was not created via the create extension.
   * @return the previous port associated with the given object(s), or {@code null} if there was no port yet.
   */
  public KPort registerExistingPort(final KPort port, final Object... os) {
    return this.getInternalPortMap().put(CollectionLiterals.<Object>newArrayList(os), port);
  }

  /**
   * A convenient port getter based on a single business object preserving the
   * element image relation by a create extension.
   */
  public KPort getPort(final Object o1) {
    return this.internalCreatePort(o1);
  }

  /**
   * A convenient port getter based on a two business objects preserving the
   * element image relation by a create extension.
   */
  public KPort getPort(final Object o1, final Object o2) {
    return this.internalCreatePort(o1, o2);
  }

  /**
   * A convenient port getter based on a three business objects preserving the
   * element image relation by a create extension.
   */
  public KPort getPort(final Object o1, final Object o2, final Object o3) {
    return this.internalCreatePort(o1, o2, o3);
  }

  /**
   * A convenient port getter based on a four business objects preserving the
   * element image relation by a create extension.
   */
  public KPort getPort(final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.internalCreatePort(o1, o2, o3, o4);
  }

  /**
   * A convenient port getter based on a single business object preserving the
   * element image relation by a create extension.
   */
  public KPort getPort(final Object... os) {
    return this.internalCreatePort(os);
  }

  /**
   * A convenience method to create a KPort without relating it to a business object.
   */
  public KPort createPort() {
    return KGraphUtil.createInitializedPort();
  }

  /**
   * An alias of {@link #getPort(Object o1)} allowing to express in business that the KPort will
   * be created at this place. It is just syntactic sugar.
   */
  public KPort createPort(final Object o1) {
    return this.getPort(o1);
  }

  /**
   * An alias of {@link #getPort(Object o1, Object o2)} allowing to express in business that the
   * KPort will be created at this place. It is just syntactic sugar.
   */
  public KPort createPort(final Object o1, final Object o2) {
    return this.getPort(o1, o2);
  }

  /**
   * An alias of {@link #getPort(Object o1, Object o2, Object o3)} allowing to express in business
   * that the KPort will be created at this place. It is just syntactic sugar.
   */
  public KPort createPort(final Object o1, final Object o2, final Object o3) {
    return this.getPort(o1, o2, o3);
  }

  /**
   * An alias of {@link #getPort(Object o1, Object o2, Object o3, Object o4)} allowing to express in
   * business that the KPort will be created at this place. It is just syntactic sugar.
   */
  public KPort createPort(final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.getPort(o1, o2, o3, o4);
  }

  /**
   * An alias of {@link #getPort(Object o1)} allowing to express in business that the KPort will
   * be created at this place. It is just syntactic sugar.
   */
  public KPort createPort(final Object... os) {
    return this.getPort(os);
  }

  /**
   * Creates a port and a related port figure as well as a port label
   *  onto the east side of a given node with the port label text 'label'.
   *  The port is related to the given EObjects.
   * 
   * The create nature of these extensions avoids the relocation the an
   *  already created port due to the recall of the initializer extension.
   */
  public KPort createLabeledEPort(final KNode node, final String label, final Object o1) {
    return this.createLabeledEPort(node, label, CollectionLiterals.<Object>newArrayList(o1));
  }

  public KPort createLabeledEPort(final KNode node, final String label, final Object o1, final Object o2) {
    return this.createLabeledEPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2));
  }

  public KPort createLabeledEPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3) {
    return this.createLabeledEPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3));
  }

  public KPort createLabeledEPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.createLabeledEPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3, o4));
  }

  /**
   * Creates an anchor and a related port figure as well as a port label
   *  onto the north side of a given shape with the port label text 'label'.
   *  The anchor is related to the given EObjects.
   * 
   * The create nature of these extensions avoids the relocation the an
   *  already created port due to the recall of the initializer extension.
   */
  public KPort createLabeledNPort(final KNode node, final String label, final Object o1) {
    return this.createLabeledNPort(node, label, CollectionLiterals.<Object>newArrayList(o1));
  }

  public KPort createLabeledNPort(final KNode node, final String label, final Object o1, final Object o2) {
    return this.createLabeledNPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2));
  }

  public KPort createLabeledNPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3) {
    return this.createLabeledNPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3));
  }

  public KPort createLabeledNPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.createLabeledNPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3, o4));
  }

  /**
   * Creates an anchor and a related port figure as well as a port label
   *  onto the south side of a given shape with the port label text 'label'.
   *  The anchor is related to the given EObjects.
   * 
   * The create nature of these extensions avoids the relocation the an
   *  already created port due to the recall of the initializer extension.
   */
  public KPort createLabeledSPort(final KNode node, final String label, final Object o1) {
    return this.createLabeledSPort(node, label, CollectionLiterals.<Object>newArrayList(o1));
  }

  public KPort createLabeledSPort(final KNode node, final String label, final Object o1, final Object o2) {
    return this.createLabeledSPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2));
  }

  public KPort createLabeledSPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3) {
    return this.createLabeledSPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3));
  }

  public KPort createLabeledSPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.createLabeledSPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3, o4));
  }

  /**
   * Creates an anchor and a related port figure as well as a port label
   *  onto the west side of a given shape with the port label text 'label'.
   *  The anchor is related to the given EObjects.
   * 
   * The create nature of these extensions avoids the relocation the an
   *  already created port due to the recall of the initializer extension.
   */
  public KPort createLabeledWPort(final KNode node, final String label, final Object o1) {
    return this.createLabeledWPort(node, label, CollectionLiterals.<Object>newArrayList(o1));
  }

  public KPort createLabeledWPort(final KNode node, final String label, final Object o1, final Object o2) {
    return this.createLabeledWPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2));
  }

  public KPort createLabeledWPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3) {
    return this.createLabeledWPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3));
  }

  public KPort createLabeledWPort(final KNode node, final String label, final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.createLabeledWPort(node, label, CollectionLiterals.<Object>newArrayList(o1, o2, o3, o4));
  }

  private KPort createLabeledEPort(final KNode node, final String label, final ArrayList<Object> os) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node, label, os);
    final KPort _result;
    synchronized (_createCache_createLabeledEPort) {
      if (_createCache_createLabeledEPort.containsKey(_cacheKey)) {
        return _createCache_createLabeledEPort.get(_cacheKey);
      }
      KPort _internalCreatePort = this.internalCreatePort(((Object[])Conversions.unwrapArray(os, Object.class)));
      _result = _internalCreatePort;
      _createCache_createLabeledEPort.put(_cacheKey, _result);
    }
    _init_createLabeledEPort(_result, node, label, os);
    return _result;
  }

  private final HashMap<ArrayList<?>, KPort> _createCache_createLabeledEPort = CollectionLiterals.newHashMap();

  private void _init_createLabeledEPort(final KPort port, final KNode node, final String label, final ArrayList<Object> os) {
    final Procedure1<KPort> _function = (KPort it) -> {
      EList<KPort> _ports = node.getPorts();
      _ports.add(it);
      this.setPortSize(it, (this.getPortEdgeLength()).floatValue(), (this.getPortEdgeLength()).floatValue());
      this.<PortSide>addLayoutParam(it, CoreOptions.PORT_SIDE, PortSide.EAST);
      float _width = node.getWidth();
      float _minus = (_width - 1);
      this.setPortPos(it, _minus, (this.nextEPortYPosition(node)).floatValue());
      EList<KGraphData> _data = it.getData();
      KRendering _createEPortRendering = this.createEPortRendering(label);
      _data.add(_createEPortRendering);
    };
    ObjectExtensions.<KPort>operator_doubleArrow(port, _function);
  }

  private KPort createLabeledNPort(final KNode node, final String label, final ArrayList<Object> os) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node, label, os);
    final KPort _result;
    synchronized (_createCache_createLabeledNPort) {
      if (_createCache_createLabeledNPort.containsKey(_cacheKey)) {
        return _createCache_createLabeledNPort.get(_cacheKey);
      }
      KPort _internalCreatePort = this.internalCreatePort(((Object[])Conversions.unwrapArray(os, Object.class)));
      _result = _internalCreatePort;
      _createCache_createLabeledNPort.put(_cacheKey, _result);
    }
    _init_createLabeledNPort(_result, node, label, os);
    return _result;
  }

  private final HashMap<ArrayList<?>, KPort> _createCache_createLabeledNPort = CollectionLiterals.newHashMap();

  private void _init_createLabeledNPort(final KPort port, final KNode node, final String label, final ArrayList<Object> os) {
    final Procedure1<KPort> _function = (KPort it) -> {
      EList<KPort> _ports = node.getPorts();
      _ports.add(it);
      this.setPortSize(it, (this.getPortEdgeLength()).floatValue(), (this.getPortEdgeLength()).floatValue());
      this.<PortSide>addLayoutParam(it, CoreOptions.PORT_SIDE, PortSide.NORTH);
      Float _portEdgeLength = this.getPortEdgeLength();
      double _minus = (-((double) (_portEdgeLength).floatValue()));
      this.<Double>addLayoutParam(it, CoreOptions.PORT_BORDER_OFFSET, Double.valueOf(_minus));
      this.setPortPos(it, (this.nextNPortYPosition(node)).floatValue(), 1);
      EList<KGraphData> _data = it.getData();
      KRendering _setRotation = this._kRenderingExtensions.<KRendering>setRotation(this.createEPortRendering(label), Float.valueOf((-90f)));
      _data.add(_setRotation);
    };
    ObjectExtensions.<KPort>operator_doubleArrow(port, _function);
  }

  private KPort createLabeledSPort(final KNode node, final String label, final ArrayList<Object> os) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node, label, os);
    final KPort _result;
    synchronized (_createCache_createLabeledSPort) {
      if (_createCache_createLabeledSPort.containsKey(_cacheKey)) {
        return _createCache_createLabeledSPort.get(_cacheKey);
      }
      KPort _internalCreatePort = this.internalCreatePort(((Object[])Conversions.unwrapArray(os, Object.class)));
      _result = _internalCreatePort;
      _createCache_createLabeledSPort.put(_cacheKey, _result);
    }
    _init_createLabeledSPort(_result, node, label, os);
    return _result;
  }

  private final HashMap<ArrayList<?>, KPort> _createCache_createLabeledSPort = CollectionLiterals.newHashMap();

  private void _init_createLabeledSPort(final KPort port, final KNode node, final String label, final ArrayList<Object> os) {
    final Procedure1<KPort> _function = (KPort it) -> {
      EList<KPort> _ports = node.getPorts();
      _ports.add(it);
      this.setPortSize(it, (this.getPortEdgeLength()).floatValue(), (this.getPortEdgeLength()).floatValue());
      this.<PortSide>addLayoutParam(it, CoreOptions.PORT_SIDE, PortSide.SOUTH);
      this.<Double>addLayoutParam(it, CoreOptions.PORT_BORDER_OFFSET, Double.valueOf(0.0));
      Float _nextSPortYPosition = this.nextSPortYPosition(node);
      float _height = node.getHeight();
      float _minus = (_height - 1);
      this.setPortPos(it, (_nextSPortYPosition).floatValue(), _minus);
      EList<KGraphData> _data = it.getData();
      KRendering _setRotation = this._kRenderingExtensions.<KRendering>setRotation(this.createEPortRendering(label), Float.valueOf(90f));
      _data.add(_setRotation);
    };
    ObjectExtensions.<KPort>operator_doubleArrow(port, _function);
  }

  private KPort createLabeledWPort(final KNode node, final String label, final ArrayList<Object> os) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node, label, os);
    final KPort _result;
    synchronized (_createCache_createLabeledWPort) {
      if (_createCache_createLabeledWPort.containsKey(_cacheKey)) {
        return _createCache_createLabeledWPort.get(_cacheKey);
      }
      KPort _internalCreatePort = this.internalCreatePort(((Object[])Conversions.unwrapArray(os, Object.class)));
      _result = _internalCreatePort;
      _createCache_createLabeledWPort.put(_cacheKey, _result);
    }
    _init_createLabeledWPort(_result, node, label, os);
    return _result;
  }

  private final HashMap<ArrayList<?>, KPort> _createCache_createLabeledWPort = CollectionLiterals.newHashMap();

  private void _init_createLabeledWPort(final KPort port, final KNode node, final String label, final ArrayList<Object> os) {
    final Procedure1<KPort> _function = (KPort it) -> {
      EList<KPort> _ports = node.getPorts();
      _ports.add(it);
      this.setPortSize(it, (this.getPortEdgeLength()).floatValue(), (this.getPortEdgeLength()).floatValue());
      this.<PortSide>addLayoutParam(it, CoreOptions.PORT_SIDE, PortSide.WEST);
      this.<Double>addLayoutParam(it, CoreOptions.PORT_BORDER_OFFSET, Double.valueOf(0.0));
      this.setPortPos(it, (-6), (this.nextWPortYPosition(node)).floatValue());
      EList<KGraphData> _data = it.getData();
      KRendering _createWPortRendering = this.createWPortRendering(label);
      _data.add(_createWPortRendering);
    };
    ObjectExtensions.<KPort>operator_doubleArrow(port, _function);
  }

  private KRendering createWPortRendering(final String label) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      KForeground _createKForeground = this._kRenderingFactory.createKForeground();
      final Procedure1<KForeground> _function_1 = (KForeground it_1) -> {
        it_1.setAlpha(255);
        it_1.setColor(this._kRenderingFactory.createKColor());
      };
      KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_createKForeground, _function_1);
      this._kRenderingExtensions.setForeground(it, _doubleArrow);
      KBackground _createKBackground = this._kRenderingFactory.createKBackground();
      final Procedure1<KBackground> _function_2 = (KBackground it_1) -> {
        it_1.setAlpha(255);
        it_1.setColor(this._kRenderingFactory.createKColor());
      };
      KBackground _doubleArrow_1 = ObjectExtensions.<KBackground>operator_doubleArrow(_createKBackground, _function_2);
      this._kRenderingExtensions.setBackground(it, _doubleArrow_1);
      EList<KRendering> _children = it.getChildren();
      KText _withCopyOf = this._kRenderingExtensions.<KText>withCopyOf(this._kRenderingFactory.createKText(), this.portLabelFontSize());
      final Procedure1<KText> _function_3 = (KText it_1) -> {
        it_1.setText(label);
        HorizontalAlignment _xifexpression = null;
        Boolean _inlyingPortLabels = this.getInlyingPortLabels();
        if ((_inlyingPortLabels).booleanValue()) {
          _xifexpression = HorizontalAlignment.LEFT;
        } else {
          _xifexpression = HorizontalAlignment.RIGHT;
        }
        this._kRenderingExtensions.<KText>setHorizontalAlignment(it_1, _xifexpression);
        this._kRenderingExtensions.<KText>setVerticalAlignment(it_1, VerticalAlignment.CENTER);
        KAreaPlacementData _createKAreaPlacementData = this._kRenderingFactory.createKAreaPlacementData();
        final Procedure1<KAreaPlacementData> _function_4 = (KAreaPlacementData it_2) -> {
          Boolean _inlyingPortLabels_1 = this.getInlyingPortLabels();
          if ((_inlyingPortLabels_1).booleanValue()) {
            KPosition _createKPosition = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_5 = (KPosition it_3) -> {
              KRightPosition _createKRightPosition = this._kRenderingFactory.createKRightPosition();
              final Procedure1<KRightPosition> _function_6 = (KRightPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KRightPosition _doubleArrow_2 = ObjectExtensions.<KRightPosition>operator_doubleArrow(_createKRightPosition, _function_6);
              it_3.setX(_doubleArrow_2);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_7 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_3 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_7);
              it_3.setY(_doubleArrow_3);
            };
            KPosition _doubleArrow_2 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition, _function_5);
            it_2.setTopLeft(_doubleArrow_2);
            KPosition _createKPosition_1 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_6 = (KPosition it_3) -> {
              KRightPosition _createKRightPosition = this._kRenderingFactory.createKRightPosition();
              final Procedure1<KRightPosition> _function_7 = (KRightPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KRightPosition _doubleArrow_3 = ObjectExtensions.<KRightPosition>operator_doubleArrow(_createKRightPosition, _function_7);
              it_3.setX(_doubleArrow_3);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_8 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_4 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_8);
              it_3.setY(_doubleArrow_4);
            };
            KPosition _doubleArrow_3 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_1, _function_6);
            it_2.setBottomRight(_doubleArrow_3);
          } else {
            KPosition _createKPosition_2 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_7 = (KPosition it_3) -> {
              KLeftPosition _createKLeftPosition = this._kRenderingFactory.createKLeftPosition();
              final Procedure1<KLeftPosition> _function_8 = (KLeftPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KLeftPosition _doubleArrow_4 = ObjectExtensions.<KLeftPosition>operator_doubleArrow(_createKLeftPosition, _function_8);
              it_3.setX(_doubleArrow_4);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_9 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_5 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_9);
              it_3.setY(_doubleArrow_5);
            };
            KPosition _doubleArrow_4 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_2, _function_7);
            it_2.setTopLeft(_doubleArrow_4);
            KPosition _createKPosition_3 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_8 = (KPosition it_3) -> {
              KLeftPosition _createKLeftPosition = this._kRenderingFactory.createKLeftPosition();
              final Procedure1<KLeftPosition> _function_9 = (KLeftPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KLeftPosition _doubleArrow_5 = ObjectExtensions.<KLeftPosition>operator_doubleArrow(_createKLeftPosition, _function_9);
              it_3.setX(_doubleArrow_5);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_10 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_6 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_10);
              it_3.setY(_doubleArrow_6);
            };
            KPosition _doubleArrow_5 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_3, _function_8);
            it_2.setBottomRight(_doubleArrow_5);
          }
        };
        KAreaPlacementData _doubleArrow_2 = ObjectExtensions.<KAreaPlacementData>operator_doubleArrow(_createKAreaPlacementData, _function_4);
        it_1.setPlacementData(_doubleArrow_2);
      };
      KText _doubleArrow_2 = ObjectExtensions.<KText>operator_doubleArrow(_withCopyOf, _function_3);
      _children.add(_doubleArrow_2);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  private KRendering createEPortRendering(final String label) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      KForeground _createKForeground = this._kRenderingFactory.createKForeground();
      final Procedure1<KForeground> _function_1 = (KForeground it_1) -> {
        it_1.setAlpha(255);
        it_1.setColor(this._kRenderingFactory.createKColor());
      };
      KForeground _doubleArrow = ObjectExtensions.<KForeground>operator_doubleArrow(_createKForeground, _function_1);
      this._kRenderingExtensions.setForeground(it, _doubleArrow);
      KBackground _createKBackground = this._kRenderingFactory.createKBackground();
      final Procedure1<KBackground> _function_2 = (KBackground it_1) -> {
        it_1.setAlpha(255);
        it_1.setColor(this._kRenderingFactory.createKColor());
      };
      KBackground _doubleArrow_1 = ObjectExtensions.<KBackground>operator_doubleArrow(_createKBackground, _function_2);
      this._kRenderingExtensions.setBackground(it, _doubleArrow_1);
      EList<KRendering> _children = it.getChildren();
      KText _withCopyOf = this._kRenderingExtensions.<KText>withCopyOf(this._kRenderingFactory.createKText(), this.portLabelFontSize());
      final Procedure1<KText> _function_3 = (KText it_1) -> {
        it_1.setText(label);
        HorizontalAlignment _xifexpression = null;
        Boolean _inlyingPortLabels = this.getInlyingPortLabels();
        if ((_inlyingPortLabels).booleanValue()) {
          _xifexpression = HorizontalAlignment.RIGHT;
        } else {
          _xifexpression = HorizontalAlignment.LEFT;
        }
        this._kRenderingExtensions.<KText>setHorizontalAlignment(it_1, _xifexpression);
        this._kRenderingExtensions.<KText>setVerticalAlignment(it_1, VerticalAlignment.CENTER);
        KAreaPlacementData _createKAreaPlacementData = this._kRenderingFactory.createKAreaPlacementData();
        final Procedure1<KAreaPlacementData> _function_4 = (KAreaPlacementData it_2) -> {
          Boolean _inlyingPortLabels_1 = this.getInlyingPortLabels();
          if ((_inlyingPortLabels_1).booleanValue()) {
            KPosition _createKPosition = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_5 = (KPosition it_3) -> {
              KLeftPosition _createKLeftPosition = this._kRenderingFactory.createKLeftPosition();
              final Procedure1<KLeftPosition> _function_6 = (KLeftPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KLeftPosition _doubleArrow_2 = ObjectExtensions.<KLeftPosition>operator_doubleArrow(_createKLeftPosition, _function_6);
              it_3.setX(_doubleArrow_2);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_7 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_3 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_7);
              it_3.setY(_doubleArrow_3);
            };
            KPosition _doubleArrow_2 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition, _function_5);
            it_2.setTopLeft(_doubleArrow_2);
            KPosition _createKPosition_1 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_6 = (KPosition it_3) -> {
              KLeftPosition _createKLeftPosition = this._kRenderingFactory.createKLeftPosition();
              final Procedure1<KLeftPosition> _function_7 = (KLeftPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KLeftPosition _doubleArrow_3 = ObjectExtensions.<KLeftPosition>operator_doubleArrow(_createKLeftPosition, _function_7);
              it_3.setX(_doubleArrow_3);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_8 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_4 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_8);
              it_3.setY(_doubleArrow_4);
            };
            KPosition _doubleArrow_3 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_1, _function_6);
            it_2.setBottomRight(_doubleArrow_3);
          } else {
            KPosition _createKPosition_2 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_7 = (KPosition it_3) -> {
              KRightPosition _createKRightPosition = this._kRenderingFactory.createKRightPosition();
              final Procedure1<KRightPosition> _function_8 = (KRightPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KRightPosition _doubleArrow_4 = ObjectExtensions.<KRightPosition>operator_doubleArrow(_createKRightPosition, _function_8);
              it_3.setX(_doubleArrow_4);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_9 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_5 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_9);
              it_3.setY(_doubleArrow_5);
            };
            KPosition _doubleArrow_4 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_2, _function_7);
            it_2.setTopLeft(_doubleArrow_4);
            KPosition _createKPosition_3 = this._kRenderingFactory.createKPosition();
            final Procedure1<KPosition> _function_8 = (KPosition it_3) -> {
              KRightPosition _createKRightPosition = this._kRenderingFactory.createKRightPosition();
              final Procedure1<KRightPosition> _function_9 = (KRightPosition it_4) -> {
                it_4.setAbsolute((-2));
              };
              KRightPosition _doubleArrow_5 = ObjectExtensions.<KRightPosition>operator_doubleArrow(_createKRightPosition, _function_9);
              it_3.setX(_doubleArrow_5);
              KTopPosition _createKTopPosition = this._kRenderingFactory.createKTopPosition();
              final Procedure1<KTopPosition> _function_10 = (KTopPosition it_4) -> {
                Float _portEdgeLength = this.getPortEdgeLength();
                float _divide = ((_portEdgeLength).floatValue() / 2);
                it_4.setAbsolute(_divide);
              };
              KTopPosition _doubleArrow_6 = ObjectExtensions.<KTopPosition>operator_doubleArrow(_createKTopPosition, _function_10);
              it_3.setY(_doubleArrow_6);
            };
            KPosition _doubleArrow_5 = ObjectExtensions.<KPosition>operator_doubleArrow(_createKPosition_3, _function_8);
            it_2.setBottomRight(_doubleArrow_5);
          }
        };
        KAreaPlacementData _doubleArrow_2 = ObjectExtensions.<KAreaPlacementData>operator_doubleArrow(_createKAreaPlacementData, _function_4);
        it_1.setPlacementData(_doubleArrow_2);
      };
      KText _doubleArrow_2 = ObjectExtensions.<KText>operator_doubleArrow(_withCopyOf, _function_3);
      _children.add(_doubleArrow_2);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  /**
   * Reveals the first KText element of a port KRendering, which is assumed to be the port label.
   * This is useful for additionally linking it with the business element represented by the port.
   */
  public KText getFirstText(final KPort port) {
    KText _elvis = null;
    KText _data = null;
    if (port!=null) {
      _data=port.<KText>getData(KText.class);
    }
    if (_data != null) {
      _elvis = _data;
    } else {
      KRendering _data_1 = null;
      if (port!=null) {
        _data_1=port.<KRendering>getData(KRendering.class);
      }
      TreeIterator<EObject> _eAllContents = null;
      if (_data_1!=null) {
        _eAllContents=_data_1.eAllContents();
      }
      Iterator<KText> _filter = null;
      if (_eAllContents!=null) {
        _filter=Iterators.<KText>filter(_eAllContents, KText.class);
      }
      KText _head = null;
      if (_filter!=null) {
        _head=IteratorExtensions.<KText>head(_filter);
      }
      _elvis = _head;
    }
    return _elvis;
  }

  public KPort setPortSize(final KPort port, final float with, final float height) {
    port.setSize(with, height);
    return port;
  }

  public KPort setPortPos(final KPort port, final float x, final float y) {
    port.setPos(x, y);
    return port;
  }

  public <T extends Object> KPort addLayoutParam(final KPort port, final IProperty<? super T> property, final T value) {
    port.<T>setProperty(property, value);
    return port;
  }

  public KShapeLayout getShapeLayout(final KPort port) {
    return port;
  }

  public KFontSize portLabelFontSize() {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList();
    final KFontSize _result;
    synchronized (_createCache_portLabelFontSize) {
      if (_createCache_portLabelFontSize.containsKey(_cacheKey)) {
        return _createCache_portLabelFontSize.get(_cacheKey);
      }
      KFontSize _createKFontSize = this._kRenderingFactory.createKFontSize();
      _result = _createKFontSize;
      _createCache_portLabelFontSize.put(_cacheKey, _result);
    }
    _init_portLabelFontSize(_result);
    return _result;
  }

  private final HashMap<ArrayList<?>, KFontSize> _createCache_portLabelFontSize = CollectionLiterals.newHashMap();

  private void _init_portLabelFontSize(final KFontSize it) {
    it.setSize((KPortExtensions.PORT_LABEL_FONT_SIZE).intValue());
  }

  public Maybe<Float> portEdgeLengthMemo() {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList();
    final Maybe<Float> _result;
    synchronized (_createCache_portEdgeLengthMemo) {
      if (_createCache_portEdgeLengthMemo.containsKey(_cacheKey)) {
        return _createCache_portEdgeLengthMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      _result = _create;
      _createCache_portEdgeLengthMemo.put(_cacheKey, _result);
    }
    _init_portEdgeLengthMemo(_result);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_portEdgeLengthMemo = CollectionLiterals.newHashMap();

  private void _init_portEdgeLengthMemo(final Maybe<Float> it) {
    it.set(KPortExtensions.PORT_FIGURE_EDGE_LENGTH);
  }

  public Float getPortEdgeLength() {
    return this.portEdgeLengthMemo().get();
  }

  public void setPortEdgeLength(final Float f) {
    this.portEdgeLengthMemo().set(f);
  }

  private Maybe<Float> portTopOffsetMemo() {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList();
    final Maybe<Float> _result;
    synchronized (_createCache_portTopOffsetMemo) {
      if (_createCache_portTopOffsetMemo.containsKey(_cacheKey)) {
        return _createCache_portTopOffsetMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      _result = _create;
      _createCache_portTopOffsetMemo.put(_cacheKey, _result);
    }
    _init_portTopOffsetMemo(_result);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_portTopOffsetMemo = CollectionLiterals.newHashMap();

  private void _init_portTopOffsetMemo(final Maybe<Float> it) {
    it.set(KPortExtensions.PORT_TOP_OFFSET);
  }

  public Float getPortTopOffset() {
    return this.portTopOffsetMemo().get();
  }

  public void setPortTopOffset(final Float f) {
    this.portTopOffsetMemo().set(f);
  }

  private Maybe<Float> verticalPortSpacingMemo() {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList();
    final Maybe<Float> _result;
    synchronized (_createCache_verticalPortSpacingMemo) {
      if (_createCache_verticalPortSpacingMemo.containsKey(_cacheKey)) {
        return _createCache_verticalPortSpacingMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      _result = _create;
      _createCache_verticalPortSpacingMemo.put(_cacheKey, _result);
    }
    _init_verticalPortSpacingMemo(_result);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_verticalPortSpacingMemo = CollectionLiterals.newHashMap();

  private void _init_verticalPortSpacingMemo(final Maybe<Float> it) {
    it.set(KPortExtensions.VERTICAL_PORT_SPACING);
  }

  public Float getVerticalPortSpacing() {
    return this.verticalPortSpacingMemo().get();
  }

  public void setVerticalPortSpacing(final Float f) {
    this.verticalPortSpacingMemo().set(f);
  }

  private Maybe<Boolean> inlyingPortLabelsMemo() {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList();
    final Maybe<Boolean> _result;
    synchronized (_createCache_inlyingPortLabelsMemo) {
      if (_createCache_inlyingPortLabelsMemo.containsKey(_cacheKey)) {
        return _createCache_inlyingPortLabelsMemo.get(_cacheKey);
      }
      Maybe<Boolean> _create = Maybe.<Boolean>create();
      _result = _create;
      _createCache_inlyingPortLabelsMemo.put(_cacheKey, _result);
    }
    _init_inlyingPortLabelsMemo(_result);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Boolean>> _createCache_inlyingPortLabelsMemo = CollectionLiterals.newHashMap();

  private void _init_inlyingPortLabelsMemo(final Maybe<Boolean> it) {
    it.set(Boolean.valueOf(true));
  }

  public Boolean getInlyingPortLabels() {
    return this.inlyingPortLabelsMemo().get();
  }

  public void setInlyingPortLabels(final Boolean b) {
    this.inlyingPortLabelsMemo().set(b);
  }

  private Maybe<Float> ePortYPositionMemo(final KNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final Maybe<Float> _result;
    synchronized (_createCache_ePortYPositionMemo) {
      if (_createCache_ePortYPositionMemo.containsKey(_cacheKey)) {
        return _createCache_ePortYPositionMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      final Procedure1<Maybe<Float>> _function = (Maybe<Float> it) -> {
        it.set(this.getPortTopOffset());
      };
      Maybe<Float> _doubleArrow = ObjectExtensions.<Maybe<Float>>operator_doubleArrow(_create, _function);
      _result = _doubleArrow;
      _createCache_ePortYPositionMemo.put(_cacheKey, _result);
    }
    _init_ePortYPositionMemo(_result, node);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_ePortYPositionMemo = CollectionLiterals.newHashMap();

  private void _init_ePortYPositionMemo(final Maybe<Float> it, final KNode node) {
  }

  private Maybe<Float> nPortYPositionMemo(final KNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final Maybe<Float> _result;
    synchronized (_createCache_nPortYPositionMemo) {
      if (_createCache_nPortYPositionMemo.containsKey(_cacheKey)) {
        return _createCache_nPortYPositionMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      final Procedure1<Maybe<Float>> _function = (Maybe<Float> it) -> {
        it.set(this.getPortTopOffset());
      };
      Maybe<Float> _doubleArrow = ObjectExtensions.<Maybe<Float>>operator_doubleArrow(_create, _function);
      _result = _doubleArrow;
      _createCache_nPortYPositionMemo.put(_cacheKey, _result);
    }
    _init_nPortYPositionMemo(_result, node);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_nPortYPositionMemo = CollectionLiterals.newHashMap();

  private void _init_nPortYPositionMemo(final Maybe<Float> it, final KNode node) {
  }

  private Maybe<Float> sPortYPositionMemo(final KNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final Maybe<Float> _result;
    synchronized (_createCache_sPortYPositionMemo) {
      if (_createCache_sPortYPositionMemo.containsKey(_cacheKey)) {
        return _createCache_sPortYPositionMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      final Procedure1<Maybe<Float>> _function = (Maybe<Float> it) -> {
        it.set(this.getPortTopOffset());
      };
      Maybe<Float> _doubleArrow = ObjectExtensions.<Maybe<Float>>operator_doubleArrow(_create, _function);
      _result = _doubleArrow;
      _createCache_sPortYPositionMemo.put(_cacheKey, _result);
    }
    _init_sPortYPositionMemo(_result, node);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_sPortYPositionMemo = CollectionLiterals.newHashMap();

  private void _init_sPortYPositionMemo(final Maybe<Float> it, final KNode node) {
  }

  private Maybe<Float> wPortYPositionMemo(final KNode node) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(node);
    final Maybe<Float> _result;
    synchronized (_createCache_wPortYPositionMemo) {
      if (_createCache_wPortYPositionMemo.containsKey(_cacheKey)) {
        return _createCache_wPortYPositionMemo.get(_cacheKey);
      }
      Maybe<Float> _create = Maybe.<Float>create();
      final Procedure1<Maybe<Float>> _function = (Maybe<Float> it) -> {
        it.set(this.getPortTopOffset());
      };
      Maybe<Float> _doubleArrow = ObjectExtensions.<Maybe<Float>>operator_doubleArrow(_create, _function);
      _result = _doubleArrow;
      _createCache_wPortYPositionMemo.put(_cacheKey, _result);
    }
    _init_wPortYPositionMemo(_result, node);
    return _result;
  }

  private final HashMap<ArrayList<?>, Maybe<Float>> _createCache_wPortYPositionMemo = CollectionLiterals.newHashMap();

  private void _init_wPortYPositionMemo(final Maybe<Float> it, final KNode node) {
  }

  public Float nextEPortYPosition(final KNode node) {
    final Maybe<Float> memo = this.ePortYPositionMemo(node);
    final Float f = memo.get();
    Float _verticalPortSpacing = this.getVerticalPortSpacing();
    float _plus = ((f).floatValue() + (_verticalPortSpacing).floatValue());
    memo.set(Float.valueOf(_plus));
    return f;
  }

  public Float nextNPortYPosition(final KNode node) {
    final Maybe<Float> memo = this.nPortYPositionMemo(node);
    final Float f = memo.get();
    Float _verticalPortSpacing = this.getVerticalPortSpacing();
    float _plus = ((f).floatValue() + (_verticalPortSpacing).floatValue());
    memo.set(Float.valueOf(_plus));
    return f;
  }

  public Float nextSPortYPosition(final KNode node) {
    final Maybe<Float> memo = this.sPortYPositionMemo(node);
    final Float f = memo.get();
    Float _verticalPortSpacing = this.getVerticalPortSpacing();
    float _plus = ((f).floatValue() + (_verticalPortSpacing).floatValue());
    memo.set(Float.valueOf(_plus));
    return f;
  }

  public Float nextWPortYPosition(final KNode node) {
    final Maybe<Float> memo = this.wPortYPositionMemo(node);
    final Float f = memo.get();
    Float _verticalPortSpacing = this.getVerticalPortSpacing();
    float _plus = ((f).floatValue() + (_verticalPortSpacing).floatValue());
    memo.set(Float.valueOf(_plus));
    return f;
  }
}
