/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012-2019 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Scope;
import de.cau.cs.kieler.klighd.kgraph.KGraphElement;
import de.cau.cs.kieler.klighd.kgraph.KNode;
import de.cau.cs.kieler.klighd.kgraph.util.KGraphUtil;
import de.cau.cs.kieler.klighd.krendering.ViewSynthesisShared;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import org.eclipse.elk.core.math.KVector;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.options.NodeLabelPlacement;
import org.eclipse.elk.core.util.Pair;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Provides some helpful extension methods for simplifying the composition of KGraph/KRendering-based view models.<br>
 * <br>
 * In order to employ them beyond KLighD diagram syntheses you best declare a field of type
 * {@link KNodeExtensions} in your class and annotate it with {@link Inject Inject}.<br>
 * <br>
 * Make sure to bind the {@link ViewSynthesisShared} annotation in the employed
 * {@link Injector Injector} to a {@link Scope}, e.g. by calling
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createSingletonScopeBindingModule());} or
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createNoScopeBindingModule());}.<br>
 * <br>
 * By means of that {@link Injector Injector} you may get a new instance of your class,
 * or you may inject the above mentioned attribute within instances of your class, e.g. by calling
 * {@code injector.injectMembers(this)} in the constructor of your class.
 * 
 * @author chsch
 * @author nre
 * 
 * @containsExtensions
 */
@ViewSynthesisShared
@SuppressWarnings("all")
public class KNodeExtensions {
  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  private KNode internalCreateNode(final Object... oc) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(oc);
    final KNode _result;
    synchronized (_createCache_internalCreateNode) {
      if (_createCache_internalCreateNode.containsKey(_cacheKey)) {
        return _createCache_internalCreateNode.get(_cacheKey);
      }
      KNode _createInitializedNode = KGraphUtil.createInitializedNode();
      _result = _createInitializedNode;
      _createCache_internalCreateNode.put(_cacheKey, _result);
    }
    _init_internalCreateNode(_result, oc);
    return _result;
  }

  private final HashMap<ArrayList<?>, KNode> _createCache_internalCreateNode = CollectionLiterals.newHashMap();

  private void _init_internalCreateNode(final KNode node, final Object oc) {
  }

  /**
   * The Xtend-generated internal create map for {@link #internalCreateNode} with a more accessible name.
   */
  private HashMap<ArrayList<?>, KNode> getInternalNodeMap() {
    return this._createCache_internalCreateNode;
  }

  /**
   * A convenient test method to check whether or not a specific node exists in the create extension
   */
  public boolean nodeExists(final Object... os) {
    return this.getInternalNodeMap().containsKey(CollectionLiterals.<Object>newArrayList(os));
  }

  /**
   * A convenient method to register a node that was not created via the create extension.
   * @return the previous node associated with the given object(s), or {@code null} if there was no node.
   */
  public KNode registerExistingNode(final KNode node, final Object... os) {
    return this.getInternalNodeMap().put(CollectionLiterals.<Object>newArrayList(os), node);
  }

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  public KNode getNode(final Object o) {
    return this.internalCreateNode(o);
  }

  /**
   * A convenient getter preserving the element image relation.
   */
  public KNode getNode(final Object o1, final Object o2) {
    return this.internalCreateNode(o1, o2);
  }

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  public KNode getNode(final Object... os) {
    return this.internalCreateNode(os);
  }

  /**
   * A convenience method to create a KNode without relating it to a business object.
   */
  public KNode createNode() {
    return KGraphUtil.createInitializedNode();
  }

  /**
   * An alias of {@link #getNode} allowing to express in business that the KNode will
   * be created at this place. It is just syntactic sugar.
   */
  public KNode createNode(final Object o) {
    return this.getNode(o);
  }

  /**
   * An alias of {@link #getNode} allowing to express in business that the KNode will
   * be created at this place. It is just syntactic sugar.
   */
  public KNode createNode(final Object o1, final Object o2) {
    return this.getNode(o1, o2);
  }

  /**
   * An alias of {@link #getNode} allowing to express in business that the KNode will
   * be created at this place. It is just syntactic sugar.
   */
  public KNode createNode(final Object... os) {
    return this.getNode(os);
  }

  public Pair<Float, Float> getNodeSize(final KNode node) {
    Pair<Float, Float> _pair = new Pair<Float, Float>();
    final Procedure1<Pair<Float, Float>> _function = (Pair<Float, Float> it) -> {
      it.setFirst(Float.valueOf(node.getHeight()));
      it.setSecond(Float.valueOf(node.getHeight()));
    };
    return ObjectExtensions.<Pair<Float, Float>>operator_doubleArrow(_pair, _function);
  }

  public float getHeight(final KNode node) {
    return node.getHeight();
  }

  /**
   * Is used in KPortExtensions
   */
  public float getWidth(final KNode node) {
    return node.getWidth();
  }

  public KNode setNodeSize(final KNode node, final float width, final float height) {
    final Procedure1<KNode> _function = (KNode it) -> {
      node.setSize(width, height);
      this.setMinimalNodeSize(it, width, height);
    };
    return ObjectExtensions.<KNode>operator_doubleArrow(node, _function);
  }

  public KNode setWidth(final KNode node, final float width) {
    final float height = node.getHeight();
    this.setNodeSize(node, width, height);
    return node;
  }

  public KNode setHeight(final KNode node, final float height) {
    final float width = node.getWidth();
    this.setNodeSize(node, width, height);
    return node;
  }

  private static final IProperty<KVector> MINIMAL_NODE_SIZE = new Property<KVector>(
    "de.cau.cs.kieler.klighd.minimalNodeSize", new KVector(10d, 10d));

  public KNode setMinimalNodeSize(final KNode node, final float width, final float height) {
    KVector _kVector = new KVector(width, height);
    return this.<KVector>addLayoutParam(node, KNodeExtensions.MINIMAL_NODE_SIZE, _kVector);
  }

  public <T extends Object> KNode addLayoutParam(final KNode node, final IProperty<? super T> property, final T value) {
    if (node!=null) {
      node.<T>setProperty(property, value);
    }
    return node;
  }

  /**
   * Internal helper for setting layout options without the need to check for KNode, KEdge, ...
   */
  private <S extends Object, T extends KGraphElement> T setLayoutOption(final T kgraphElement, final IProperty<S> option, final S value) {
    if (kgraphElement!=null) {
      kgraphElement.<S>setProperty(option, value);
    }
    return kgraphElement;
  }

  /**
   * Configures labels on this node to be outside bottom left-aligned labels by default
   */
  public KNode configureOutsideBottomLeftNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomLeft());
  }

  /**
   * Configures labels on this node to be outside bottom centrally-aligned labels by default
   */
  public KNode configureOutsideBottomCenteredNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomCenter());
  }

  /**
   * Configures labels on this node to be outside bottom right-aligned labels by default
   */
  public KNode configureOutsideBottomRightNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomRight());
  }

  /**
   * Configures labels on this node to be outside top left-aligned labels by default
   */
  public KNode configureOutsideTopLeftNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopLeft());
  }

  /**
   * Configures labels on this node to be outside top centrally-aligned labels by default
   */
  public KNode configureOutsideTopCenteredNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopCenter());
  }

  /**
   * Configures labels on this node to be outside top right-aligned labels by default
   */
  public KNode configureOutsideTopRightNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopRight());
  }

  /**
   * Configures labels on this node to be inside bottom left-aligned labels by default
   */
  public KNode configureInsideBottomLeftNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomLeft());
  }

  /**
   * Configures labels on this node to be inside bottom centrally-aligned labels by default
   */
  public KNode configureInsideBottomCenteredNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomCenter());
  }

  /**
   * Configures labels on this node to be inside bottom right-aligned labels by default
   */
  public KNode configureInsideBottomRightNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomRight());
  }

  /**
   * Configures labels on this node to be inside centrally-aligned labels by default
   */
  public KNode configureInsideCenteredNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideCenter());
  }

  /**
   * Configures labels on this node to be inside top left-aligned labels by default
   */
  public KNode configureInsideTopLeftNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopLeft());
  }

  /**
   * Configures labels on this node to be inside top centrally-aligned labels by default
   */
  public KNode configureInsideTopCenteredNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopCenter());
  }

  /**
   * Configures labels on this node to be inside top right-aligned labels by default
   */
  public KNode configureInsideTopRightNodeLabelPlacement(final KNode node) {
    return this.<EnumSet<NodeLabelPlacement>, KNode>setLayoutOption(node, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopRight());
  }
}
