/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2013 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.base.Objects;
import de.cau.cs.kieler.klighd.kgraph.KGraphData;
import de.cau.cs.kieler.klighd.kgraph.KGraphElement;
import de.cau.cs.kieler.klighd.kgraph.KNode;
import de.cau.cs.kieler.klighd.krendering.KRendering;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KRenderingLibrary;
import de.cau.cs.kieler.klighd.krendering.KRenderingRef;
import de.cau.cs.kieler.klighd.krendering.KStyleHolder;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Extension that allow a convenient use of the {@link KRenderingLibrary} mechanism.
 * 
 * @author uru
 * 
 * @containsExtensions
 */
@SuppressWarnings("all")
public class KLibraryExtensions {
  /**
   * Rendering factory used to instantiate KRendering instances.
   */
  @Extension
  private KRenderingFactory _kRenderingFactory = KRenderingFactory.eINSTANCE;

  /**
   * Retrieves the rendering library valid for the given node. The library must
   * be located in the node's topmost ancestor. If that ancestor does not yet
   * have a rendering library attached, one is generated.
   * 
   * @example
   * val library = node.library
   * 
   * @extensionCategory library
   */
  public KRenderingLibrary getLibrary(final KNode node) {
    KNode parent = node;
    while ((parent.getParent() != null)) {
      parent = parent.getParent();
    }
    KRenderingLibrary library = parent.<KRenderingLibrary>getData(KRenderingLibrary.class);
    if ((library == null)) {
      library = this._kRenderingFactory.createKRenderingLibrary();
      parent.getData().add(library);
    }
    return library;
  }

  /**
   * Retrieves the rendering element that has been identified with the {@code id} string from the
   * given rendering library.
   * 
   * @example
   * val libJunction = edge.source.getFromLibrary("ren_junction")
   * 
   * @extensionCategory library
   */
  public KRenderingRef getFromLibrary(final KRenderingLibrary library, final String id) {
    final Function1<KStyleHolder, Boolean> _function = (KStyleHolder r) -> {
      String _id = r.getId();
      return Boolean.valueOf(Objects.equal(_id, id));
    };
    KStyleHolder _findFirst = IterableExtensions.<KStyleHolder>findFirst(library.getRenderings(), _function);
    final KRendering rendering = ((KRendering) _findFirst);
    if ((rendering != null)) {
      final KRenderingRef ref = this._kRenderingFactory.createKRenderingRef();
      ref.setRendering(rendering);
      return ref;
    }
    return null;
  }

  /**
   * Retrieves the rendering element that has been identified with the {@code id} string from the
   * rendering library valid for the given node.
   * 
   * @example
   * val libJunction = edge.source.getFromLibrary("ren_junction")
   * 
   * @extensionCategory library
   */
  public KRenderingRef getFromLibrary(final KNode node, final String id) {
    final KRenderingLibrary library = this.getLibrary(node);
    return this.getFromLibrary(library, id);
  }

  /**
   * Identifies the passed <code>rendering</code> element with the passed <code>id</code> within
   * the KRendering library. Afterwards the rendering can be retrieved from the library using the
   * {@code getFromLibrary} method.
   * 
   * @example
   * edge.source.library.addToLibrary("ren_junction",
   *  createKRoundedRectangle =&gt; [ rr |
   *   rr.background = color
   *   rr.foreground = color
   *   rr.cornerWidth = 2
   *   rr.cornerHeight = 2
   *   rr.setPointPlacementData(LEFT, 0, 0, TOP, 0, 0,
   *      H_CENTRAL, V_CENTRAL, 0, 0, 10, 10)
   *  ]
   * )
   * 
   * @extensionCategory library
   */
  public KRenderingRef addToLibrary(final KRenderingLibrary library, final String id, final KRendering rendering) {
    rendering.setId(id);
    library.getRenderings().add(rendering);
    final KRenderingRef ref = this._kRenderingFactory.createKRenderingRef();
    ref.setRendering(rendering);
    return ref;
  }

  /**
   * Identifies the passed <code>rendering</code> element with the passed <code>id</code> within
   * the KRendering library valid for the given node. Afterwards the rendering can be retrieved from
   * the library using the {@code getFromLibrary} method.
   * 
   * @example
   * edge.source.addToLibrary("ren_junction",
   *  createKRoundedRectangle =&gt; [ rr |
   *   rr.background = color
   *   rr.foreground = color
   *   rr.cornerWidth = 2
   *   rr.cornerHeight = 2
   *   rr.setPointPlacementData(LEFT, 0, 0, TOP, 0, 0,
   *      H_CENTRAL, V_CENTRAL, 0, 0, 10, 10)
   *  ]
   * )
   * 
   * @extensionCategory library
   */
  public KRenderingRef addToLibrary(final KNode node, final String id, final KRendering rendering) {
    final KRenderingLibrary library = this.getLibrary(node);
    return this.addToLibrary(library, id, rendering);
  }

  public KRenderingRef addRenderingRef(final KGraphElement kge, final KRendering rendering) {
    KRenderingRef _createKRenderingRef = this._kRenderingFactory.createKRenderingRef();
    final Procedure1<KRenderingRef> _function = (KRenderingRef ref) -> {
      ref.setRendering(rendering);
      EList<KGraphData> _data = kge.getData();
      _data.add(ref);
    };
    return ObjectExtensions.<KRenderingRef>operator_doubleArrow(_createKRenderingRef, _function);
  }
}
