/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Scope;
import de.cau.cs.kieler.klighd.kgraph.KEdge;
import de.cau.cs.kieler.klighd.kgraph.KGraphData;
import de.cau.cs.kieler.klighd.kgraph.KGraphElement;
import de.cau.cs.kieler.klighd.kgraph.KLabel;
import de.cau.cs.kieler.klighd.kgraph.KLabeledGraphElement;
import de.cau.cs.kieler.klighd.kgraph.KNode;
import de.cau.cs.kieler.klighd.kgraph.KPort;
import de.cau.cs.kieler.klighd.kgraph.util.KGraphUtil;
import de.cau.cs.kieler.klighd.krendering.KColor;
import de.cau.cs.kieler.klighd.krendering.KRendering;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KText;
import de.cau.cs.kieler.klighd.krendering.ViewSynthesisShared;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.options.EdgeLabelPlacement;
import org.eclipse.elk.core.options.NodeLabelPlacement;
import org.eclipse.elk.core.options.PortLabelPlacement;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Provides some helpful extension methods for simplifying the composition of KGraph/KRendering-based view models.<br>
 * <br>
 * In order to employ them beyond KLighD diagram syntheses you best declare a field of type
 * {@link KNodeExtensions} in your class and annotate it with {@link Inject Inject}.<br>
 * <br>
 * Make sure to bind the {@link ViewSynthesisShared} annotation in the employed
 * {@link Injector Injector} to a {@link Scope}, e.g. by calling
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createSingletonScopeBindingModule());} or
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createNoScopeBindingModule());}.<br>
 * <br>
 * By means of that {@link Injector Injector} you may get a new instance of your class,
 * or you may inject the above mentioned attribute within instances of your class, e.g. by calling
 * {@code injector.injectMembers(this)} in the constructor of your class.
 * 
 * @author chsch
 * @author ssm
 * 
 * @containsExtensions
 */
@ViewSynthesisShared
@SuppressWarnings("all")
public class KLabelExtensions {
  @Extension
  private KRenderingFactory _kRenderingFactory = KRenderingFactory.eINSTANCE;

  @Inject
  @Extension
  private KRenderingExtensions _kRenderingExtensions;

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  private KLabel getLabel(final Iterable<?> o, final KLabeledGraphElement labeledElement) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(o, labeledElement);
    final KLabel _result;
    synchronized (_createCache_getLabel) {
      if (_createCache_getLabel.containsKey(_cacheKey)) {
        return _createCache_getLabel.get(_cacheKey);
      }
      KLabel _createInitializedLabel = KGraphUtil.createInitializedLabel(labeledElement);
      _result = _createInitializedLabel;
      _createCache_getLabel.put(_cacheKey, _result);
    }
    _init_getLabel(_result, o, labeledElement);
    return _result;
  }

  private final HashMap<ArrayList<?>, KLabel> _createCache_getLabel = CollectionLiterals.newHashMap();

  private void _init_getLabel(final KLabel node, final Iterable<?> o, final KLabeledGraphElement labeledElement) {
  }

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  public KLabel getLabel(final Object o, final KLabeledGraphElement labeledElement) {
    return this.getLabel(CollectionLiterals.<Object>newArrayList(o), labeledElement);
  }

  /**
   * Convenience creator of KLabels that are not related to semantic elements.
   * It is just syntactic sugar.
   */
  public KLabel createLabel(final KLabeledGraphElement labeledElement) {
    return KGraphUtil.createInitializedLabel(labeledElement);
  }

  /**
   * An alias of {@link #getLabel(Object, KLabeledGraphElement)} allowing to express in business
   *  that the KLabel will be created at this place. It is just syntactic sugar.
   */
  public KLabel createLabel(final Object o, final KLabeledGraphElement labeledElement) {
    return this.getLabel(CollectionLiterals.<Object>newArrayList(o), labeledElement);
  }

  /**
   * A convenient getter associating the label with to elements
   *  preserving the element image relation by a create extension.
   */
  public KLabel getLabel(final Object o1, final Object o2, final KLabeledGraphElement labeledElement) {
    return this.getLabel(CollectionLiterals.<Object>newArrayList(o1, o2), labeledElement);
  }

  /**
   * An alias of {@link #getLabel(Object, Object, KLabeledGraphElement)} allowing to express in business
   *  that the KLabel will be created at this place. It is just syntactic sugar.
   */
  public KLabel createLabel(final Object o1, final Object o2, final KLabeledGraphElement labeledElement) {
    return this.getLabel(CollectionLiterals.<Object>newArrayList(o1, o2), labeledElement);
  }

  public KLabel setLabelSize(final KLabel label, final float width, final float height) {
    label.setSize(width, height);
    return label;
  }

  public KLabel setLabelPos(final KLabel label, final float x, final float y) {
    label.setPos(x, y);
    return label;
  }

  public <T extends Object> KLabel addLayoutParam(final KLabel label, final IProperty<? super T> property, final T value) {
    if (label!=null) {
      label.<T>setProperty(property, value);
    }
    return label;
  }

  /**
   * Configures an inside centrally-aligned node label!
   */
  public KLabel configureInsideCenteredNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideCenter());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside centrally-aligned node label!
   */
  public KLabel configureInsideCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideCenter());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside centrally-aligned node label!
   */
  public KLabel configureInsideCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideCenter());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideCenteredNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideCenteredNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideCenteredNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideCenteredNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideCenteredNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideCenteredNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside bottom centrally-aligned node label!
   */
  public KLabel configureInsideBottomCenteredNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomCenter());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside bottom centrally-aligned node label!
   */
  public KLabel configureInsideBottomCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomCenter());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside bottom centrally-aligned node label!
   */
  public KLabel configureInsideBottomCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomCenter());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside bottom centrally-aligned node label!
   */
  public KLabel addInsideBottomCenteredNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideBottomCenteredNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside bottom centrally-aligned node label!
   */
  public KLabel addInsideBottomCenteredNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideBottomCenteredNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside bottom centrally-aligned node label!
   */
  public KLabel addInsideBottomCenteredNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideBottomCenteredNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside bottom left-aligned node label!
   */
  public KLabel configureInsideBottomLeftNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomLeft());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside bottom left-aligned node label!
   */
  public KLabel configureInsideBottomLeftNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomLeft());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside bottom left-aligned node label!
   */
  public KLabel configureInsideBottomLeftNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomLeft());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside bottom left-aligned node label!
   */
  public KLabel addInsideBottomLeftNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideBottomLeftNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside bottom left-aligned node label!
   */
  public KLabel addInsideBottomLeftNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideBottomLeftNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside bottom left-aligned node label!
   */
  public KLabel addInsideBottomLeftNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideBottomLeftNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside bottom right-aligned node label!
   */
  public KLabel configureInsideBottomRightNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomRight());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside bottom right-aligned node label!
   */
  public KLabel configureInsideBottomRightNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomRight());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside bottom right-aligned node label!
   */
  public KLabel configureInsideBottomRightNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideBottomRight());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideBottomRightNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideBottomRightNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideBottomRightNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideBottomRightNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside bottom right-aligned node label!
   */
  public KLabel addInsideBottomRightNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideBottomRightNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside top centrally-aligned node label!
   */
  public KLabel configureInsideTopCenteredNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopCenter());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside top centrally-aligned node label!
   */
  public KLabel configureInsideTopCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopCenter());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside top centrally-aligned node label!
   */
  public KLabel configureInsideTopCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopCenter());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside top centrally-aligned node label!
   */
  public KLabel addInsideTopCenteredNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideTopCenteredNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside top centrally-aligned node label!
   */
  public KLabel addInsideTopCenteredNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideTopCenteredNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside top centrally-aligned node label!
   */
  public KLabel addInsideTopCenteredNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideTopCenteredNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside top left-aligned node label!
   */
  public KLabel configureInsideTopLeftNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopLeft());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside top left-aligned node label!
   */
  public KLabel configureInsideTopLeftNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopLeft());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside top left-aligned node label!
   */
  public KLabel configureInsideTopLeftNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopLeft());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside top left-aligned node label!
   */
  public KLabel addInsideTopLeftNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideTopLeftNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside top left-aligned node label!
   */
  public KLabel addInsideTopLeftNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideTopLeftNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside top left-aligned node label!
   */
  public KLabel addInsideTopLeftNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideTopLeftNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside top right-aligned node label!
   */
  public KLabel configureInsideTopRightNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopRight());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside top right-aligned node label!
   */
  public KLabel configureInsideTopRightNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopRight());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside top right-aligned node label!
   */
  public KLabel configureInsideTopRightNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.insideTopRight());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside top right-aligned node label!
   */
  public KLabel addInsideTopRightNodeLabel(final KNode node, final String labelText) {
    return this.configureInsideTopRightNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an inside top right-aligned node label!
   */
  public KLabel addInsideTopRightNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureInsideTopRightNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an inside top right-aligned node label!
   */
  public KLabel addInsideTopRightNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsideTopRightNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside bottom centrally-aligned node label!
   */
  public KLabel configureOutsideBottomCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomCenter());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Configures an outside bottom centrally-aligned node label!
   */
  public KLabel configureOutsideBottomCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomCenter());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside bottom centrally-aligned node label!
   */
  public KLabel configureOutsideBottomCenteredNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomCenter());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Adds an outside bottom centrally-aligned node label!
   */
  public KLabel addOutsideBottomCenteredNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideBottomCenteredNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside bottom centrally-aligned node label!
   */
  public KLabel addOutsideBottomCenteredNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideBottomCenteredNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside bottom centrally-aligned node label!
   */
  public KLabel addOutsideBottomCenteredNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideBottomCenteredNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside bottom left-aligned node label!
   */
  public KLabel configureOutsideBottomLeftNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomLeft());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an outside bottom left-aligned node label!
   */
  public KLabel configureOutsideBottomLeftNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomLeft());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside bottom left-aligned node label!
   */
  public KLabel configureOutsideBottomLeftNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomLeft());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside bottom left-aligned node label!
   */
  public KLabel addOutsideBottomLeftNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideBottomLeftNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside bottom left-aligned node label!
   */
  public KLabel addOutsideBottomLeftNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideBottomLeftNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside bottom left-aligned node label!
   */
  public KLabel addOutsideBottomLeftNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideBottomLeftNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside bottom right-aligned node label!
   */
  public KLabel configureOutsideBottomRightNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomRight());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an outside bottom right-aligned node label!
   */
  public KLabel configureOutsideBottomRightNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomRight());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside bottom right-aligned node label!
   */
  public KLabel configureOutsideBottomRightNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideBottomRight());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside bottom right-aligned node label!
   */
  public KLabel addOutsideBottomRightNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideBottomRightNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside bottom right-aligned node label!
   */
  public KLabel addOutsideBottomRightNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideBottomRightNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside bottom right-aligned node label!
   */
  public KLabel addOutsideBottomRightNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideBottomRightNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside top centrally-aligned node label!
   */
  public KLabel configureOutsideTopCenteredNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopCenter());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an outside top centrally-aligned node label!
   */
  public KLabel configureOutsideTopCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopCenter());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside top centrally-aligned node label!
   */
  public KLabel configureOutsideTopCenteredNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopCenter());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside top centrally-aligned node label!
   */
  public KLabel addOutsideTopCenteredNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideTopCenteredNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside top centrally-aligned node label!
   */
  public KLabel addOutsideTopCenteredNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideTopCenteredNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside top centrally-aligned node label!
   */
  public KLabel addOutsideTopCenteredNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideTopCenteredNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside top left-aligned node label!
   */
  public KLabel configureOutsideTopLeftNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopLeft());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an outside top left-aligned node label!
   */
  public KLabel configureOutsideTopLeftNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopLeft());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside top left-aligned node label!
   */
  public KLabel configureOutsideTopLeftNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopLeft());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside top left-aligned node label!
   */
  public KLabel addOutsideTopLeftNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideTopLeftNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside top left-aligned node label!
   */
  public KLabel addOutsideTopLeftNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideTopLeftNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside top left-aligned node label!
   */
  public KLabel addOutsideTopLeftNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideTopLeftNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an outside top right-aligned node label!
   */
  public KLabel configureOutsideTopRightNodeLabel(final KLabel label, final String labelText) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopRight());
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an outside top right-aligned node label!
   */
  public KLabel configureOutsideTopRightNodeLabel(final KLabel label, final String labelText, final int fontSize) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopRight());
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an outside top right-aligned node label!
   */
  public KLabel configureOutsideTopRightNodeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    this.<EnumSet<NodeLabelPlacement>, KLabel>setLayoutOption(label, CoreOptions.NODE_LABELS_PLACEMENT, NodeLabelPlacement.outsideTopRight());
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside top right-aligned node label!
   */
  public KLabel addOutsideTopRightNodeLabel(final KNode node, final String labelText) {
    return this.configureOutsideTopRightNodeLabel(this.createLabel(node), labelText);
  }

  /**
   * Adds an outside top right-aligned node label!
   */
  public KLabel addOutsideTopRightNodeLabel(final KNode node, final String labelText, final int fontSize) {
    return this.configureOutsideTopRightNodeLabel(this.createLabel(node), labelText, fontSize);
  }

  /**
   * Adds an outside top right-aligned node label!
   */
  public KLabel addOutsideTopRightNodeLabel(final KNode node, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsideTopRightNodeLabel(this.createLabel(node), labelText, fontSize, fontName);
  }

  /**
   * Configures an inside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureInsidePortLabel(final KLabel label, final String labelText) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.INSIDE));
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures an inside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureInsidePortLabel(final KLabel label, final String labelText, final int fontSize) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.INSIDE));
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures an inside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureInsidePortLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.INSIDE));
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an inside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addInsidePortLabel(final KPort port, final String labelText) {
    return this.configureInsidePortLabel(this.createLabel(port), labelText);
  }

  /**
   * Adds an inside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addInsidePortLabel(final KPort port, final String labelText, final int fontSize) {
    return this.configureInsidePortLabel(this.createLabel(port), labelText, fontSize);
  }

  /**
   * Adds an inside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addInsidePortLabel(final KPort port, final String labelText, final int fontSize, final String fontName) {
    return this.configureInsidePortLabel(this.createLabel(port), labelText, fontSize, fontName);
  }

  /**
   * Configures a outside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureOutsidePortLabel(final KLabel label, final String labelText) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.OUTSIDE));
    return this.basicConfigureLabel(label, labelText);
  }

  /**
   * Configures a outside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureOutsidePortLabel(final KLabel label, final String labelText, final int fontSize) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.OUTSIDE));
    return this.basicConfigureLabel(label, labelText, fontSize);
  }

  /**
   * Configures a outside port label!<br>
   * Note that <code>label</code> must be already contained in a {@link KPort} that in turn is contained
   * in a {@link KNode} in order to let the configuration be performed properly.
   */
  public KLabel configureOutsidePortLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    KLabeledGraphElement _parent = label.getParent();
    KGraphElement _parent_1 = null;
    if (_parent!=null) {
      _parent_1=this.getParent(_parent);
    }
    this.<EnumSet<PortLabelPlacement>, KGraphElement>setLayoutOption(_parent_1, CoreOptions.PORT_LABELS_PLACEMENT, EnumSet.<PortLabelPlacement>of(PortLabelPlacement.OUTSIDE));
    return this.basicConfigureLabel(label, labelText, fontSize, fontName);
  }

  /**
   * Adds an outside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addOutsidePortLabel(final KPort port, final String labelText) {
    return this.configureOutsidePortLabel(this.createLabel(port), labelText);
  }

  /**
   * Adds an outside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addOutsidePortLabel(final KPort port, final String labelText, final int fontSize) {
    return this.configureOutsidePortLabel(this.createLabel(port), labelText, fontSize);
  }

  /**
   * Adds an outside label to {@link KPort} <code>port</code>!<br>
   * Note that <code>port</code> must be already contained in a {@link KNode} in order to let the
   * configuration be performed properly.
   */
  public KLabel addOutsidePortLabel(final KPort port, final String labelText, final int fontSize, final String fontName) {
    return this.configureOutsidePortLabel(this.createLabel(port), labelText, fontSize, fontName);
  }

  /**
   * Configures a central (main) edge label, e.g. a state transition guard/effect label!
   */
  public KLabel configureCenterEdgeLabel(final KLabel label, final String labelText) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.CENTER);
  }

  /**
   * Configures a central (main) edge label, e.g. a state transition guard/effect label!
   */
  public KLabel configureCenterEdgeLabel(final KLabel label, final String labelText, final int fontSize) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.CENTER);
  }

  /**
   * Configures a central (main) edge label, e.g. a state transition guard/effect label!
   */
  public KLabel configureCenterEdgeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize, fontName), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.CENTER);
  }

  /**
   * Adds a central (main) label to {@link KEdge} <code>edge</code>.
   */
  public KLabel addCenterEdgeLabel(final KEdge edge, final String labelText) {
    return this.configureCenterEdgeLabel(this.createLabel(edge), labelText);
  }

  /**
   * Adds a central (main) label to {@link KEdge} <code>edge</code>.
   */
  public KLabel addCenterEdgeLabel(final KEdge edge, final String labelText, final int fontSize) {
    return this.configureCenterEdgeLabel(this.createLabel(edge), labelText, fontSize);
  }

  /**
   * Adds a central (main) label to {@link KEdge} <code>edge</code>.
   */
  public KLabel addCenterEdgeLabel(final KEdge edge, final String labelText, final int fontSize, final String fontName) {
    return this.configureCenterEdgeLabel(this.createLabel(edge), labelText, fontSize, fontName);
  }

  /**
   * Configures a head edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureHeadEdgeLabel(final KLabel label, final String labelText) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.HEAD);
  }

  /**
   * Configures a head edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureHeadEdgeLabel(final KLabel label, final String labelText, final int fontSize) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.HEAD);
  }

  /**
   * Configures a head edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureHeadEdgeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize, fontName), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.HEAD);
  }

  /**
   * Adds a head label (target side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addHeadEdgeLabel(final KEdge edge, final String labelText) {
    return this.configureHeadEdgeLabel(this.createLabel(edge), labelText);
  }

  /**
   * Adds a head label (target side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addHeadEdgeLabel(final KEdge edge, final String labelText, final int fontSize) {
    return this.configureHeadEdgeLabel(this.createLabel(edge), labelText, fontSize);
  }

  /**
   * Adds a head label (target side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addHeadEdgeLabel(final KEdge edge, final String labelText, final int fontSize, final String fontName) {
    return this.configureHeadEdgeLabel(this.createLabel(edge), labelText, fontSize, fontName);
  }

  /**
   * Configures a tail edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureTailEdgeLabel(final KLabel label, final String labelText) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.TAIL);
  }

  /**
   * Configures a tail edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureTailEdgeLabel(final KLabel label, final String labelText, final int fontSize) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.TAIL);
  }

  /**
   * Configures a tail edge label, e.g. the cardinality of a relation in an class diagram!
   */
  public KLabel configureTailEdgeLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    return this.<EdgeLabelPlacement>addLayoutParam(this.basicConfigureLabel(label, labelText, fontSize, fontName), 
      CoreOptions.EDGE_LABELS_PLACEMENT, EdgeLabelPlacement.TAIL);
  }

  /**
   * Adds a tail label (source side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addTailEdgeLabel(final KEdge edge, final String labelText) {
    return this.configureTailEdgeLabel(this.createLabel(edge), labelText);
  }

  /**
   * Adds a tail label (source side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addTailEdgeLabel(final KEdge edge, final String labelText, final int fontSize) {
    return this.configureTailEdgeLabel(this.createLabel(edge), labelText, fontSize);
  }

  /**
   * Adds a tail label (source side) to {@link KEdge} <code>edge</code>.
   */
  public KLabel addTailEdgeLabel(final KEdge edge, final String labelText, final int fontSize, final String fontName) {
    return this.configureTailEdgeLabel(this.createLabel(edge), labelText, fontSize, fontName);
  }

  /**
   * The least common denominator of all the 'configure...Label' methods.<br>
   * Is private as it's to be used internally only!
   */
  private KLabel basicConfigureLabel(final KLabel label, final String labelText) {
    label.setText(labelText);
    EList<KGraphData> _data = label.getData();
    KText _createKText = this._kRenderingFactory.createKText();
    _data.add(_createKText);
    return label;
  }

  /**
   * The least common denominator of all the 'configure...Label' methods.<br>
   * Is private as it's to be used internally only!
   */
  private KLabel basicConfigureLabel(final KLabel label, final String labelText, final int fontSize) {
    label.setText(labelText);
    EList<KGraphData> _data = label.getData();
    KText _setFontSize = this._kRenderingExtensions.<KText>setFontSize(this._kRenderingFactory.createKText(), fontSize);
    _data.add(_setFontSize);
    return label;
  }

  /**
   * The least common denominator of all the 'configure...Label' methods.<br>
   * Is private as it's to be used internally only!
   */
  private KLabel basicConfigureLabel(final KLabel label, final String labelText, final int fontSize, final String fontName) {
    label.setText(labelText);
    EList<KGraphData> _data = label.getData();
    KText _setFontName = this._kRenderingExtensions.<KText>setFontName(this._kRenderingExtensions.<KText>setFontSize(this._kRenderingFactory.createKText(), fontSize), fontName);
    _data.add(_setFontName);
    return label;
  }

  /**
   * Internal helper returning the parent KGraphElement if available, <code>null</code> otherwise.
   */
  private KGraphElement getParent(final KGraphElement kgraphElement) {
    final EObject container = kgraphElement.eContainer();
    KGraphElement _switchResult = null;
    boolean _matched = false;
    if (container instanceof KGraphElement) {
      _matched=true;
      _switchResult = ((KGraphElement)container);
    }
    return _switchResult;
  }

  /**
   * Internal helper for setting layout options without the need to check for KNode, KEdge, ...
   */
  private <S extends Object, T extends KGraphElement> T setLayoutOption(final T kgraphElement, final IProperty<S> option, final S value) {
    if (kgraphElement!=null) {
      kgraphElement.<S>setProperty(option, value);
    }
    return kgraphElement;
  }

  /**
   * Reveals the first KText element of a label KRendering, which is assumed to be the label text configuration.
   * This is useful for additionally linking it with the business element represented by the label.
   * 
   * Note: KLabelNodes of the Piccolo2D binding are configured to ignore the KText element while selecting them.
   * Thus, only the KLabel needs to be linked to the source element.
   */
  public KText getFirstText(final KLabel label) {
    KText _elvis = null;
    KText _data = null;
    if (label!=null) {
      _data=label.<KText>getData(KText.class);
    }
    if (_data != null) {
      _elvis = _data;
    } else {
      KRendering _data_1 = null;
      if (label!=null) {
        _data_1=label.<KRendering>getData(KRendering.class);
      }
      TreeIterator<EObject> _eAllContents = null;
      if (_data_1!=null) {
        _eAllContents=_data_1.eAllContents();
      }
      Iterator<KText> _filter = null;
      if (_eAllContents!=null) {
        _filter=Iterators.<KText>filter(_eAllContents, KText.class);
      }
      KText _head = null;
      if (_filter!=null) {
        _head=IteratorExtensions.<KText>head(_filter);
      }
      _elvis = _head;
    }
    return _elvis;
  }

  /**
   * Shortcut for setting the label's foreground color directly.
   */
  public KLabel foreground(final KLabel label, final KColor color) {
    final Procedure1<KLabel> _function = (KLabel it) -> {
      KRendering _kRendering = this._kRenderingExtensions.getKRendering(it);
      this._kRenderingExtensions.setForeground(_kRendering, color);
    };
    return ObjectExtensions.<KLabel>operator_doubleArrow(label, _function);
  }

  /**
   * Shortcut for setting the label's background color directly.
   */
  public KLabel background(final KLabel label, final KColor color) {
    final Procedure1<KLabel> _function = (KLabel it) -> {
      KRendering _kRendering = this._kRenderingExtensions.getKRendering(it);
      this._kRenderingExtensions.setBackground(_kRendering, color);
    };
    return ObjectExtensions.<KLabel>operator_doubleArrow(label, _function);
  }
}
