/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.Scope;
import de.cau.cs.kieler.klighd.kgraph.KEdge;
import de.cau.cs.kieler.klighd.kgraph.KGraphData;
import de.cau.cs.kieler.klighd.kgraph.util.KGraphUtil;
import de.cau.cs.kieler.klighd.krendering.KPolyline;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KRoundedBendsPolyline;
import de.cau.cs.kieler.klighd.krendering.KSpline;
import de.cau.cs.kieler.klighd.krendering.ViewSynthesisShared;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Provides some helpful extension methods for simplifying the composition of KGraph/KRendering-based view models.<br>
 * <br>
 * In order to employ them beyond KLighD diagram syntheses you best declare a field of type
 * {@link KNodeExtensions} in your class and annotate it with {@link Inject Inject}.<br>
 * <br>
 * Make sure to bind the {@link ViewSynthesisShared} annotation in the employed
 * {@link Injector Injector} to a {@link Scope}, e.g. by calling
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createSingletonScopeBindingModule());} or
 * {@code Guice.createInjector(KRenderingExtensionsPlugin.createNoScopeBindingModule());}.<br>
 * <br>
 * By means of that {@link Injector Injector} you may get a new instance of your class,
 * or you may inject the above mentioned attribute within instances of your class, e.g. by calling
 * {@code injector.injectMembers(this)} in the constructor of your class.
 * 
 * @author chsch
 * @author ssm
 * @author nre
 * 
 * @containsExtensions
 */
@ViewSynthesisShared
@SuppressWarnings("all")
public class KEdgeExtensions {
  private static final KRenderingFactory renderingFactory = KRenderingFactory.eINSTANCE;

  @Inject
  @Extension
  private KRenderingExtensions _kRenderingExtensions;

  /**
   * A convenient getter preserving the element image relation by a create extension.
   */
  private KEdge internalCreateEdge(final Object... oc) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(oc);
    final KEdge _result;
    synchronized (_createCache_internalCreateEdge) {
      if (_createCache_internalCreateEdge.containsKey(_cacheKey)) {
        return _createCache_internalCreateEdge.get(_cacheKey);
      }
      KEdge _createInitializedEdge = KGraphUtil.createInitializedEdge();
      _result = _createInitializedEdge;
      _createCache_internalCreateEdge.put(_cacheKey, _result);
    }
    _init_internalCreateEdge(_result, oc);
    return _result;
  }

  private final HashMap<ArrayList<?>, KEdge> _createCache_internalCreateEdge = CollectionLiterals.newHashMap();

  private void _init_internalCreateEdge(final KEdge port, final Object oc) {
  }

  /**
   * The Xtend-generated internal create map for {@link #internalCreateEdge} with a more accessible name.
   */
  private HashMap<ArrayList<?>, KEdge> getInternalEdgeMap() {
    return this._createCache_internalCreateEdge;
  }

  /**
   * A convenient test method to check whether or not a specific edge exists in the create extension
   */
  public boolean edgeExists(final Object o1) {
    return this.getInternalEdgeMap().containsKey(CollectionLiterals.<Object>newArrayList(o1));
  }

  /**
   * A convenient test method to check whether or not a specific edge exists in the create extension
   */
  public boolean edgeExists(final Object o1, final Object o2) {
    return this.getInternalEdgeMap().containsKey(CollectionLiterals.<Object>newArrayList(o1, o2));
  }

  /**
   * A convenient test method to check whether or not a specific edge exists in the create extension
   */
  public boolean edgeExists(final Object o1, final Object o2, final Object o3) {
    return this.getInternalEdgeMap().containsKey(CollectionLiterals.<Object>newArrayList(o1, o2, o3));
  }

  /**
   * A convenient test method to check whether or not a specific edge exists in the create extension
   */
  public boolean edgeExists(final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.getInternalEdgeMap().containsKey(CollectionLiterals.<Object>newArrayList(o1, o2, o3, o4));
  }

  /**
   * A convenient test method to check whether or not a specific edge exists in the create extension
   */
  public boolean edgeExists(final Object... os) {
    return this.getInternalEdgeMap().containsKey(CollectionLiterals.<Object>newArrayList(os));
  }

  /**
   * A convenient port getter based on a single business object preserving the
   * element image relation by a create extension.
   */
  public KEdge getEdge(final Object o1) {
    return this.internalCreateEdge(o1);
  }

  /**
   * A convenient port getter based on a two business objects preserving the
   * element image relation by a create extension.
   */
  public KEdge getEdge(final Object o1, final Object o2) {
    return this.internalCreateEdge(o1, o2);
  }

  /**
   * A convenient port getter based on a three business objects preserving the
   * element image relation by a create extension.
   */
  public KEdge getEdge(final Object o1, final Object o2, final Object o3) {
    return this.internalCreateEdge(o1, o2, o3);
  }

  /**
   * A convenient port getter based on a four business objects preserving the
   * element image relation by a create extension.
   */
  public KEdge getEdge(final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.internalCreateEdge(o1, o2, o3, o4);
  }

  /**
   * A convenient port getter based on a single business object preserving the
   * element image relation by a create extension.
   */
  public KEdge getEdge(final Object... os) {
    return this.internalCreateEdge(os);
  }

  /**
   * A convenience method to create a KEdge without relating it to a business object.
   */
  public KEdge createEdge() {
    return KGraphUtil.createInitializedEdge();
  }

  /**
   * An alias of {@link #getEdge(Object o1)} allowing to express in business that the KEdge will
   * be created at this place. It is just syntactic sugar.
   */
  public KEdge createEdge(final Object o1) {
    return this.getEdge(o1);
  }

  /**
   * An alias of {@link #getEdge(Object o1, Object o2)} allowing to express in business that the
   * KEdge will be created at this place. It is just syntactic sugar.
   */
  public KEdge createEdge(final Object o1, final Object o2) {
    return this.getEdge(o1, o2);
  }

  /**
   * An alias of {@link #getEdge(Object o1, Object o2, Object o3)} allowing to express in business
   * that the KEdge will be created at this place. It is just syntactic sugar.
   */
  public KEdge createEdge(final Object o1, final Object o2, final Object o3) {
    return this.getEdge(o1, o2, o3);
  }

  /**
   * An alias of {@link #getEdge(Object o1, Object o2, Object o3, Object o4)} allowing to express in
   * business that the KEdge will be created at this place. It is just syntactic sugar.
   */
  public KEdge createEdge(final Object o1, final Object o2, final Object o3, final Object o4) {
    return this.getEdge(o1, o2, o3, o4);
  }

  /**
   * An alias of {@link #getEdge(Object o1)} allowing to express in business that the KEdge will
   * be created at this place. It is just syntactic sugar.
   */
  public KEdge createEdge(final Object... os) {
    return this.getEdge(os);
  }

  /**
   * Similar to createEdge, createNewEdge creates a new edge related to an object. Furthermore, the
   * method makes sure a new edge is created and stored in the create extensions.
   */
  public KEdge createNewEdge(final Object o1) {
    int counter = 0;
    while (this.edgeExists(o1, Integer.valueOf(counter))) {
      counter = (counter + 1);
    }
    return this.createEdge(o1, Integer.valueOf(counter));
  }

  /**
   * Similar to createEdge, createNewEdge creates a new edge related to an object. Furthermore, the
   * method makes sure a new edge is created and stored in the create extensions.
   */
  public KEdge createNewEdge(final Object o1, final Object o2) {
    int counter = 0;
    while (this.edgeExists(o1, o2, Integer.valueOf(counter))) {
      counter = (counter + 1);
    }
    return this.createEdge(o1, o2, Integer.valueOf(counter));
  }

  /**
   * Similar to createEdge, createNewEdge creates a new edge related to an object. Furthermore, the
   * method makes sure a new edge is created and stored in the create extensions.
   */
  public KEdge createNewEdge(final Object o1, final Object o2, final Object o3) {
    int counter = 0;
    while (this.edgeExists(o1, o2, o3, Integer.valueOf(counter))) {
      counter = (counter + 1);
    }
    return this.createEdge(o1, o2, o3, Integer.valueOf(counter));
  }

  /**
   * The method getAllEdges retrieves all edges linked to a specific (semantic) object.
   */
  public List<KEdge> getAllEdges(final Object o1) {
    ArrayList<KEdge> _xblockexpression = null;
    {
      int counter = 0;
      final ArrayList<KEdge> edges = CollectionLiterals.<KEdge>newArrayList();
      while (this.edgeExists(o1, Integer.valueOf(counter))) {
        {
          edges.add(this.getEdge(o1, Integer.valueOf(counter)));
          counter = (counter + 1);
        }
      }
      _xblockexpression = edges;
    }
    return _xblockexpression;
  }

  /**
   * The method getAllEdges retrieves all edges linked to a specific (semantic) object.
   */
  public List<KEdge> getAllEdges(final Object o1, final Object o2) {
    ArrayList<KEdge> _xblockexpression = null;
    {
      int counter = 0;
      final ArrayList<KEdge> edges = CollectionLiterals.<KEdge>newArrayList();
      while (this.edgeExists(o1, o2, Integer.valueOf(counter))) {
        {
          edges.add(this.getEdge(o1, o2, Integer.valueOf(counter)));
          counter = (counter + 1);
        }
      }
      _xblockexpression = edges;
    }
    return _xblockexpression;
  }

  /**
   * The method getAllEdges retrieves all edges linked to a specific (semantic) object.
   */
  public List<KEdge> getAllEdges(final Object o1, final Object o2, final Object o3) {
    ArrayList<KEdge> _xblockexpression = null;
    {
      int counter = 0;
      final ArrayList<KEdge> edges = CollectionLiterals.<KEdge>newArrayList();
      while (this.edgeExists(o1, o2, o3, Integer.valueOf(counter))) {
        {
          edges.add(this.getEdge(o1, o2, o3, Integer.valueOf(counter)));
          counter = (counter + 1);
        }
      }
      _xblockexpression = edges;
    }
    return _xblockexpression;
  }

  /**
   * getSemanticObject returns the primary (semantic) object of an edge.
   */
  public Object getSemanticObject(final KEdge edge) {
    final Function2<ArrayList<?>, KEdge, Boolean> _function = (ArrayList<?> p1, KEdge p2) -> {
      return Boolean.valueOf(Objects.equal(p2, edge));
    };
    ArrayList<?> _head = IterableExtensions.<ArrayList<?>>head(MapExtensions.<ArrayList<?>, KEdge>filter(this.getInternalEdgeMap(), _function).keySet());
    return IterableExtensions.head(((List<?>) _head));
  }

  public <T extends Object> KEdge addLayoutParam(final KEdge edge, final IProperty<? super T> property, final T value) {
    edge.<T>setProperty(property, value);
    return edge;
  }

  public KPolyline addPolyline(final KEdge e) {
    KPolyline _createKPolyline = KEdgeExtensions.renderingFactory.createKPolyline();
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      EList<KGraphData> _data = e.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_createKPolyline, _function);
  }

  public KPolyline addPolyline(final KEdge e, final float lineWidth) {
    KPolyline _addPolyline = this.addPolyline(e);
    return this._kRenderingExtensions.setLineWidth(_addPolyline, lineWidth);
  }

  public KRoundedBendsPolyline addRoundedBendsPolyline(final KEdge e, final float bendRadius) {
    KRoundedBendsPolyline _createKRoundedBendsPolyline = KEdgeExtensions.renderingFactory.createKRoundedBendsPolyline();
    final Procedure1<KRoundedBendsPolyline> _function = (KRoundedBendsPolyline it) -> {
      EList<KGraphData> _data = e.getData();
      _data.add(it);
      it.setBendRadius(bendRadius);
    };
    return ObjectExtensions.<KRoundedBendsPolyline>operator_doubleArrow(_createKRoundedBendsPolyline, _function);
  }

  public KRoundedBendsPolyline addRoundedBendsPolyline(final KEdge e, final float bendRadius, final float lineWidth) {
    KRoundedBendsPolyline _addRoundedBendsPolyline = this.addRoundedBendsPolyline(e, bendRadius);
    return this._kRenderingExtensions.setLineWidth(_addRoundedBendsPolyline, lineWidth);
  }

  public KSpline addSpline(final KEdge e) {
    KSpline _createKSpline = KEdgeExtensions.renderingFactory.createKSpline();
    final Procedure1<KSpline> _function = (KSpline it) -> {
      EList<KGraphData> _data = e.getData();
      _data.add(it);
    };
    return ObjectExtensions.<KSpline>operator_doubleArrow(_createKSpline, _function);
  }

  public KSpline addSpline(final KEdge e, final float lineWidth) {
    KSpline _addSpline = this.addSpline(e);
    return this._kRenderingExtensions.setLineWidth(_addSpline, lineWidth);
  }
}
