/**
 * KIELER - Kiel Integrated Environment for Layout Eclipse RichClient
 * 
 * http://www.informatik.uni-kiel.de/rtsys/kieler/
 * 
 * Copyright 2012 by
 * + Kiel University
 *   + Department of Computer Science
 *     + Real-Time and Embedded Systems Group
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package de.cau.cs.kieler.klighd.krendering.extensions;

import com.google.common.collect.Iterables;
import de.cau.cs.kieler.klighd.krendering.KArc;
import de.cau.cs.kieler.klighd.krendering.KBackground;
import de.cau.cs.kieler.klighd.krendering.KChildArea;
import de.cau.cs.kieler.klighd.krendering.KContainerRendering;
import de.cau.cs.kieler.klighd.krendering.KCustomRendering;
import de.cau.cs.kieler.klighd.krendering.KEllipse;
import de.cau.cs.kieler.klighd.krendering.KForeground;
import de.cau.cs.kieler.klighd.krendering.KGridPlacement;
import de.cau.cs.kieler.klighd.krendering.KGridPlacementData;
import de.cau.cs.kieler.klighd.krendering.KImage;
import de.cau.cs.kieler.klighd.krendering.KPolygon;
import de.cau.cs.kieler.klighd.krendering.KPolyline;
import de.cau.cs.kieler.klighd.krendering.KPosition;
import de.cau.cs.kieler.klighd.krendering.KRectangle;
import de.cau.cs.kieler.klighd.krendering.KRendering;
import de.cau.cs.kieler.klighd.krendering.KRenderingFactory;
import de.cau.cs.kieler.klighd.krendering.KRoundedRectangle;
import de.cau.cs.kieler.klighd.krendering.KText;
import de.cau.cs.kieler.klighd.krendering.LineJoin;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * This class contains lots of convenient helper functions for composing and configuring
 * KRendering-based view models. In order to be consistent with the further extension classes
 * the extension methods are non-static ones requiring this class to be instantiated. Since this
 * class doesn't declare any fields (i.e. required memory) except the reference of further extensions
 * classes the instantiation should not be a problem. The instantiation may be done directly by calling
 * 'new KContainerRenderingExtensions()' or by delegating that to a dependency injection framework.<br>
 * <br>
 * NOTE: Do NOT introduce <i>create extensions</i> or other continuous memory in that class!
 * 
 * @author chsch, alb
 * 
 * @containsExtensions
 */
@SuppressWarnings("all")
public class KContainerRenderingExtensions {
  @Extension
  private KRenderingFactory _kRenderingFactory = KRenderingFactory.eINSTANCE;

  @Extension
  private KRenderingExtensions _kRenderingExtensions = new KRenderingExtensions();

  @Extension
  private KColorExtensions _kColorExtensions = new KColorExtensions();

  /**
   * @return the child!
   */
  public <T extends KRendering> T addChild(final KContainerRendering parent, final T child) {
    final Procedure1<T> _function = (T it) -> {
      parent.getChildren().add(it);
    };
    return ObjectExtensions.<T>operator_doubleArrow(child, _function);
  }

  public KChildArea addChildArea(final KContainerRendering cr) {
    KChildArea _createKChildArea = this._kRenderingFactory.createKChildArea();
    final Procedure1<KChildArea> _function = (KChildArea it) -> {
      cr.getChildren().removeAll(IterableExtensions.<KChildArea>toList(Iterables.<KChildArea>filter(cr.getChildren(), KChildArea.class)));
      cr.getChildren().add(it);
    };
    return ObjectExtensions.<KChildArea>operator_doubleArrow(_createKChildArea, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KArc addArc(final KContainerRendering cr) {
    KArc _createKArc = this._kRenderingFactory.createKArc();
    final Procedure1<KArc> _function = (KArc it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
    };
    return ObjectExtensions.<KArc>operator_doubleArrow(_createKArc, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KEllipse addEllipse(final KContainerRendering cr) {
    KEllipse _createKEllipse = this._kRenderingFactory.createKEllipse();
    final Procedure1<KEllipse> _function = (KEllipse it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
    };
    return ObjectExtensions.<KEllipse>operator_doubleArrow(_createKEllipse, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KPolygon addPolygon(final KContainerRendering cr) {
    KPolygon _createKPolygon = this._kRenderingFactory.createKPolygon();
    final Procedure1<KPolygon> _function = (KPolygon it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
    };
    return ObjectExtensions.<KPolygon>operator_doubleArrow(_createKPolygon, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KRectangle addRectangle(final KContainerRendering cr) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KRoundedRectangle addRoundedRectangle(final KContainerRendering cr, final float cWidth, final float cHeight) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  /**
   * @extensionCategory composition
   */
  public KRoundedRectangle addRoundedRectangle(final KContainerRendering cr, final float cWidth, final float cHeight, final float lineWidth) {
    KRoundedRectangle _createKRoundedRectangle = this._kRenderingFactory.createKRoundedRectangle();
    final Procedure1<KRoundedRectangle> _function = (KRoundedRectangle it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setCornerWidth(cWidth);
      it.setCornerHeight(cHeight);
      this._kRenderingExtensions.setLineWidth(it, lineWidth);
    };
    return ObjectExtensions.<KRoundedRectangle>operator_doubleArrow(_createKRoundedRectangle, _function);
  }

  public KText addText(final KContainerRendering cr, final String text) {
    KText _createKText = this._kRenderingFactory.createKText();
    final Procedure1<KText> _function = (KText it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setText(text);
    };
    return ObjectExtensions.<KText>operator_doubleArrow(_createKText, _function);
  }

  public KImage addImage(final KContainerRendering cr, final String bundleId, final String imagePath) {
    KImage _createKImage = this._kRenderingFactory.createKImage();
    final Procedure1<KImage> _function = (KImage it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setBundleName(bundleId);
      it.setImagePath(imagePath);
    };
    return ObjectExtensions.<KImage>operator_doubleArrow(_createKImage, _function);
  }

  public KImage addImage(final KContainerRendering cr, final Object imageObj) {
    KImage _createKImage = this._kRenderingFactory.createKImage();
    final Procedure1<KImage> _function = (KImage it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setImageObject(imageObj);
    };
    return ObjectExtensions.<KImage>operator_doubleArrow(_createKImage, _function);
  }

  /**
   * Sets a {@link KRectangle} clip shape on the provided <code>image</code>.<br>
   * Set area or point placement data on this (returned) rectangle as usual in order
   * to determine the clip area.
   * 
   * @extensionCategory composition
   * 
   * @example
   * ...addImage(...).addRectangularClip
   *          .addAreaPlacementData.from(LEFT, 3, 0, TOP, 3, 0).to(RIGHT, 3, 0, BOTTOM, 3, 0);
   */
  public KRectangle addRectangularClip(final KImage image) {
    final KRectangle rect = this._kRenderingFactory.createKRectangle();
    image.setClipShape(rect);
    return rect;
  }

  /**
   * Sets a {@link KEllipse} clip shape on the provided <code>image</code>.<br>
   * Set area or point placement data on this (returned) ellipse as usual in order
   * to determine the clip area.
   * 
   * @extensionCategory composition
   * 
   * @example
   * ...addImage(...).addEllipticalClip
   *          .addAreaPlacementData.from(LEFT, 3, 0, TOP, 3, 0).to(RIGHT, 3, 0, BOTTOM, 3, 0);
   */
  public KEllipse addEllipticalClip(final KImage image) {
    final KEllipse ellipse = this._kRenderingFactory.createKEllipse();
    image.setClipShape(ellipse);
    return ellipse;
  }

  /**
   * Sets a {@link KPolygon} clip shape on the provided <code>image</code>.<br>
   * Add {@link KPosition KPositions} on this (returned) polygon as usual in order
   * to determine the clip area.
   * 
   * @extensionCategory composition
   * 
   * @example
   * ..addImage(...).addPolygonClip.addKPosition(LEFT, 3, 0, TOP, 3, 0)
   *          .addKPosition(RIGHT, 3, 0, TOP, 3, 0)
   *          .addKPosition(LEFT, 0, 0.5f, BOTTOM, 3, 0);
   */
  public KPolygon addPolygonClip(final KImage image) {
    final KPolygon polygon = this._kRenderingFactory.createKPolygon();
    image.setClipShape(polygon);
    return polygon;
  }

  public KCustomRendering addCustomRendering(final KContainerRendering cr) {
    KCustomRendering _createKCustomRendering = this._kRenderingFactory.createKCustomRendering();
    final Procedure1<KCustomRendering> _function = (KCustomRendering it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
    };
    return ObjectExtensions.<KCustomRendering>operator_doubleArrow(_createKCustomRendering, _function);
  }

  public KCustomRendering addCustomRendering(final KContainerRendering cr, final Object figureObject) {
    KCustomRendering _createKCustomRendering = this._kRenderingFactory.createKCustomRendering();
    final Procedure1<KCustomRendering> _function = (KCustomRendering it) -> {
      EList<KRendering> _children = cr.getChildren();
      _children.add(it);
      it.setFigureObject(figureObject);
    };
    return ObjectExtensions.<KCustomRendering>operator_doubleArrow(_createKCustomRendering, _function);
  }

  /**
   * @example
   * rectangle.setGridPlacement(1).from(LEFT, 0, 0, TOP, padding - 3, 0).to(RIGHT, 0, 0, BOTTOM, 0, 0)
   * 
   * @extensionCategory microLayout
   */
  public KGridPlacement setGridPlacement(final KContainerRendering cr, final int cols) {
    KGridPlacement _createKGridPlacement = this._kRenderingFactory.createKGridPlacement();
    final Procedure1<KGridPlacement> _function = (KGridPlacement it) -> {
      cr.setChildPlacement(it);
      it.setNumColumns(cols);
    };
    return ObjectExtensions.<KGridPlacement>operator_doubleArrow(_createKGridPlacement, _function);
  }

  public KRectangle addGridBox(final KContainerRendering cr, final float widthHint, final float heightHint, final KPosition topLeft, final KPosition bottomRight) {
    KRectangle _createKRectangle = this._kRenderingFactory.createKRectangle();
    final Procedure1<KRectangle> _function = (KRectangle it) -> {
      cr.getChildren().add(it);
      KBackground _createKBackground = this._kRenderingFactory.createKBackground();
      final Procedure1<KBackground> _function_1 = (KBackground it_1) -> {
        it_1.setAlpha(0);
      };
      KBackground _doubleArrow = ObjectExtensions.<KBackground>operator_doubleArrow(_createKBackground, _function_1);
      this._kRenderingExtensions.<KRectangle>setBackground(it, _doubleArrow);
      KForeground _createKForeground = this._kRenderingFactory.createKForeground();
      final Procedure1<KForeground> _function_2 = (KForeground it_1) -> {
        it_1.setAlpha(0);
      };
      KForeground _doubleArrow_1 = ObjectExtensions.<KForeground>operator_doubleArrow(_createKForeground, _function_2);
      this._kRenderingExtensions.<KRectangle>setForeground(it, _doubleArrow_1);
      this._kRenderingExtensions.<KRectangle>setGridPlacementData(it, widthHint, heightHint, topLeft, bottomRight);
    };
    return ObjectExtensions.<KRectangle>operator_doubleArrow(_createKRectangle, _function);
  }

  public KPolyline addPolyline(final KContainerRendering cr) {
    KPolyline _createKPolyline = this._kRenderingFactory.createKPolyline();
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this.<KPolyline>addChild(cr, it);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_createKPolyline, _function);
  }

  public KPolyline addPolyline(final KContainerRendering cr, final float lineWidth) {
    KPolyline _createKPolyline = this._kRenderingFactory.createKPolyline();
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this.<KPolyline>addChild(cr, it);
      this._kRenderingExtensions.setLineWidth(it, lineWidth);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_createKPolyline, _function);
  }

  public KPolyline addPolyline(final KContainerRendering cr, final float lineWidth, final List<KPosition> points) {
    KPolyline _createKPolyline = this._kRenderingFactory.createKPolyline();
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this.<KPolyline>addChild(cr, it);
      this._kRenderingExtensions.setLineWidth(it, lineWidth);
      EList<KPosition> _points = it.getPoints();
      Iterables.<KPosition>addAll(_points, points);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_createKPolyline, _function);
  }

  public KPolyline addHorizontalLine(final KContainerRendering cr, final float leftRightAbsIndent) {
    KPolyline _addChild = this.<KPolyline>addChild(cr, this._kRenderingFactory.createKPolyline());
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this._kRenderingExtensions.setLineWidth(it, 1);
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, leftRightAbsIndent, 0, this._kRenderingExtensions.TOP, 0, 0.5f);
      _points.add(_createKPosition);
      EList<KPosition> _points_1 = it.getPoints();
      KPosition _createKPosition_1 = this._kRenderingExtensions.createKPosition(PositionReferenceX.RIGHT, leftRightAbsIndent, 0, this._kRenderingExtensions.TOP, 0, 0.5f);
      _points_1.add(_createKPosition_1);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(((KPolyline) _addChild), _function);
  }

  public KPolyline addHorizontalLine(final KContainerRendering cr, final float leftRightAbsIndent, final float lineWidth) {
    KPolyline _addHorizontalLine = this.addHorizontalLine(cr, leftRightAbsIndent);
    return this._kRenderingExtensions.setLineWidth(_addHorizontalLine, lineWidth);
  }

  public KPolyline addHorizontalLine(final KContainerRendering cr, final PositionReferenceY y, final float absIndent) {
    KPolyline _addChild = this.<KPolyline>addChild(cr, this._kRenderingFactory.createKPolyline());
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this._kRenderingExtensions.setLineWidth(it, 1);
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, absIndent, 0, y, 0, 0);
      _points.add(_createKPosition);
      EList<KPosition> _points_1 = it.getPoints();
      KPosition _createKPosition_1 = this._kRenderingExtensions.createKPosition(PositionReferenceX.RIGHT, absIndent, 0, y, 0, 0);
      _points_1.add(_createKPosition_1);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(((KPolyline) _addChild), _function);
  }

  public KPolyline addHorizontalLine(final KContainerRendering cr, final PositionReferenceY y, final float absIndent, final float lineWidth) {
    KPolyline _addHorizontalLine = this.addHorizontalLine(cr, y, absIndent);
    return this._kRenderingExtensions.setLineWidth(_addHorizontalLine, lineWidth);
  }

  public KPolyline addVerticalLine(final KContainerRendering cr, final PositionReferenceX x, final float absIndent) {
    KPolyline _addChild = this.<KPolyline>addChild(cr, this._kRenderingFactory.createKPolyline());
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this._kRenderingExtensions.setLineWidth(it, 1);
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this._kRenderingExtensions.createKPosition(x, 0, 0, this._kRenderingExtensions.TOP, absIndent, 0);
      _points.add(_createKPosition);
      EList<KPosition> _points_1 = it.getPoints();
      KPosition _createKPosition_1 = this._kRenderingExtensions.createKPosition(x, 0, 0, this._kRenderingExtensions.BOTTOM, absIndent, 0);
      _points_1.add(_createKPosition_1);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_addChild, _function);
  }

  public KPolyline addVerticalLine(final KContainerRendering cr, final PositionReferenceX x, final float absIndent, final float lineWidth) {
    KPolyline _addVerticalLine = this.addVerticalLine(cr, x, absIndent);
    return this._kRenderingExtensions.setLineWidth(_addVerticalLine, lineWidth);
  }

  public KPolyline addHorizontalSeperatorLine(final KContainerRendering cr, final float lineWidth, final int spacing) {
    KPolyline _createKPolyline = this._kRenderingFactory.createKPolyline();
    final Procedure1<KPolyline> _function = (KPolyline it) -> {
      this.<KPolyline>addChild(cr, it);
      this._kRenderingExtensions.<KPolyline>setLineWidth(it, lineWidth);
      it.getPoints().add(this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0, PositionReferenceY.TOP, 0, 0.5f));
      it.getPoints().add(this._kRenderingExtensions.createKPosition(PositionReferenceX.RIGHT, 0, 0, PositionReferenceY.TOP, 0, 0.5f));
      KGridPlacementData _createKGridPlacementData = this._kRenderingFactory.createKGridPlacementData();
      final Procedure1<KGridPlacementData> _function_1 = (KGridPlacementData it_1) -> {
        it_1.setMinCellHeight((lineWidth + spacing));
      };
      KGridPlacementData _doubleArrow = ObjectExtensions.<KGridPlacementData>operator_doubleArrow(_createKGridPlacementData, _function_1);
      it.setPlacementData(_doubleArrow);
    };
    return ObjectExtensions.<KPolyline>operator_doubleArrow(_createKPolyline, _function);
  }

  public KPolygon drawArrow(final KContainerRendering cr) {
    KPolygon _createKPolygon = this._kRenderingFactory.createKPolygon();
    final Procedure1<KPolygon> _function = (KPolygon it) -> {
      this._kRenderingExtensions.<KPolygon>setLineJoin(this._kRenderingExtensions.<KPolygon>setBackground(this._kRenderingExtensions.<KPolygon>withCopyOf(this._kRenderingExtensions.<KPolygon>withCopyOf(this.<KPolygon>addChild(cr, it), this._kRenderingExtensions.getLineWidth(cr)), this._kRenderingExtensions.getForeground(cr)), this._kRenderingExtensions.getForeground(cr)), LineJoin.JOIN_ROUND);
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0, PositionReferenceY.TOP, 0, 0);
      _points.add(_createKPosition);
      EList<KPosition> _points_1 = it.getPoints();
      KPosition _createKPosition_1 = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0.4f, PositionReferenceY.TOP, 0, 0.5f);
      _points_1.add(_createKPosition_1);
      EList<KPosition> _points_2 = it.getPoints();
      KPosition _createKPosition_2 = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0, PositionReferenceY.BOTTOM, 0, 0);
      _points_2.add(_createKPosition_2);
      EList<KPosition> _points_3 = it.getPoints();
      KPosition _createKPosition_3 = this._kRenderingExtensions.createKPosition(PositionReferenceX.RIGHT, 0, 0, PositionReferenceY.BOTTOM, 0, 0.5f);
      _points_3.add(_createKPosition_3);
    };
    return ObjectExtensions.<KPolygon>operator_doubleArrow(_createKPolygon, _function);
  }

  public KPolygon drawTriangle(final KContainerRendering cr) {
    KPolygon _createKPolygon = this._kRenderingFactory.createKPolygon();
    final Procedure1<KPolygon> _function = (KPolygon it) -> {
      this._kRenderingExtensions.<KPolygon>setBackground(this._kRenderingExtensions.<KPolygon>withCopyOf(this._kRenderingExtensions.<KPolygon>withCopyOf(this.<KPolygon>addChild(cr, it), this._kRenderingExtensions.getLineWidth(cr)), this._kRenderingExtensions.getForeground(cr)), this._kColorExtensions.getColor("white"));
      EList<KPosition> _points = it.getPoints();
      KPosition _createKPosition = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0, PositionReferenceY.TOP, 0, 0);
      _points.add(_createKPosition);
      EList<KPosition> _points_1 = it.getPoints();
      KPosition _createKPosition_1 = this._kRenderingExtensions.createKPosition(PositionReferenceX.LEFT, 0, 0, PositionReferenceY.BOTTOM, 0, 0);
      _points_1.add(_createKPosition_1);
      EList<KPosition> _points_2 = it.getPoints();
      KPosition _createKPosition_2 = this._kRenderingExtensions.createKPosition(PositionReferenceX.RIGHT, 0, 0, PositionReferenceY.BOTTOM, 0, 0.5f);
      _points_2.add(_createKPosition_2);
    };
    return ObjectExtensions.<KPolygon>operator_doubleArrow(_createKPolygon, _function);
  }
}
