/*
 * Decompiled with CFR 0.152.
 */
package de.carne.filescanner.engine;

import de.carne.filescanner.engine.FileScannerProgress;
import de.carne.filescanner.engine.FileScannerResult;
import de.carne.filescanner.engine.FileScannerResultBuilder;
import de.carne.filescanner.engine.FileScannerResultDecodeContext;
import de.carne.filescanner.engine.FileScannerRunnableV;
import de.carne.filescanner.engine.FileScannerStatus;
import de.carne.filescanner.engine.FormatDecodeException;
import de.carne.filescanner.engine.FormatMatcherBuilder;
import de.carne.filescanner.engine.input.BufferedFileChannelInput;
import de.carne.filescanner.engine.input.DecodedInputMapper;
import de.carne.filescanner.engine.input.FileScannerInput;
import de.carne.filescanner.engine.input.FileScannerInputRange;
import de.carne.filescanner.engine.input.InputDecodeCache;
import de.carne.filescanner.engine.input.InputDecoderTable;
import de.carne.filescanner.engine.spi.Format;
import de.carne.filescanner.engine.util.HexFormat;
import de.carne.util.Exceptions;
import de.carne.util.SystemProperties;
import de.carne.util.logging.Log;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;

public final class FileScanner
implements Closeable {
    private static final Log LOG = new Log();
    private static final int THREAD_COUNT = SystemProperties.intValue(FileScanner.class, (String)".threadCount", (int)Runtime.getRuntime().availableProcessors());
    private static final long STOP_TIMEOUT = SystemProperties.longValue(FileScanner.class, (String)".stopTimeout", (long)5000L);
    private final ExecutorService threadPool = Executors.newFixedThreadPool(THREAD_COUNT);
    private final FormatMatcherBuilder formatMatcherBuilder;
    private final InputDecodeCache inputDecodeCache;
    private final BufferedFileChannelInput rootInput;
    private final FileScannerResultBuilder rootResult;
    private final FileScannerStatus status;
    private int runningScanTasks = 0;
    private long scanStartedNanos = 0L;
    private long scanTimeNanos = 0L;
    private long lastProgressTimeNanos = 0L;
    private long totalInputBytes = 0L;
    private long scannedBytes = 0L;
    private boolean suppressStatus = false;

    private FileScanner(BufferedFileChannelInput input, Collection<Format> formats, FileScannerStatus status) throws IOException {
        this.formatMatcherBuilder = new FormatMatcherBuilder(formats);
        this.inputDecodeCache = new InputDecodeCache(this.threadPool::isShutdown);
        this.rootInput = input;
        this.rootResult = FileScannerResultBuilder.inputResult(this.rootInput);
        this.status = status;
        this.rootResult.updateAndCommit(-1L, true);
        this.queueScanTask(() -> this.scanRootInput(this.rootResult));
    }

    private void scanRootInput(FileScannerResultBuilder inputResult) {
        LOG.info("Starting scan (using {0} threads)...", new Object[]{THREAD_COUNT});
        this.scanStarted();
        this.scanInput(inputResult);
    }

    private void scanInput(FileScannerResultBuilder inputResult) {
        LOG.notice("Scanning input ''{0}''...", new Object[]{inputResult.name()});
        this.scanProgress(inputResult.size(), 0L);
        this.onScanResultCommit(inputResult);
        try {
            this.scanInputRange(inputResult, inputResult.input(), inputResult.start(), inputResult.end());
        }
        catch (IOException e) {
            LOG.warning((Throwable)e, "An exception occurred while scanning input ''{0}''", new Object[]{inputResult.name()});
            this.callStatus(() -> this.status.scanException(this, e));
        }
    }

    private void scanInputRange(FileScannerResultBuilder parent, FileScannerInput input, long start, long end) throws IOException {
        FormatMatcherBuilder.Matcher formatMatcher = this.formatMatcherBuilder.matcher();
        long scanPosition = start;
        FileScannerInputRange scanRange = input.range(scanPosition, end);
        while (scanPosition < end && !this.threadPool.isShutdown()) {
            List<Format> matchingFormats = formatMatcher.match(scanRange, scanPosition);
            FileScannerResult decodeResult = null;
            long decodeResultSize = -1L;
            for (Format format : matchingFormats) {
                FileScannerResultDecodeContext context = new FileScannerResultDecodeContext(this, parent, scanRange, scanPosition);
                try {
                    decodeResult = format.decode(context);
                    decodeResultSize = decodeResult.size();
                    if (decodeResultSize > 0L) break;
                    LOG.info("Format ''{0}'' failed to decode input", new Object[]{format.name()});
                }
                catch (FormatDecodeException e) {
                    LOG.warning((Throwable)e, "Format ''{0}'' failed to decode input", new Object[]{format.name()});
                }
            }
            if (decodeResult != null && decodeResultSize > 0L) {
                this.scanProgress(0L, decodeResultSize);
                long decodeResultStart = decodeResult.start();
                if (scanPosition < decodeResultStart) {
                    long reScanPosition = scanPosition;
                    this.queueScanTask(() -> this.scanInputRange(parent, input, reScanPosition, decodeResultStart));
                }
                scanPosition = decodeResultStart + decodeResultSize;
                scanRange = input.range(scanPosition, end);
                continue;
            }
            this.scanProgress(0L, 1L);
            ++scanPosition;
        }
    }

    InputDecodeCache.DecodeResult decodeInputs(DecodedInputMapper decodedInputMapper, InputDecoderTable inputDecoderTable, FileScannerInput input, long start) throws IOException {
        return this.inputDecodeCache.decodeInputs(decodedInputMapper, inputDecoderTable, input, start);
    }

    void queueInputResults(Collection<FileScannerResultBuilder> inputResults) {
        for (FileScannerResultBuilder inputResult : inputResults) {
            this.queueScanTask(() -> this.scanInput(inputResult));
        }
    }

    void onScanResultCommit(FileScannerResultBuilder scanResult) {
        this.callStatus(() -> this.status.scanResult(this, scanResult));
    }

    public static FileScanner scan(Path file, Collection<Format> formats, FileScannerStatus status) throws IOException {
        return new FileScanner(FileScannerInput.open(file), formats, status);
    }

    public void stop(boolean wait) {
        this.stop0(wait);
        if (wait) {
            try {
                boolean terminated = this.threadPool.awaitTermination(STOP_TIMEOUT, TimeUnit.MILLISECONDS);
                if (!terminated) {
                    LOG.warning("Failed to stop all scan threads", new Object[0]);
                }
            }
            catch (InterruptedException e) {
                LOG.warning((Throwable)e, "Scan stop has been interrupted", new Object[0]);
                Thread.currentThread().interrupt();
            }
        }
    }

    private synchronized void stop0(boolean wait) {
        LOG.info("Stopping scan threads...", new Object[0]);
        this.threadPool.shutdown();
        this.suppressStatus = wait;
    }

    public synchronized boolean isScanning() {
        return this.runningScanTasks > 0;
    }

    public synchronized FileScannerProgress progress() {
        return new FileScannerProgress(this.scanStartedNanos, this.scanTimeNanos, this.scannedBytes, this.totalInputBytes);
    }

    public FileScannerResult result() {
        return this.rootResult;
    }

    public @NonNull FileScannerResult[] getResultPath(byte[] resultKey) {
        StringBuilder resultKeyString = new StringBuilder();
        ArrayList<FileScannerResult> results = new ArrayList<FileScannerResult>();
        FileScannerResult lastResult = this.rootResult;
        resultKeyString.append(HexFormat.formatLong(0L));
        results.add(lastResult);
        int resultKeyIndex = 0;
        while (resultKeyIndex < resultKey.length) {
            FileScannerResult currentResult = null;
            FileScannerResult[] lastResultChildren = lastResult.children();
            if (lastResult.type() != FileScannerResult.Type.ENCODED_INPUT) {
                long resultStart = ((long)resultKey[resultKeyIndex] & 0xFFL) << 56 | ((long)resultKey[resultKeyIndex + 1] & 0xFFL) << 48 | ((long)resultKey[resultKeyIndex + 2] & 0xFFL) << 40 | ((long)resultKey[resultKeyIndex + 3] & 0xFFL) << 32 | ((long)resultKey[resultKeyIndex + 4] & 0xFFL) << 24 | ((long)resultKey[resultKeyIndex + 5] & 0xFFL) << 16 | ((long)resultKey[resultKeyIndex + 6] & 0xFFL) << 8 | (long)resultKey[resultKeyIndex + 7] & 0xFFL;
                resultKeyString.append(", ").append(HexFormat.formatLong(resultStart));
                currentResult = this.getResultByStart(lastResultChildren, resultStart);
                resultKeyIndex += 8;
            } else {
                int resultIndex = (resultKey[resultKeyIndex] & 0xFF) << 24 | (resultKey[resultKeyIndex + 1] & 0xFF) << 16 | (resultKey[resultKeyIndex + 2] & 0xFF) << 8 | resultKey[resultKeyIndex + 3] & 0xFF;
                resultKeyString.append(", ").append(HexFormat.formatInt(resultIndex));
                currentResult = lastResultChildren[resultIndex];
                resultKeyIndex += 4;
            }
            if (currentResult == null) {
                throw new IllegalArgumentException("Invalid result key: " + resultKeyString);
            }
            results.add(currentResult);
            lastResult = currentResult;
        }
        return results.toArray(new FileScannerResult[results.size()]);
    }

    private @Nullable FileScannerResult getResultByStart(FileScannerResult[] results, long resultStart) {
        int first = 0;
        int last = results.length - 1;
        FileScannerResult result = null;
        while (result == null && first <= last) {
            int median = first + (last - first) / 2;
            FileScannerResult currentResult = results[median];
            long currentResultStart = currentResult.start();
            if (resultStart == currentResultStart) {
                result = currentResult;
                continue;
            }
            if (resultStart < currentResultStart) {
                last = median - 1;
                continue;
            }
            first = median + 1;
        }
        return result;
    }

    @Override
    public void close() throws IOException {
        this.stop(true);
        this.rootInput.close();
        this.inputDecodeCache.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void queueScanTask(FileScannerRunnableV task) {
        FileScanner fileScanner = this;
        synchronized (fileScanner) {
            ++this.runningScanTasks;
        }
        try {
            this.threadPool.execute(() -> {
                try {
                    if (!this.threadPool.isShutdown()) {
                        task.run();
                    }
                }
                catch (Exception e) {
                    LOG.warning((Throwable)e, "Scan thread failed with exception", new Object[0]);
                }
                finally {
                    this.cleanUpScanTask();
                }
            });
        }
        catch (RejectedExecutionException e) {
            Exceptions.ignore((Throwable)e);
            this.cleanUpScanTask();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void cleanUpScanTask() {
        boolean scanFinished;
        FileScanner fileScanner = this;
        synchronized (fileScanner) {
            --this.runningScanTasks;
            scanFinished = this.runningScanTasks == 0;
        }
        if (scanFinished) {
            this.scanFinished();
            this.threadPool.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scanProgress(long totalInputBytesDelta, long scannedBytesDelta) {
        boolean suppressCallStatus;
        FileScannerProgress reportProgress = null;
        FileScanner fileScanner = this;
        synchronized (fileScanner) {
            this.totalInputBytes += totalInputBytesDelta;
            this.scannedBytes += scannedBytesDelta;
            long currentNanos = System.nanoTime();
            this.scanTimeNanos = currentNanos - this.scanStartedNanos;
            if (totalInputBytesDelta > 0L || this.scannedBytes == this.totalInputBytes || this.scanTimeNanos - this.lastProgressTimeNanos > 700000000L) {
                this.lastProgressTimeNanos = this.scanTimeNanos;
                reportProgress = new FileScannerProgress(this.scanStartedNanos, this.scanTimeNanos, this.scannedBytes, this.totalInputBytes);
            }
            suppressCallStatus = this.suppressStatus;
        }
        if (reportProgress != null && !suppressCallStatus) {
            FileScannerProgress progress = reportProgress;
            this.callStatus(() -> this.status.scanProgress(this, progress));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scanStarted() {
        boolean suppressCallStatus;
        FileScanner fileScanner = this;
        synchronized (fileScanner) {
            this.scanStartedNanos = System.nanoTime();
            this.notifyAll();
            suppressCallStatus = this.suppressStatus;
        }
        if (!suppressCallStatus) {
            this.callStatus(() -> this.status.scanStarted(this));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scanFinished() {
        boolean suppressCallStatus;
        long scanTime;
        FileScanner fileScanner = this;
        synchronized (fileScanner) {
            scanTime = this.scanTimeNanos = System.nanoTime() - this.scanStartedNanos;
            suppressCallStatus = this.suppressStatus;
        }
        LOG.notice("Finished scanning ''{0}'' (scan took: {1} ms)", new Object[]{this.rootResult.name(), scanTime / 1000000L});
        if (!suppressCallStatus) {
            this.callStatus(() -> this.status.scanFinished(this));
        }
        fileScanner = this;
        synchronized (fileScanner) {
            this.notifyAll();
        }
    }

    private void callStatus(Runnable runnable) {
        try {
            runnable.run();
        }
        catch (RuntimeException e) {
            LOG.warning((Throwable)e, "Status callback failed with exception", new Object[0]);
        }
    }
}

