/*
 * Decompiled with CFR 0.152.
 */
package de.captaingoldfish.scim.sdk.client.keys;

import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreCreationFailedException;
import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreEntryException;
import de.captaingoldfish.scim.sdk.client.keys.KeyReader;
import de.captaingoldfish.scim.sdk.client.keys.SecurityProvider;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyStoreSupporter {
    private static final Logger log = LoggerFactory.getLogger(KeyStoreSupporter.class);
    private static final String SUN_PROVIDER = "SUN";
    private static final String SUN_JCE_PROVIDER = "SunJCE";

    public static Provider selectProvider(KeyStoreType keyStoreType) {
        if (keyStoreType.equals((Object)KeyStoreType.PKCS12)) {
            return SecurityProvider.BOUNCY_CASTLE_PROVIDER;
        }
        if (keyStoreType.equals((Object)KeyStoreType.JKS)) {
            return Security.getProvider(SUN_PROVIDER);
        }
        return Security.getProvider(SUN_JCE_PROVIDER);
    }

    public static KeyStore toKeyStore(PrivateKey privateKey, Certificate certificate, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        log.trace("putting private key and certificate into a keystore of type '{}'", (Object)keyStoreType.name());
        if (privateKey == null) {
            throw new KeyStoreEntryException("private key is null and cannot be added into a keystore!");
        }
        if (certificate == null) {
            throw new KeyStoreEntryException("certificate is null and thus the given private key cannot be added to the keystore!");
        }
        KeyStore keyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keystorePassword);
        KeyStoreSupporter.addCertificateEntryToKeyStore(keyStore, certificate, alias);
        log.trace("adding the private key to the keystore with alias '{}'", (Object)alias);
        Certificate[] certificateChain = new Certificate[]{certificate};
        keyStore.setEntry(alias, new KeyStore.PrivateKeyEntry(privateKey, certificateChain), new KeyStore.PasswordProtection(keystorePassword.toCharArray()));
        return keyStore;
    }

    public static KeyStore toKeyStore(Certificate certificate, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        log.trace("putting private key and certificate into a keystore of type '{}'", (Object)keyStoreType.name());
        if (certificate == null) {
            throw new KeyStoreEntryException("certificate is null and thus the given private key cannot be added to the keystore!");
        }
        KeyStore keyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keystorePassword);
        KeyStoreSupporter.addCertificateEntryToKeyStore(keyStore, certificate, alias);
        return keyStore;
    }

    public static KeyStore toKeyStore(byte[] privateKeyBytes, byte[] certificateBytes, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        PrivateKey privateKey = KeyReader.readPrivateRSAKey(privateKeyBytes);
        X509Certificate certificate = KeyReader.readX509Certificate(certificateBytes);
        return KeyStoreSupporter.toKeyStore(privateKey, certificate, alias, keystorePassword, keyStoreType);
    }

    public static byte[] getBytes(KeyStore keyStore, String password) {
        byte[] byArray;
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            keyStore.store(outputStream, password.toCharArray());
            byArray = outputStream.toByteArray();
        }
        return byArray;
    }

    public static KeyStore createEmptyKeyStore(KeyStoreType keyStoreType, String keystorePassword) {
        Provider provider = KeyStoreSupporter.selectProvider(keyStoreType);
        log.trace("creating a {} keystore with '{}' Provider", (Object)keyStoreType, (Object)provider.getName());
        KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), provider);
        keyStore.load(null, keystorePassword == null ? null : keystorePassword.toCharArray());
        return keyStore;
    }

    public static KeyStore addCertificateEntryToKeyStore(KeyStore keyStore, Certificate certificate, String alias) {
        Optional<Certificate> certificateOptional = KeyStoreSupporter.getCertificate(keyStore, alias);
        if (certificateOptional.isPresent()) {
            throw new IllegalArgumentException(String.format("certificate entry for alias '%s' does already exist", alias));
        }
        KeyStoreSupporter.addCertificateEntry(keyStore, alias, certificate);
        return keyStore;
    }

    public static KeyStore addCertificateEntry(KeyStore keyStore, String alias, Certificate certificate) {
        keyStore.setCertificateEntry(alias, certificate);
        log.trace("successfully added certificate entry under alias '{}' to keystore '{}'", (Object)alias, (Object)keyStore);
        return keyStore;
    }

    public static KeyStore addEntryToKeystore(KeyStore keyStore, String alias, Key key, Certificate[] certificateChain, String password) {
        Optional<Key> existingKey = KeyStoreSupporter.getKeyEntry(keyStore, alias, password);
        if (existingKey.isPresent()) {
            throw new IllegalArgumentException(String.format("key entry for alias '%s' does already exist", alias));
        }
        KeyStoreSupporter.addEntryToKeystore(keyStore, alias, key, password, certificateChain);
        return keyStore;
    }

    private static KeyStore addEntryToKeystore(KeyStore keyStore, String alias, Key key, String password, Certificate ... certificateChain) {
        if (key == null) {
            if (certificateChain == null) {
                throw new IllegalArgumentException("missing certificate");
            }
            if (certificateChain.length > 1) {
                throw new IllegalArgumentException("only a single certificate can be added as certificate entry");
            }
            keyStore.setCertificateEntry(alias, certificateChain[0]);
        } else {
            keyStore.setKeyEntry(alias, key, password.toCharArray(), certificateChain);
        }
        log.trace("successfully added key-entry under alias '{}' to keystore '{}'", (Object)alias, (Object)keyStore);
        return keyStore;
    }

    public static KeyStore convertKeyStore(KeyStore keyStore, String keyStorePassword, KeyStoreType keyStoreType) {
        if (keyStore.getType().equals(keyStoreType.name())) {
            log.warn("you tried to convert type '{}' to type '{}', this is unnecessary and the original keystore will be returned.", (Object)keyStore.getType(), (Object)keyStoreType.name());
            return keyStore;
        }
        log.trace("converting '{}'-keystore to '{}'-type", (Object)keyStore.getType(), (Object)keyStoreType.name());
        Enumeration<String> aliases = KeyStoreSupporter.getAliases(keyStore);
        KeyStore newKeyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keyStorePassword);
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            KeyStoreSupporter.tryCopyEntry(keyStore, keyStorePassword, keyStorePassword, keyStoreType, newKeyStore, alias);
        }
        return newKeyStore;
    }

    public static void tryCopyEntry(KeyStore keyStore, String keyStorePassword, String keyPassword, KeyStoreType keyStoreType, KeyStore newKeyStore, String alias) {
        log.trace("adding key-entry of alias '{}' to new keystore of type '{}'", (Object)alias, (Object)keyStoreType.name());
        Optional<Certificate[]> certificateChainOptional = KeyStoreSupporter.getCertificateChain(keyStore, alias);
        Optional<Certificate> certificateOptional = KeyStoreSupporter.getCertificate(keyStore, alias);
        Optional<Key> key = KeyStoreSupporter.getKeyEntry(keyStore, alias, keyPassword);
        if (key.isPresent() && certificateChainOptional.isPresent()) {
            KeyStoreSupporter.addEntryToKeystore(newKeyStore, alias, key.get(), keyStorePassword, certificateChainOptional.get());
        } else if (certificateOptional.isPresent()) {
            Certificate certificate = certificateOptional.get();
            KeyStoreSupporter.addCertificateEntryToKeyStore(newKeyStore, certificate, alias);
        } else {
            log.warn("could not find any entries to copy under the alias '{}'", (Object)alias);
        }
    }

    public static void keyStoreToFile(File file, KeyStore keyStore, String keystorePassword) {
        log.trace("creating file '{}' for keystore of type '{}'.", (Object)file.getAbsolutePath(), (Object)keyStore.getType());
        try (FileOutputStream outputStream = new FileOutputStream(file);){
            keyStore.store(outputStream, keystorePassword.toCharArray());
            log.trace("keystore was successfully saved in file '{}'", (Object)file.getAbsolutePath());
        }
    }

    public static void keyStoreToFile(File directory, String filename, KeyStore keyStore, String keystorePassword) {
        log.trace("creating file '{}/{}.{}' for keystore of type '{}'.", new Object[]{directory.getAbsolutePath(), filename, KeyStoreType.valueOf(keyStore.getType()).getFileExtension(), keyStore.getType()});
        KeyStoreType keyStoreType = KeyStoreType.valueOf(keyStore.getType());
        File keyStoreFile = new File(directory.getAbsolutePath() + File.separator + filename + "." + keyStoreType.getFileExtension());
        KeyStoreSupporter.keyStoreToFile(keyStoreFile, keyStore, keystorePassword);
    }

    public static KeyStore readKeyStore(File file, String keyStorePassword) {
        if (!file.exists()) {
            throw new KeyStoreCreationFailedException("The file '" + file.getAbsolutePath() + "' does not exist!");
        }
        String[] fileParts = file.getName().split("\\.");
        String fileExtension = fileParts[fileParts.length - 1];
        KeyStoreType keyStoreType = KeyStoreType.byFileExtension(fileExtension).orElseThrow(() -> new KeyStoreCreationFailedException("could not determine the type of the keystore. A specific file extension like jks, jceks, p12 or pfx is needed."));
        return KeyStoreSupporter.getKeyStoreFromFile(file, keyStorePassword, keyStoreType);
    }

    private static KeyStore getKeyStoreFromFile(File file, String keyStorePassword, KeyStoreType keyStoreType) {
        KeyStore keyStore;
        try (FileInputStream inputStream = new FileInputStream(file);){
            KeyStore keyStore2 = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore2.load(inputStream, keyStorePassword.toCharArray());
            keyStore = keyStore2;
        }
        return keyStore;
    }

    public static KeyStore readKeyStore(byte[] keyStoreBytes, KeyStoreType keyStoreType, String keyStorePassword) {
        KeyStore keyStore;
        if (keyStoreBytes == null || keyStoreType == null || keyStorePassword == null) {
            throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        }
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(keyStoreBytes);){
            KeyStore keyStore2 = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore2.load(inputStream, keyStorePassword.toCharArray());
            keyStore = keyStore2;
        }
        return keyStore;
    }

    public static KeyStore readKeyStore(InputStream keyStoreStream, KeyStoreType keyStoreType, String keyStorePassword) {
        KeyStore keyStore;
        block9: {
            if (keyStoreStream == null || keyStoreType == null || keyStorePassword == null) {
                throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
            }
            InputStream inputStream = keyStoreStream;
            try {
                KeyStore keyStore2 = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
                keyStore2.load(inputStream, keyStorePassword.toCharArray());
                keyStore = keyStore2;
                if (inputStream == null) break block9;
            }
            catch (Throwable throwable) {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            inputStream.close();
        }
        return keyStore;
    }

    public static KeyStore readTruststore(byte[] truststoreBytes, KeyStoreType keyStoreType) {
        if (truststoreBytes == null) {
            throw new KeyStoreCreationFailedException("Cannot create a truststore if truststoreBytes is null");
        }
        return KeyStoreSupporter.readTruststore(new ByteArrayInputStream(truststoreBytes), keyStoreType, null);
    }

    public static KeyStore readTruststore(byte[] truststoreBytes, KeyStoreType keyStoreType, String password) {
        if (truststoreBytes == null) {
            throw new KeyStoreCreationFailedException("Cannot create a truststore if truststoreBytes is null");
        }
        return KeyStoreSupporter.readTruststore(new ByteArrayInputStream(truststoreBytes), keyStoreType, password);
    }

    public static KeyStore readTruststore(InputStream truststoreStream, KeyStoreType keyStoreType) {
        return KeyStoreSupporter.readTruststore(truststoreStream, keyStoreType, null);
    }

    public static KeyStore readTruststore(InputStream truststoreStream, KeyStoreType keyStoreType, String password) {
        KeyStore keyStore;
        block10: {
            if (truststoreStream == null) {
                throw new KeyStoreCreationFailedException("Cannot create a truststore if truststore is null");
            }
            InputStream inputStream = truststoreStream;
            try {
                KeyStore keyStore2 = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
                String keystorePin = password;
                if (KeyStoreType.PKCS12.equals((Object)keyStoreType) && password == null) {
                    keystorePin = "";
                }
                keyStore2.load(inputStream, keystorePin == null ? null : keystorePin.toCharArray());
                keyStore = keyStore2;
                if (inputStream == null) break block10;
            }
            catch (Throwable throwable) {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            inputStream.close();
        }
        return keyStore;
    }

    public static KeyStore mergeKeyStores(KeyStore keyStore1, String password1, KeyStore keyStore2, String password2, KeyStoreType keyStoreType, String mergedKeyStoreKeyPassword) {
        log.trace("trying to merge the following keystores {}-{} and {}-{}", new Object[]{keyStore1.getType(), keyStore1, keyStore2.getType(), keyStore2});
        KeyStore mergedKeyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, null);
        ArrayList<String> aliasMap = new ArrayList<String>();
        Enumeration<String> aliases1 = KeyStoreSupporter.getAliases(keyStore1);
        log.trace("adding the entries of keystore1 '{}'", (Object)keyStore1);
        while (aliases1.hasMoreElements()) {
            String alias = aliases1.nextElement();
            aliasMap.add(alias);
            KeyStoreSupporter.tryCopyEntry(keyStore1, mergedKeyStoreKeyPassword, password1, keyStoreType, mergedKeyStore, alias);
        }
        Enumeration<String> aliases2 = KeyStoreSupporter.getAliases(keyStore2);
        log.trace("adding the entries of keystore2 '{}'", (Object)keyStore2);
        while (aliases2.hasMoreElements()) {
            String alias = aliases2.nextElement();
            Optional<Key> key = KeyStoreSupporter.getKeyEntry(keyStore2, alias, password2);
            if (key.isPresent()) {
                KeyStoreSupporter.getCertificateChain(keyStore2, alias).ifPresent(certificates -> KeyStoreSupporter.addEntryToKeystore(mergedKeyStore, alias, (Key)key.get(), certificates, password2));
                continue;
            }
            Optional<Certificate> certificate = KeyStoreSupporter.getCertificate(keyStore2, alias);
            certificate.ifPresent(cert -> KeyStoreSupporter.addCertificateEntryToKeyStore(mergedKeyStore, cert, alias));
        }
        return mergedKeyStore;
    }

    public static KeyPair readFirstKeyPairEntryFromKeyStore(KeyStore keyStore, String privateKeyPassword) {
        Enumeration<String> aliases = KeyStoreSupporter.getAliases(keyStore);
        KeyPair keyPair = null;
        if (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            PrivateKey privateKey = (PrivateKey)keyStore.getKey(alias, privateKeyPassword.toCharArray());
            PublicKey publicKey = keyStore.getCertificate(alias).getPublicKey();
            keyPair = new KeyPair(publicKey, privateKey);
        }
        return keyPair;
    }

    public static Enumeration<String> getAliases(KeyStore keyStore) {
        return keyStore.aliases();
    }

    public static Optional<Key> getKeyEntry(KeyStore keyStore, String alias, String password) {
        try {
            return Optional.ofNullable(keyStore.getKey(alias, Optional.ofNullable(password).map(String::toCharArray).orElse(null)));
        }
        catch (UnrecoverableKeyException e) {
            Throwable root = ExceptionUtils.getRootCause((Throwable)e);
            String rootMessage = root == null ? e.getMessage() : root.getMessage();
            log.debug("could not recover key: {}", (Object)rootMessage);
            return Optional.empty();
        }
    }

    public static Optional<Certificate[]> getCertificateChain(KeyStore keyStore, String alias) {
        return Optional.ofNullable(keyStore.getCertificateChain(alias));
    }

    public static Optional<Certificate> getCertificate(KeyStore keyStore, String alias) {
        return Optional.ofNullable(keyStore.getCertificate(alias));
    }

    private KeyStoreSupporter() {
    }

    public static enum KeyStoreType {
        JKS("jks"),
        JCEKS("jceks"),
        PKCS12("p12");

        private String fileExtension;

        private KeyStoreType(String fileExtension) {
            this.fileExtension = fileExtension;
        }

        public static Optional<KeyStoreType> byFileExtension(String fileExtension) {
            if (StringUtils.isBlank((CharSequence)fileExtension)) {
                return Optional.empty();
            }
            for (KeyStoreType keyStoreType : KeyStoreType.values()) {
                if (!fileExtension.toLowerCase(Locale.ENGLISH).endsWith(keyStoreType.getFileExtension())) continue;
                return Optional.of(keyStoreType);
            }
            if (fileExtension.toLowerCase(Locale.ENGLISH).endsWith("pfx")) {
                return Optional.of(PKCS12);
            }
            return Optional.empty();
        }

        public String getFileExtension() {
            return this.fileExtension;
        }
    }
}

