/*
 * Decompiled with CFR 0.152.
 */
package de.captaingoldfish.scim.sdk.client.keys;

import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreConvertException;
import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreCreationFailedException;
import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreEntryException;
import de.captaingoldfish.scim.sdk.client.exceptions.KeyStoreReadingException;
import de.captaingoldfish.scim.sdk.client.keys.KeyReader;
import de.captaingoldfish.scim.sdk.client.keys.SecurityProvider;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Locale;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyStoreSupporter {
    private static final Logger log = LoggerFactory.getLogger(KeyStoreSupporter.class);
    private static final String STRING_BYTE_ARRAY_CANNOT_BE_READ_IN_THE_GIVEN = "this byte array cannot be read in the given ";
    private static final String STRING_NOT_PKCS12 = "stream does not represent a PKCS12 key store";
    private static final String STRING_INVALID_KEYSTORE_FORMAT = "invalid keystore format";
    private static final String MESSAGE_KEYSTORE_READ_ABORTED = "an unexpected error aborted the reading of the keystore.";
    private static final String SUN_PROVIDER = "SUN";
    private static final String SUN_JCE_PROVIDER = "SunJCE";

    private KeyStoreSupporter() {
    }

    public static Provider selectProvider(KeyStoreType keyStoreType) {
        if (keyStoreType.equals((Object)KeyStoreType.PKCS12)) {
            return SecurityProvider.BOUNCY_CASTLE_PROVIDER;
        }
        if (keyStoreType.equals((Object)KeyStoreType.JKS)) {
            return Security.getProvider(SUN_PROVIDER);
        }
        if (keyStoreType.equals((Object)KeyStoreType.JCEKS)) {
            return Security.getProvider(SUN_JCE_PROVIDER);
        }
        throw new IllegalStateException("unsupported keystore-type: " + (Object)((Object)keyStoreType));
    }

    public static KeyStore toKeyStore(PrivateKey privateKey, Certificate certificate, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        if (log.isTraceEnabled()) {
            log.trace("putting private key and certificate into a keystore of type '{}'", (Object)keyStoreType.name());
        }
        if (privateKey == null) {
            throw new KeyStoreEntryException("private key is null and cannot be added into a keystore!");
        }
        if (certificate == null) {
            throw new KeyStoreEntryException("certificate is null and thus the given private key cannot be added to the keystore!");
        }
        KeyStore keyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keystorePassword);
        KeyStoreSupporter.addCertificateEntryToKeyStore(keyStore, certificate, alias);
        try {
            if (log.isTraceEnabled()) {
                log.trace("adding the private key to the keystore with alias '{}'", (Object)alias);
            }
            Certificate[] certificateChain = new Certificate[]{certificate};
            keyStore.setEntry(alias, new KeyStore.PrivateKeyEntry(privateKey, certificateChain), new KeyStore.PasswordProtection(keystorePassword.toCharArray()));
        }
        catch (KeyStoreException e) {
            throw new KeyStoreEntryException("could not add the given private key into the keystore with the given alias '" + alias + "'", e);
        }
        return keyStore;
    }

    public static KeyStore toKeyStore(Certificate certificate, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        if (log.isTraceEnabled()) {
            log.trace("putting private key and certificate into a keystore of type '{}'", (Object)keyStoreType.name());
        }
        if (certificate == null) {
            throw new KeyStoreEntryException("certificate is null and thus the given private key cannot be added to the keystore!");
        }
        KeyStore keyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keystorePassword);
        KeyStoreSupporter.addCertificateEntryToKeyStore(keyStore, certificate, alias);
        return keyStore;
    }

    public static KeyStore toKeyStore(byte[] privateKeyBytes, byte[] certificateBytes, String alias, String keystorePassword, KeyStoreType keyStoreType) {
        PrivateKey privateKey = KeyReader.readPrivateRSAKey(privateKeyBytes);
        X509Certificate certificate = KeyReader.readX509Certificate(certificateBytes);
        return KeyStoreSupporter.toKeyStore(privateKey, certificate, alias, keystorePassword, keyStoreType);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static byte[] getBytes(KeyStore keyStore, String password) {
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            keyStore.store(outputStream, password.toCharArray());
            byte[] byArray = outputStream.toByteArray();
            return byArray;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreConvertException("keystore could not be translated to a byte array", e);
        }
    }

    public static KeyStore createEmptyKeyStore(KeyStoreType keyStoreType, String keystorePassword) {
        KeyStore keyStore;
        try {
            Provider provider = KeyStoreSupporter.selectProvider(keyStoreType);
            if (log.isTraceEnabled()) {
                log.trace("creating a {} keystore with '{}' Provider", (Object)keyStoreType, (Object)provider.getName());
            }
            keyStore = KeyStore.getInstance(keyStoreType.name(), provider);
            keyStore.load(null, keystorePassword == null ? null : keystorePassword.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException("for some unexpected reason the keystore instance could not be created", e);
        }
        return keyStore;
    }

    public static KeyStore addCertificateEntryToKeyStore(KeyStore keyStore, Certificate certificate, String alias) {
        Optional<Certificate> certificateOptional = KeyStoreSupporter.getCertificate(keyStore, alias);
        if (certificateOptional.isPresent()) {
            if (certificateOptional.get().equals(certificate)) {
                if (log.isWarnEnabled()) {
                    log.warn("not adding certificate '{}' with alias '{}' to keystore '{}' because the exact same entry does already exist", new Object[]{alias, keyStore, certificate});
                }
            } else {
                String nonAmbigiousAlias = alias + "_";
                KeyStoreSupporter.addCertificateEntry(keyStore, nonAmbigiousAlias, certificate);
            }
        } else {
            KeyStoreSupporter.addCertificateEntry(keyStore, alias, certificate);
        }
        return keyStore;
    }

    private static KeyStore addCertificateEntry(KeyStore keyStore, String alias, Certificate certificate) {
        try {
            keyStore.setCertificateEntry(alias, certificate);
            log.trace("successfully added certificate entry under alias '{}' to keystore '{}'", (Object)alias, (Object)keyStore);
        }
        catch (KeyStoreException e) {
            throw new KeyStoreCreationFailedException("could not add certificate to keystore '" + keyStore + "' with alias '" + alias + "'", e);
        }
        return keyStore;
    }

    public static KeyStore addKeyEntryToKeystore(KeyStore keyStore, String alias, Key key, Certificate[] certificateChain, String password) {
        Optional<Key> existingKey = KeyStoreSupporter.getKeyEntry(keyStore, alias, password);
        if (existingKey.isPresent()) {
            if (Arrays.equals(existingKey.get().getEncoded(), key.getEncoded())) {
                log.warn("will not add entry '{}' to keystore '{}' because it does already exist", (Object)alias, (Object)keyStore);
            } else {
                String nonAmbigiousAlias = alias + "_";
                KeyStoreSupporter.addKeyEntryToKeystore(keyStore, nonAmbigiousAlias, key, password, certificateChain);
            }
        } else {
            KeyStoreSupporter.addKeyEntryToKeystore(keyStore, alias, key, password, certificateChain);
        }
        return keyStore;
    }

    private static KeyStore addKeyEntryToKeystore(KeyStore keyStore, String alias, Key key, String password, Certificate ... certificateChain) {
        try {
            keyStore.setKeyEntry(alias, key, password.toCharArray(), certificateChain);
            log.trace("successfully added key-entry under alias '{}' to keystore '{}'", (Object)alias, (Object)keyStore);
        }
        catch (KeyStoreException e) {
            throw new KeyStoreCreationFailedException("could not add key entry with alias '" + alias + "' to keystore '" + keyStore + "'", e);
        }
        return keyStore;
    }

    public static KeyStore convertKeyStore(KeyStore keyStore, String keyStorePassword, KeyStoreType keyStoreType) {
        if (keyStore.getType().equals(keyStoreType.name())) {
            if (log.isWarnEnabled()) {
                log.warn("you tried to convert type '{}' to type '{}', this is unnecessary and the original keystore will be returned.", (Object)keyStore.getType(), (Object)keyStoreType.name());
            }
            return keyStore;
        }
        log.trace("converting '{}'-keystore to '{}'-type", (Object)keyStore.getType(), (Object)keyStoreType.name());
        Enumeration<String> aliases = KeyStoreSupporter.getAliases(keyStore);
        KeyStore newKeyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, keyStorePassword);
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            KeyStoreSupporter.tryCopyKeyEntry(keyStore, keyStorePassword, keyStorePassword, keyStoreType, newKeyStore, alias);
        }
        return newKeyStore;
    }

    public static void tryCopyKeyEntry(KeyStore keyStore, String keyStorePassword, String keyPassword, KeyStoreType keyStoreType, KeyStore newKeyStore, String alias) {
        log.trace("adding key-entry of alias '{}' to new keystore of type '{}'", (Object)alias, (Object)keyStoreType.name());
        Optional<Certificate[]> certificateChainOptional = KeyStoreSupporter.getCertificateChain(keyStore, alias);
        Optional<Certificate> certificateOptional = KeyStoreSupporter.getCertificate(keyStore, alias);
        Optional<Key> key = KeyStoreSupporter.getKeyEntry(keyStore, alias, keyPassword);
        if (key.isPresent() && certificateChainOptional.isPresent()) {
            KeyStoreSupporter.addKeyEntryToKeystore(newKeyStore, alias, key.get(), keyStorePassword, certificateChainOptional.get());
        } else if (certificateChainOptional.isPresent()) {
            Certificate[] certificateChain = certificateChainOptional.get();
            KeyStoreSupporter.addCertificateEntryToKeyStore(newKeyStore, certificateChain[certificateChain.length - 1], alias);
        } else if (certificateOptional.isPresent()) {
            Certificate certificate = certificateOptional.get();
            KeyStoreSupporter.addCertificateEntryToKeyStore(newKeyStore, certificate, alias);
        } else {
            log.warn("could not find any entries to copy under the alias '{}'", (Object)alias);
        }
    }

    public static void keyStoreToFile(File file, KeyStore keyStore, String keystorePassword) {
        if (log.isTraceEnabled()) {
            log.trace("creating file '{}' for keystore of type '{}'.", (Object)file.getAbsolutePath(), (Object)keyStore.getType());
        }
        if (file == null || keyStore == null) {
            throw new KeyStoreCreationFailedException("if the given keystore or file is null the file cannot be created.");
        }
        if (file.exists() && file.isDirectory()) {
            throw new KeyStoreCreationFailedException("given file '" + file + "' is a directory. Keystore cannot be saved.");
        }
        File parentFile = file.getParentFile();
        if (!(file.exists() || parentFile != null && parentFile.exists())) {
            throw new KeyStoreCreationFailedException("The target directory '" + file.getAbsolutePath() + "' does not exist. KeyStore cannot be saved.");
        }
        try (FileOutputStream outputStream = new FileOutputStream(file);){
            keyStore.store(outputStream, keystorePassword.toCharArray());
            if (log.isTraceEnabled()) {
                log.trace("keystore was successfully saved in file '{}'", (Object)file.getAbsolutePath());
            }
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException("could not create keystore file for some unexpected reason.", e);
        }
    }

    public static void keyStoreToFile(File directory, String filename, KeyStore keyStore, String keystorePassword) {
        if (log.isTraceEnabled()) {
            log.trace("creating file '{}/{}.{}' for keystore of type '{}'.", new Object[]{directory.getAbsolutePath(), filename, KeyStoreType.valueOf(keyStore.getType()).getFileExtension(), keyStore.getType()});
        }
        KeyStoreType keyStoreType = KeyStoreType.valueOf(keyStore.getType());
        File keyStoreFile = new File(directory.getAbsolutePath() + File.separator + filename + "." + keyStoreType.getFileExtension());
        KeyStoreSupporter.keyStoreToFile(keyStoreFile, keyStore, keystorePassword);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore readKeyStore(File file, String keyStorePassword) {
        if (!file.exists()) {
            throw new KeyStoreCreationFailedException("The file '" + file.getAbsolutePath() + "' does not exist!");
        }
        String[] fileParts = file.getName().split("\\.");
        String fileExtension = fileParts[fileParts.length - 1];
        KeyStoreType keyStoreType = KeyStoreType.byFileExtension(fileExtension).orElseThrow(() -> new KeyStoreCreationFailedException("could not determine the type of the keystore. A specific file extension like jks, jceks, p12 or pfx is needed."));
        try (FileInputStream inputStream = new FileInputStream(file);){
            KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore.load(inputStream, keyStorePassword.toCharArray());
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore readKeyStore(File file, KeyStoreType keyStoreType, String keyStorePassword) {
        if (!file.exists()) {
            throw new KeyStoreCreationFailedException("The file '" + file.getAbsolutePath() + "' does not exist!");
        }
        try (FileInputStream inputStream = new FileInputStream(file);){
            KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore.load(inputStream, keyStorePassword.toCharArray());
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore readKeyStore(byte[] keyStoreBytes, KeyStoreType keyStoreType, String keyStorePassword) {
        if (keyStoreBytes == null) throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        if (keyStoreType == null) throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        if (keyStorePassword == null) {
            throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        }
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(keyStoreBytes);){
            KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore.load(inputStream, keyStorePassword.toCharArray());
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
        }
        catch (IOException e) {
            if (e.getMessage() == null) throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
            if (e.getMessage().toLowerCase(Locale.ENGLISH).contains(STRING_INVALID_KEYSTORE_FORMAT)) throw new KeyStoreCreationFailedException("this byte array cannot be read in the given keystore-format '" + keyStoreType.name() + "'.", e);
            if (!e.getMessage().contains(STRING_NOT_PKCS12)) throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
            throw new KeyStoreCreationFailedException("this byte array cannot be read in the given keystore-format '" + keyStoreType.name() + "'.", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore readKeyStore(InputStream keyStoreStream, KeyStoreType keyStoreType, String keyStorePassword) {
        if (keyStoreStream == null) throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        if (keyStoreType == null) throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        if (keyStorePassword == null) {
            throw new KeyStoreCreationFailedException("Cannot create a keystore if null values are given...");
        }
        try (InputStream inputStream = keyStoreStream;){
            KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            keyStore.load(inputStream, keyStorePassword.toCharArray());
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
        }
        catch (IOException e) {
            if (e.getMessage() == null) throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
            if (e.getMessage().toLowerCase(Locale.ENGLISH).contains(STRING_INVALID_KEYSTORE_FORMAT)) throw new KeyStoreCreationFailedException("this byte array cannot be read in the given keystore-format '" + keyStoreType.name() + "'.", e);
            if (!e.getMessage().contains(STRING_NOT_PKCS12)) throw new KeyStoreCreationFailedException(MESSAGE_KEYSTORE_READ_ABORTED, e);
            throw new KeyStoreCreationFailedException("this byte array cannot be read in the given keystore-format '" + keyStoreType.name() + "'.", e);
        }
    }

    public static KeyStore readTruststore(byte[] truststoreBytes, KeyStoreType keyStoreType) {
        if (truststoreBytes == null) {
            throw new KeyStoreCreationFailedException("Cannot create a truststore if truststoreBytes is null");
        }
        return KeyStoreSupporter.readTruststore(new ByteArrayInputStream(truststoreBytes), keyStoreType, null);
    }

    public static KeyStore readTruststore(byte[] truststoreBytes, KeyStoreType keyStoreType, String password) {
        if (truststoreBytes == null) {
            throw new KeyStoreCreationFailedException("Cannot create a truststore if truststoreBytes is null");
        }
        return KeyStoreSupporter.readTruststore(new ByteArrayInputStream(truststoreBytes), keyStoreType, password);
    }

    public static KeyStore readTruststore(InputStream truststoreStream, KeyStoreType keyStoreType) {
        return KeyStoreSupporter.readTruststore(truststoreStream, keyStoreType, null);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore readTruststore(InputStream truststoreStream, KeyStoreType keyStoreType, String password) {
        if (truststoreStream == null) {
            throw new KeyStoreCreationFailedException("Cannot create a truststore if truststore is null");
        }
        try (InputStream inputStream = truststoreStream;){
            KeyStore keyStore = KeyStore.getInstance(keyStoreType.name(), KeyStoreSupporter.selectProvider(keyStoreType));
            String keystorePin = password;
            if (KeyStoreType.PKCS12.equals((Object)keyStoreType) && password == null) {
                keystorePin = "";
            }
            keyStore.load(inputStream, keystorePin == null ? null : keystorePin.toCharArray());
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new KeyStoreCreationFailedException("an unexpected error aborted the reading of the truststore.", e);
        }
        catch (IOException e) {
            if (e.getMessage() == null) throw new KeyStoreCreationFailedException("an unexpected error aborted the reading of the truststore.", e);
            if (e.getMessage().toLowerCase(Locale.ENGLISH).contains(STRING_INVALID_KEYSTORE_FORMAT)) throw new KeyStoreCreationFailedException("this byte array cannot be read in the given truststore-format '" + keyStoreType.name() + "'.", e);
            if (!e.getMessage().contains(STRING_NOT_PKCS12)) throw new KeyStoreCreationFailedException("an unexpected error aborted the reading of the truststore.", e);
            throw new KeyStoreCreationFailedException("this byte array cannot be read in the given truststore-format '" + keyStoreType.name() + "'.", e);
        }
    }

    public static KeyStore mergeKeyStores(KeyStore keyStore1, String password1, KeyStore keyStore2, String password2, KeyStoreType keyStoreType, String mergedKeyStoreKeyPassword) {
        log.trace("trying to merge the following keystores {}-{} and {}-{}", new Object[]{keyStore1.getType(), keyStore1, keyStore2.getType(), keyStore2});
        KeyStore mergedKeyStore = KeyStoreSupporter.createEmptyKeyStore(keyStoreType, null);
        ArrayList<String> aliasMap = new ArrayList<String>();
        Enumeration<String> aliases1 = KeyStoreSupporter.getAliases(keyStore1);
        log.trace("adding the entries of keystore1 '{}'", (Object)keyStore1);
        while (aliases1.hasMoreElements()) {
            String alias = aliases1.nextElement();
            aliasMap.add(alias);
            KeyStoreSupporter.tryCopyKeyEntry(keyStore1, mergedKeyStoreKeyPassword, password1, keyStoreType, mergedKeyStore, alias);
        }
        Enumeration<String> aliases2 = KeyStoreSupporter.getAliases(keyStore2);
        log.trace("adding the entries of keystore2 '{}'", (Object)keyStore2);
        while (aliases2.hasMoreElements()) {
            String alias = aliases2.nextElement();
            Optional<Key> key = KeyStoreSupporter.getKeyEntry(keyStore2, alias, password2);
            if (key.isPresent()) {
                KeyStoreSupporter.getCertificateChain(keyStore2, alias).ifPresent(certificates -> KeyStoreSupporter.addKeyEntryToKeystore(mergedKeyStore, alias, (Key)key.get(), certificates, password2));
                continue;
            }
            Optional<Certificate> certificate = KeyStoreSupporter.getCertificate(keyStore2, alias);
            if (certificate.isPresent()) {
                KeyStoreSupporter.addCertificateEntryToKeyStore(mergedKeyStore, certificate.get(), alias);
                continue;
            }
            log.warn("empty keystore entry found... keystore seems to be corrupted");
        }
        return mergedKeyStore;
    }

    public static KeyPair readFirstKeyPairEntryFromKeyStore(KeyStore keyStore, String privateKeyPassword) {
        Enumeration<String> aliases = KeyStoreSupporter.getAliases(keyStore);
        KeyPair keyPair = null;
        if (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            try {
                PrivateKey privateKey = (PrivateKey)keyStore.getKey(alias, privateKeyPassword.toCharArray());
                PublicKey publicKey = keyStore.getCertificate(alias).getPublicKey();
                keyPair = new KeyPair(publicKey, privateKey);
            }
            catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
                throw new KeyStoreReadingException("could not extract private key from keystore with alias '" + alias + "'", e);
            }
        }
        return keyPair;
    }

    public static Enumeration<String> getAliases(KeyStore keyStore) {
        try {
            return keyStore.aliases();
        }
        catch (KeyStoreException e) {
            throw new KeyStoreReadingException("could not access aliases of keystore...", e);
        }
    }

    private static Optional<Key> getKeyEntry(KeyStore keyStore, String alias, String password) {
        try {
            return Optional.ofNullable(keyStore.getKey(alias, password.toCharArray()));
        }
        catch (KeyStoreException | NoSuchAlgorithmException e) {
            throw new KeyStoreReadingException("could not extract key-entry from the given keystore of alias '" + alias + "'", e);
        }
        catch (UnrecoverableKeyException e) {
            Throwable root = ExceptionUtils.getRootCause((Throwable)e);
            String rootMessage = root == null ? e.getMessage() : root.getMessage();
            log.debug("could not recover key: {}", (Object)rootMessage);
            return Optional.empty();
        }
    }

    public static Optional<Certificate[]> getCertificateChain(KeyStore keyStore, String alias) {
        try {
            return Optional.ofNullable(keyStore.getCertificateChain(alias));
        }
        catch (KeyStoreException e) {
            throw new KeyStoreReadingException("cannot read certificate chain of keystore '" + keyStore + "' for alias '" + alias + "'", e);
        }
    }

    public static Optional<Certificate> getCertificate(KeyStore keyStore, String alias) {
        try {
            return Optional.ofNullable(keyStore.getCertificate(alias));
        }
        catch (KeyStoreException e) {
            throw new KeyStoreReadingException("cannot read certificate of keystore '" + keyStore + "' for alias '" + alias + "'", e);
        }
    }

    public static enum KeyStoreType {
        JKS("jks"),
        JCEKS("jceks"),
        PKCS12("p12");

        private String fileExtension;

        private KeyStoreType(String fileExtension) {
            this.fileExtension = fileExtension;
        }

        public static Optional<KeyStoreType> byFileExtension(String fileExtension) {
            if (fileExtension == null || StringUtils.isBlank((CharSequence)fileExtension)) {
                return Optional.empty();
            }
            for (KeyStoreType keyStoreType : KeyStoreType.values()) {
                if (!fileExtension.toLowerCase(Locale.ENGLISH).endsWith(keyStoreType.getFileExtension())) continue;
                return Optional.of(keyStoreType);
            }
            if (fileExtension.toLowerCase(Locale.ENGLISH).endsWith("pfx")) {
                return Optional.of(PKCS12);
            }
            return Optional.empty();
        }

        public String getFileExtension() {
            return this.fileExtension;
        }
    }
}

