/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import com.mongodb.DBRef;
import com.mongodb.MongoCommandException;
import com.mongodb.MongoException;
import com.mongodb.MongoNamespace;
import com.mongodb.MongoQueryException;
import com.mongodb.MongoWriteException;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import com.mongodb.async.SingleResultCallback;
import com.mongodb.async.client.MongoClient;
import com.mongodb.async.client.MongoClients;
import com.mongodb.bulk.BulkWriteResult;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.BulkWriteOptions;
import com.mongodb.client.model.CountOptions;
import com.mongodb.client.model.CreateCollectionOptions;
import com.mongodb.client.model.DeleteManyModel;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.FindOneAndUpdateOptions;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.InsertOneModel;
import com.mongodb.client.model.RenameCollectionOptions;
import com.mongodb.client.model.ReplaceOptions;
import com.mongodb.client.model.ReturnDocument;
import com.mongodb.client.model.UpdateManyModel;
import com.mongodb.client.model.UpdateOneModel;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import de.bwaldvogel.mongo.MongoBackend;
import de.bwaldvogel.mongo.MongoServer;
import de.bwaldvogel.mongo.backend.TestUtils;
import java.net.InetSocketAddress;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.MapAssert;
import org.bson.BsonObjectId;
import org.bson.BsonTimestamp;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractBackendTest {
    private static final Logger log = LoggerFactory.getLogger(AbstractBackendTest.class);
    protected static final String TEST_DATABASE_NAME = "testdb";
    protected static final String OTHER_TEST_DATABASE_NAME = "bar";
    private MongoServer mongoServer;
    protected com.mongodb.MongoClient syncClient;
    private MongoClient asyncClient;
    protected MongoDatabase db;
    protected MongoCollection<Document> collection;
    private com.mongodb.async.client.MongoCollection<Document> asyncCollection;

    private Document runCommand(String commandName) {
        return this.runCommand(new Document(commandName, (Object)1));
    }

    private Document runCommand(Document command) {
        return this.getAdminDb().runCommand((Bson)command);
    }

    protected MongoCollection<Document> getCollection(String collectionName) {
        return this.db.getCollection(collectionName);
    }

    protected MongoDatabase getAdminDb() {
        return this.syncClient.getDatabase("admin");
    }

    protected abstract MongoBackend createBackend() throws Exception;

    @Before
    public void setUp() throws Exception {
        this.spinUpServer();
    }

    @After
    public void tearDown() {
        this.shutdownServer();
    }

    protected void spinUpServer() throws Exception {
        MongoBackend backend = this.createBackend();
        this.mongoServer = new MongoServer(backend);
        InetSocketAddress serverAddress = this.mongoServer.bind();
        this.syncClient = new com.mongodb.MongoClient(new ServerAddress(serverAddress));
        this.asyncClient = MongoClients.create((String)("mongodb://" + serverAddress.getHostName() + ":" + serverAddress.getPort()));
        this.db = this.syncClient.getDatabase(TEST_DATABASE_NAME);
        this.collection = this.db.getCollection("testcoll");
        MongoNamespace namespace = this.collection.getNamespace();
        com.mongodb.async.client.MongoDatabase asyncDb = this.asyncClient.getDatabase(namespace.getDatabaseName());
        this.asyncCollection = asyncDb.getCollection(namespace.getCollectionName());
    }

    protected void shutdownServer() {
        this.syncClient.close();
        this.asyncClient.close();
        this.mongoServer.shutdownNow();
    }

    @Test
    public void testSimpleInsert() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
    }

    @Test
    public void testSimpleInsertDelete() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.deleteOne((Bson)TestUtils.json("_id: 1"));
    }

    @Test
    public void testCreateCollection() throws Exception {
        String newCollectionName = "some-collection";
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).doesNotContain((Object[])new String[]{newCollectionName});
        this.db.createCollection(newCollectionName, new CreateCollectionOptions());
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).contains((Object[])new String[]{newCollectionName});
    }

    @Test
    public void testCreateCollectionAlreadyExists() throws Exception {
        this.db.createCollection("some-collection", new CreateCollectionOptions());
        try {
            this.db.createCollection("some-collection", new CreateCollectionOptions());
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(48);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"collection already exists"});
        }
    }

    @Test
    public void testUnsupportedModifier() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$foo: {}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10147);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Invalid modifier specified: $foo"});
        }
    }

    @Test
    public void testUpsertWithInc() {
        Document query = TestUtils.json("_id:{ f: 'ca', '1': { l: 2 }, t: { t: 11 } }");
        Document update = TestUtils.json("'$inc': { 'n.!' : 1 , 'n.a.b:false' : 1}");
        this.collection.updateOne((Bson)query, (Bson)update, new UpdateOptions().upsert(true));
        query.putAll((Map)TestUtils.json("n: {'!': 1, a: {'b:false': 1}}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)query);
    }

    @Test
    public void testBasicUpdate() {
        this.collection.insertOne((Object)TestUtils.json("_id:1"));
        this.collection.insertOne((Object)TestUtils.json("_id:2, b:5"));
        this.collection.insertOne((Object)TestUtils.json("_id:3"));
        this.collection.insertOne((Object)TestUtils.json("_id:4"));
        this.collection.replaceOne((Bson)TestUtils.json("_id:2"), (Object)TestUtils.json("_id:2, a:5"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id:2")).first())).isEqualTo((Object)TestUtils.json("_id:2, a:5"));
    }

    @Test
    public void testCollectionStats() throws Exception {
        try {
            this.getCollStats();
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(26);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"No such collection"});
        }
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.collection.insertOne((Object)TestUtils.json("abc: 'foo'"));
        Document stats = this.getCollStats();
        Assertions.assertThat((Integer)stats.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((long)stats.getInteger((Object)"count").longValue()).isEqualTo(2L);
        Assertions.assertThat((long)stats.getLong((Object)"size")).isEqualTo(57L);
        Assertions.assertThat((double)stats.getDouble((Object)"avgObjSize")).isEqualTo(28.5);
    }

    private Document getCollStats() {
        String collectionName = this.collection.getNamespace().getCollectionName();
        return TestUtils.getCollectionStatistics(this.db, collectionName);
    }

    @Test
    public void testGetLogStartupWarnings() throws Exception {
        Document startupWarnings = this.getAdminDb().runCommand((Bson)TestUtils.json("getLog: 'startupWarnings'"));
        Assertions.assertThat((Integer)startupWarnings.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((Object)startupWarnings.get((Object)"totalLinesWritten")).isEqualTo((Object)0);
        Assertions.assertThat((Object)startupWarnings.get((Object)"log")).isEqualTo(Collections.emptyList());
    }

    @Test
    public void testGetLogWhichDoesNotExist() throws Exception {
        try {
            this.getAdminDb().runCommand((Bson)TestUtils.json("getLog: 'illegal'"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"no RamLog"});
        }
    }

    @Test
    public void testCompoundDateIdUpserts() {
        Document query = TestUtils.json("{ _id : { $lt : { n: 'a' , t: 10} , $gte: { n: 'a', t: 1}}}");
        List<Document> toUpsert = Arrays.asList(TestUtils.json("_id: {n:'a', t: 1}"), TestUtils.json("_id: {n:'a', t: 2}"), TestUtils.json("_id: {n:'a', t: 3}"), TestUtils.json("_id: {n:'a', t: 11}"));
        for (Document dbo : toUpsert) {
            this.collection.replaceOne((Bson)dbo, (Object)new Document((Map)dbo).append("foo", (Object)OTHER_TEST_DATABASE_NAME), new ReplaceOptions().upsert(true));
        }
        List results = TestUtils.toArray(this.collection.find((Bson)query));
        Assertions.assertThat(results).containsOnly((Object[])new Document[]{TestUtils.json("_id: {n:'a', t:1}, foo:'bar'"), TestUtils.json("_id: {n:'a', t:2}, foo:'bar'"), TestUtils.json("_id: {n:'a', t:3}, foo:'bar'")});
    }

    @Test
    public void testCompoundSort() {
        this.collection.insertOne((Object)TestUtils.json("a:1, _id:1"));
        this.collection.insertOne((Object)TestUtils.json("a:2, _id:5"));
        this.collection.insertOne((Object)TestUtils.json("a:1, _id:2"));
        this.collection.insertOne((Object)TestUtils.json("a:2, _id:4"));
        this.collection.insertOne((Object)TestUtils.json("a:1, _id:3"));
        List documents = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("a:1, _id:-1")));
        Assertions.assertThat(documents).containsExactly((Object[])new Document[]{TestUtils.json("a:1, _id:3"), TestUtils.json("a:1, _id:2"), TestUtils.json("a:1, _id:1"), TestUtils.json("a:2, _id:5"), TestUtils.json("a:2, _id:4")});
    }

    @Test
    public void testCountCommand() {
        Assertions.assertThat((long)this.collection.count()).isZero();
    }

    @Test
    public void testCountCommandWithQuery() {
        this.collection.insertOne((Object)TestUtils.json("n:1"));
        this.collection.insertOne((Object)TestUtils.json("n:2"));
        this.collection.insertOne((Object)TestUtils.json("n:2"));
        Assertions.assertThat((long)this.collection.count((Bson)TestUtils.json("n:2"))).isEqualTo(2L);
    }

    @Test
    public void testCountDocuments() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testCountDocumentsWithQuery() {
        this.collection.insertOne((Object)TestUtils.json("n:1"));
        this.collection.insertOne((Object)TestUtils.json("n:2"));
        this.collection.insertOne((Object)TestUtils.json("n:2"));
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("n:2"))).isEqualTo(2L);
    }

    @Test
    public void testAggregateWithEmptyPipeline() throws Exception {
        Assertions.assertThat(TestUtils.toArray(this.collection.aggregate(Collections.emptyList()))).isEmpty();
        this.collection.insertOne((Object)TestUtils.json("_id:1"));
        this.collection.insertOne((Object)TestUtils.json("_id:2"));
        Assertions.assertThat(TestUtils.toArray(this.collection.aggregate(Collections.emptyList()))).containsExactly((Object[])new Document[]{TestUtils.json("_id:1"), TestUtils.json("_id:2")});
    }

    @Test
    public void testCreateIndexes() {
        this.collection.createIndex((Bson)new Document("n", (Object)1));
        this.collection.createIndex((Bson)new Document("b", (Object)1));
        List indexes = TestUtils.toArray(this.getCollection("system.indexes").find());
        Assertions.assertThat(indexes).containsOnly((Object[])new Document[]{TestUtils.json("key:{_id:1}").append("ns", (Object)this.collection.getNamespace().getFullName()).append("name", (Object)"_id_"), TestUtils.json("key:{n:1}").append("ns", (Object)this.collection.getNamespace().getFullName()).append("name", (Object)"n_1"), TestUtils.json("key:{b:1}").append("ns", (Object)this.collection.getNamespace().getFullName()).append("name", (Object)"b_1")});
    }

    @Test
    public void testCurrentOperations() throws Exception {
        Document currentOperations = (Document)this.getAdminDb().getCollection("$cmd.sys.inprog").find().first();
        Assertions.assertThat((Map)currentOperations).isNotNull();
        Assertions.assertThat((Object)currentOperations.get((Object)"inprog")).isInstanceOf(List.class);
    }

    @Test
    public void testListCollectionsEmpty() throws Exception {
        Document result = this.db.runCommand((Bson)TestUtils.json("listCollections: 1"));
        Assertions.assertThat((Integer)result.getInteger((Object)"ok")).isEqualTo(1);
        Document cursor = (Document)result.get((Object)"cursor");
        Assertions.assertThat((Iterable)cursor.keySet()).containsOnly((Object[])new String[]{"id", "ns", "firstBatch"});
        Assertions.assertThat((Object)cursor.get((Object)"id")).isEqualTo((Object)0L);
        Assertions.assertThat((Object)cursor.get((Object)"ns")).isEqualTo((Object)(this.db.getName() + ".$cmd.listCollections"));
        List firstBatch = (List)cursor.get((Object)"firstBatch");
        Assertions.assertThat((List)firstBatch).isEmpty();
    }

    @Test
    public void testListCollections() throws Exception {
        List<String> collections = Arrays.asList("coll1", "coll2", "coll3");
        for (String collection : collections) {
            this.getCollection(collection).insertOne((Object)TestUtils.json("_id: 1"));
        }
        Document result = this.db.runCommand((Bson)TestUtils.json("listCollections: 1"));
        Assertions.assertThat((Integer)result.getInteger((Object)"ok")).isEqualTo(1);
        Document cursor = (Document)result.get((Object)"cursor");
        Assertions.assertThat((Iterable)cursor.keySet()).containsOnly((Object[])new String[]{"id", "ns", "firstBatch"});
        Assertions.assertThat((Object)cursor.get((Object)"id")).isEqualTo((Object)0L);
        Assertions.assertThat((Object)cursor.get((Object)"ns")).isEqualTo((Object)(this.db.getName() + ".$cmd.listCollections"));
        Assertions.assertThat((Object)cursor.get((Object)"firstBatch")).isInstanceOf(List.class);
        List firstBatch = (List)cursor.get((Object)"firstBatch");
        HashSet<String> expectedCollections = new HashSet<String>();
        expectedCollections.addAll(collections);
        expectedCollections.add("system.indexes");
        Assertions.assertThat((List)firstBatch).hasSize(expectedCollections.size());
        HashSet<String> collectionNames = new HashSet<String>();
        for (Document collection : firstBatch) {
            Assertions.assertThat((Iterable)collection.keySet()).containsOnly((Object[])new String[]{"name", "options"});
            Assertions.assertThat((Object)collection.get((Object)"options")).isEqualTo((Object)TestUtils.json("{}"));
            Assertions.assertThat((Object)collection.get((Object)"name")).isInstanceOf(String.class);
            collectionNames.add((String)collection.get((Object)"name"));
        }
        Assertions.assertThat(collectionNames).isEqualTo(expectedCollections);
    }

    @Test
    public void testGetCollectionNames() throws Exception {
        this.getCollection("foo").insertOne((Object)TestUtils.json("{}"));
        this.getCollection(OTHER_TEST_DATABASE_NAME).insertOne((Object)TestUtils.json("{}"));
        List collectionNames = TestUtils.toArray(this.db.listCollectionNames());
        Assertions.assertThat(collectionNames).containsOnly((Object[])new String[]{"system.indexes", "foo", OTHER_TEST_DATABASE_NAME});
    }

    @Test
    public void testSystemIndexes() throws Exception {
        this.getCollection("foo").insertOne((Object)TestUtils.json("{}"));
        this.getCollection(OTHER_TEST_DATABASE_NAME).insertOne((Object)TestUtils.json("{}"));
        MongoCollection systemIndexes = this.db.getCollection("system.indexes");
        Assertions.assertThat(TestUtils.toArray(systemIndexes.find())).containsOnly((Object[])new Document[]{TestUtils.json("name: '_id_', ns: 'testdb.foo', key: {_id: 1}"), TestUtils.json("name: '_id_', ns: 'testdb.bar', key: {_id: 1}")});
    }

    @Test
    public void testSystemNamespaces() throws Exception {
        this.getCollection("foo").insertOne((Object)TestUtils.json("{}"));
        this.getCollection(OTHER_TEST_DATABASE_NAME).insertOne((Object)TestUtils.json("{}"));
        MongoCollection namespaces = this.db.getCollection("system.namespaces");
        Assertions.assertThat(TestUtils.toArray(namespaces.find())).containsOnly((Object[])new Document[]{TestUtils.json("name: 'testdb.system.indexes'"), TestUtils.json("name: 'testdb.foo'"), TestUtils.json("name: 'testdb.bar'")});
    }

    @Test
    public void testDatabaseStats() throws Exception {
        Document stats = this.db.runCommand((Bson)new Document("dbStats", (Object)1).append("scale", (Object)1));
        Assertions.assertThat((Integer)stats.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((Long)stats.getLong((Object)"objects")).isZero();
        Assertions.assertThat((Integer)stats.getInteger((Object)"collections")).isZero();
        Assertions.assertThat((Integer)stats.getInteger((Object)"indexes")).isZero();
        Assertions.assertThat((Long)stats.getLong((Object)"dataSize")).isZero();
        this.getCollection("foo").insertOne((Object)TestUtils.json("{}"));
        this.getCollection("foo").insertOne((Object)TestUtils.json("{}"));
        this.getCollection(OTHER_TEST_DATABASE_NAME).insertOne((Object)TestUtils.json("{}"));
        stats = this.db.runCommand((Bson)new Document("dbStats", (Object)1).append("scale", (Object)1));
        Assertions.assertThat((Integer)stats.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((Long)stats.getLong((Object)"objects")).isEqualTo(8L);
        Assertions.assertThat((Integer)stats.getInteger((Object)"collections")).isEqualTo(3);
        Assertions.assertThat((Integer)stats.getInteger((Object)"indexes")).isEqualTo(2);
        Assertions.assertThat((Long)stats.getLong((Object)"dataSize")).isEqualTo(271L);
    }

    @Test
    public void testDeleteDecrementsCount() {
        this.collection.insertOne((Object)TestUtils.json("key: 'value'"));
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
        this.collection.deleteOne((Bson)TestUtils.json("{}"));
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testDeleteInSystemNamespace() throws Exception {
        try {
            this.getCollection("system.foobar").deleteOne((Bson)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(12050);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot delete from system namespace"});
        }
        try {
            this.getCollection("system.namespaces").deleteOne((Bson)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(12050);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot delete from system namespace"});
        }
    }

    @Test
    public void testUpdateInSystemNamespace() throws Exception {
        for (String collectionName : Arrays.asList("system.foobar", "system.namespaces")) {
            MongoCollection<Document> collection = this.getCollection(collectionName);
            try {
                collection.updateMany(Filters.eq((String)"some", (Object)"value"), Updates.set((String)"field", (Object)"value"));
                Assert.fail((String)"MongoException expected");
            }
            catch (MongoException e) {
                Assertions.assertThat((int)e.getCode()).isEqualTo(10156);
                Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot update system collection"});
            }
        }
    }

    @Test
    public void testDistinctQuery() {
        this.collection.insertOne((Object)new Document("n", (Object)3));
        this.collection.insertOne((Object)new Document("n", (Object)1));
        this.collection.insertOne((Object)new Document("n", (Object)2));
        this.collection.insertOne((Object)new Document("n", (Object)1));
        this.collection.insertOne((Object)new Document("n", (Object)1));
        Assertions.assertThat(TestUtils.toArray(this.collection.distinct("n", Integer.class))).containsExactly((Object[])new Integer[]{1, 2, 3});
        Assertions.assertThat(TestUtils.toArray(this.collection.distinct("n", (Bson)TestUtils.json("n: {$gt: 1}"), Integer.class))).containsExactly((Object[])new Integer[]{2, 3});
        Assertions.assertThat((Iterable)this.collection.distinct("foobar", String.class)).isEmpty();
        Assertions.assertThat((Iterable)this.collection.distinct("_id", ObjectId.class)).hasSize((int)this.collection.countDocuments());
    }

    @Test
    public void testDropCollection() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).contains((Object[])new String[]{this.collection.getNamespace().getCollectionName()});
        this.collection.drop();
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).doesNotContain((Object[])new String[]{this.collection.getNamespace().getCollectionName()});
    }

    @Test
    public void testDropCollectionAlsoDropsFromDB() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.collection.drop();
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).doesNotContain((Object[])new String[]{this.collection.getNamespace().getCollectionName()});
    }

    @Test
    public void testDropDatabaseAlsoDropsCollectionData() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.db.drop();
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testDropDatabaseDropsAllData() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        MongoCollection<Document> collection2 = this.getCollection("testcoll2");
        collection2.insertOne((Object)TestUtils.json("{}"));
        this.syncClient.dropDatabase(this.db.getName());
        Assertions.assertThat(this.listDatabaseNames()).doesNotContain((Object[])new String[]{this.db.getName()});
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).doesNotContain((Object[])new String[]{this.collection.getNamespace().getCollectionName(), collection2.getNamespace().getCollectionName()});
    }

    @Test
    public void testEmbeddedSort() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4, counts:{done:1}"));
        this.collection.insertOne((Object)TestUtils.json("_id: 5, counts:{done:2}"));
        List objs = TestUtils.toArray(this.collection.find(Filters.ne((String)"c", (Object)true)).sort((Bson)TestUtils.json("\"counts.done\": -1, _id: 1")));
        Assertions.assertThat(objs).containsExactly((Object[])new Document[]{TestUtils.json("_id: 5, counts:{done:2}"), TestUtils.json("_id: 4, counts:{done:1}"), TestUtils.json("_id: 1"), TestUtils.json("_id: 2"), TestUtils.json("_id: 3")});
    }

    @Test
    public void testFindAndModifyCommandEmpty() throws Exception {
        Document cmd = new Document("findandmodify", (Object)this.collection.getNamespace().getCollectionName());
        try {
            this.db.runCommand((Bson)cmd);
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"need remove or update"});
        }
    }

    @Test
    public void testFindAndModifyCommandIllegalOp() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Document cmd = new Document("findAndModify", (Object)this.collection.getNamespace().getCollectionName());
        cmd.put("query", (Object)TestUtils.json("_id: 1"));
        cmd.put("update", (Object)new Document("$inc", (Object)TestUtils.json("_id: 1")));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1"));
        try {
            this.db.runCommand((Bson)cmd);
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10148);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
    }

    @Test
    public void testFindAndModifyCommandUpdate() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Document cmd = new Document("findAndModify", (Object)this.collection.getNamespace().getCollectionName());
        cmd.put("query", (Object)TestUtils.json("_id: 1"));
        cmd.put("update", (Object)TestUtils.json("$inc: {a: 1}"));
        Document result = this.db.runCommand((Bson)cmd);
        Assertions.assertThat((Object)result.get((Object)"lastErrorObject")).isEqualTo((Object)TestUtils.json("updatedExisting: true, n: 1"));
        Assertions.assertThat((Integer)result.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
    }

    @Test
    public void testFindOneAndUpdateError() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1"));
        try {
            this.collection.findOneAndUpdate((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {_id: 1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10148);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
    }

    @Test
    public void testFindOneAndUpdateFields() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1"));
        Document result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {a:1}"), new FindOneAndUpdateOptions().returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 2"));
    }

    @Test
    public void testFineOneAndUpdateNotFound() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1"));
        Document result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("_id: 2"), (Bson)new Document("$inc", (Object)TestUtils.json("a: 1")));
        Assertions.assertThat((Map)result).isNull();
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
    }

    @Test
    public void testFineOneAndUpdateRemove() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1"));
        Document result = (Document)this.collection.findOneAndDelete((Bson)TestUtils.json("_id: 1"));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testFineOneAndUpdateReturnNew() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1, b: {c: 1}"));
        Document query = TestUtils.json("_id: 1");
        Document update = TestUtils.json("$inc: {a: 1, 'b.c': 1}");
        Document result = (Document)this.collection.findOneAndUpdate((Bson)query, (Bson)update, new FindOneAndUpdateOptions().returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 2, b: {c: 2}"));
    }

    @Test
    public void testFineOneAndUpdateMax() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1, b: {c: 1}"));
        Document query = TestUtils.json("_id: 1");
        Document update = TestUtils.json("$max: {a: 2, 'b.c': 2, d : 'd'}");
        Document result = (Document)this.collection.findOneAndUpdate((Bson)query, (Bson)update, new FindOneAndUpdateOptions().returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 2, b: {c: 2}, d : 'd'"));
    }

    @Test
    public void testFineOneAndUpdateMin() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 2, b: {c: 1}"));
        Document query = TestUtils.json("_id: 1");
        Document update = TestUtils.json("$min: {a: 1, 'b.c': 2, d : 'd'}");
        Document result = (Document)this.collection.findOneAndUpdate((Bson)query, (Bson)update, new FindOneAndUpdateOptions().returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 1, b: {c: 1}, d : 'd'"));
    }

    @Test
    public void testFindOneAndUpdateReturnOld() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 1, b: {c: 1}"));
        Document query = TestUtils.json("_id: 1");
        Document update = TestUtils.json("$inc: {a: 1, 'b.c': 1}");
        Document result = (Document)this.collection.findOneAndUpdate((Bson)query, (Bson)update, new FindOneAndUpdateOptions().returnDocument(ReturnDocument.BEFORE));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 1, b: {c: 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)query).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 2, b: {c: 2}"));
    }

    @Test
    public void testFindOneAndUpdateSorted() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a:15"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, a:10"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3, a:20"));
        Document order = TestUtils.json("a:1");
        Document result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$inc: {a: 1}"), new FindOneAndUpdateOptions().sort((Bson)order).returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 2, a: 11"));
        order = TestUtils.json("a: -1");
        result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$inc: {a: 1}"), new FindOneAndUpdateOptions().sort((Bson)order).returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 3, a: 21"));
    }

    @Test
    public void testFindOneAndUpdateUpsert() {
        Document result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {a:1}"), new FindOneAndUpdateOptions().upsert(true).returnDocument(ReturnDocument.AFTER));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
    }

    @Test
    public void testFindOneAndUpdateUpsertReturnBefore() {
        Document result = (Document)this.collection.findOneAndUpdate((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {a:1}"), new FindOneAndUpdateOptions().upsert(true).returnDocument(ReturnDocument.BEFORE));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("{}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
    }

    @Test
    public void testFindAndRemoveFromEmbeddedList() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: [1]"));
        Document result = (Document)this.collection.findOneAndDelete((Bson)TestUtils.json("_id: 1"));
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1, a: [1]"));
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testFindOne() {
        this.collection.insertOne((Object)TestUtils.json("key: 'value'"));
        this.collection.insertOne((Object)TestUtils.json("key: 'value'"));
        Document result = (Document)this.collection.find().first();
        Assertions.assertThat((Map)result).isNotNull();
        Assertions.assertThat((Object)result.get((Object)"_id")).isNotNull();
    }

    @Test
    public void testFindOneById() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Document result = (Document)this.collection.find((Bson)TestUtils.json("_id: 1")).first();
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 2")).first())).isNull();
    }

    @Test
    public void testFindOneIn() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Document result = (Document)this.collection.find((Bson)TestUtils.json("_id: {$in: [1,2]}")).first();
        Assertions.assertThat((Map)result).isEqualTo((Object)TestUtils.json("_id: 1"));
    }

    @Test
    public void testFindWithLimit() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        List actual = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")).limit(2));
        Assertions.assertThat(actual).containsExactly((Object[])new Document[]{TestUtils.json("_id: 1"), TestUtils.json("_id: 2")});
        List actualNegativeLimit = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")).limit(-2));
        Assertions.assertThat(actualNegativeLimit).isEqualTo(actual);
    }

    @Test
    public void testFindInReverseNaturalOrder() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        List actual = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("$natural: -1")));
        Assertions.assertThat(actual).containsOnly((Object[])new Document[]{TestUtils.json("_id: 1"), TestUtils.json("_id: 2")});
    }

    @Test
    public void testFindWithPattern() {
        this.collection.insertOne((Object)TestUtils.json("_id: 'marta'"));
        this.collection.insertOne((Object)TestUtils.json("_id: 'john', foo: 'bar'"));
        this.collection.insertOne((Object)TestUtils.json("_id: 'jon', foo: 'ba'"));
        this.collection.insertOne((Object)TestUtils.json("_id: 'jo'"));
        Assertions.assertThat(TestUtils.toArray(this.collection.find((Bson)new Document("_id", (Object)Pattern.compile("mart"))))).containsOnly((Object[])new Document[]{TestUtils.json("_id: 'marta'")});
        Assertions.assertThat(TestUtils.toArray(this.collection.find((Bson)new Document("foo", (Object)Pattern.compile("ba"))))).containsOnly((Object[])new Document[]{TestUtils.json("_id: 'john', foo: 'bar'"), TestUtils.json("_id: 'jon', foo: 'ba'")});
        Assertions.assertThat(TestUtils.toArray(this.collection.find((Bson)new Document("foo", (Object)Pattern.compile("ba$"))))).containsOnly((Object[])new Document[]{TestUtils.json("_id: 'jon', foo: 'ba'")});
    }

    @Test
    public void testFindWithQuery() {
        this.collection.insertOne((Object)TestUtils.json("name: 'jon'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'leo'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'neil'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'neil'"));
        Assertions.assertThat(TestUtils.toArray(this.collection.find((Bson)TestUtils.json("name: 'neil'")))).hasSize(2);
    }

    @Test
    public void testFindWithSkipLimit() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        Assertions.assertThat(TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")).limit(2).skip(2))).containsExactly((Object[])new Document[]{TestUtils.json("_id: 3"), TestUtils.json("_id: 4")});
    }

    @Test
    public void testFindWithSkipLimitInReverseOrder() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        Assertions.assertThat(TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: -1")).limit(2).skip(2))).containsExactly((Object[])new Document[]{TestUtils.json("_id: 2"), TestUtils.json("_id: 1")});
    }

    @Test
    public void testFindWithSkipLimitAfterDelete() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        this.collection.insertOne((Object)TestUtils.json("_id: 5"));
        this.collection.deleteOne((Bson)TestUtils.json("_id: 1"));
        this.collection.deleteOne((Bson)TestUtils.json("_id: 3"));
        Assertions.assertThat(TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")).limit(2).skip(2))).containsExactly((Object[])new Document[]{TestUtils.json("_id: 5")});
    }

    @Test
    public void testFullUpdateWithSameId() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, b: 5"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        this.collection.replaceOne((Bson)TestUtils.json("_id: 2, b:5"), (Object)TestUtils.json("_id: 2, a:5"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 2")).first())).isEqualTo((Object)TestUtils.json("_id: 2, a:5"));
    }

    @Test
    public void testGetCollection() {
        MongoCollection<Document> collection = this.getCollection("coll");
        this.getCollection("coll").insertOne((Object)TestUtils.json("{}"));
        Assertions.assertThat(collection).isNotNull();
        Assertions.assertThat(TestUtils.toArray(this.db.listCollectionNames())).contains((Object[])new String[]{"coll"});
    }

    @Test
    public void testNullId() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{_id: null, name: 'test'}"));
        Document result = (Document)this.collection.find((Bson)TestUtils.json("name: 'test'")).first();
        Assertions.assertThat((Map)result).isNotNull();
        Assertions.assertThat((Comparable)result.getObjectId((Object)"_id")).isNull();
        try {
            this.collection.insertOne((Object)TestUtils.json("_id: null"));
            Assert.fail((String)"MongoWriteException expected");
        }
        catch (MongoWriteException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"duplicate key error"});
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: null")).first())).isEqualTo((Object)TestUtils.json("{_id: null, name: 'test'}"));
        this.collection.deleteOne((Bson)TestUtils.json("_id: null"));
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testIdInQueryResultsInIndexOrder() {
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        List docs = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("_id: {$in: [3,2,1]}")));
        Assertions.assertThat(docs).containsExactlyInAnyOrder((Object[])new Document[]{TestUtils.json("_id: 1"), TestUtils.json("_id: 2"), TestUtils.json("_id: 3")});
    }

    @Test
    public void testIdNotAllowedToBeUpdated() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        try {
            this.collection.replaceOne((Bson)TestUtils.json("_id: 1"), (Object)TestUtils.json("_id:2, a:4"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot change _id of a document old: 1, new: 2"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)new Document("$set", (Object)TestUtils.json("_id: 2")));
            Assert.fail((String)"should throw exception");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
    }

    @Test
    public void testIllegalCommand() throws Exception {
        try {
            this.db.runCommand((Bson)TestUtils.json("foo: 1"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"no such cmd: foo"});
        }
    }

    @Test
    public void testInsert() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(0L);
        for (int i = 0; i < 3; ++i) {
            this.collection.insertOne((Object)new Document("_id", (Object)i));
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(3L);
        this.collection.insertOne((Object)TestUtils.json("foo: [1,2,3]"));
        this.collection.insertOne((Object)new Document("foo", (Object)new byte[10]));
        Document insertedObject = new Document("foo", (Object)UUID.randomUUID());
        this.collection.insertOne((Object)insertedObject);
        Document document = (Document)this.collection.find((Bson)insertedObject).first();
        Assertions.assertThat((Map)document).isEqualTo((Object)insertedObject);
    }

    @Test
    public void testInsertDuplicate() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(0L);
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
        try {
            this.collection.insertOne((Object)TestUtils.json("_id: 1"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"duplicate key error"});
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
    }

    @Test(expected=MongoException.class)
    public void testInsertDuplicateThrows() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
    }

    @Test(expected=MongoException.class)
    public void testInsertDuplicateWithConcernThrows() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.withWriteConcern(WriteConcern.ACKNOWLEDGED).insertOne((Object)TestUtils.json("_id: 1"));
    }

    @Test
    public void testInsertIncrementsCount() {
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
        this.collection.insertOne((Object)TestUtils.json("key: 'value'"));
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
    }

    @Test
    public void testInsertQuery() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(0L);
        Document insertedObject = TestUtils.json("_id: 1");
        insertedObject.put("foo", (Object)OTHER_TEST_DATABASE_NAME);
        this.collection.insertOne((Object)insertedObject);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)insertedObject).first())).isEqualTo((Object)insertedObject);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)new Document("_id", (Object)1L)).first())).isEqualTo((Object)insertedObject);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)new Document("_id", (Object)1.0)).first())).isEqualTo((Object)insertedObject);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)new Document("_id", (Object)Float.valueOf(1.0001f))).first())).isNull();
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("foo: 'bar'")).first())).isEqualTo((Object)insertedObject);
    }

    @Test
    public void testInsertRemove() throws Exception {
        for (int i = 0; i < 10; ++i) {
            this.collection.insertOne((Object)TestUtils.json("_id: 1"));
            Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
            this.collection.deleteOne((Bson)TestUtils.json("_id: 1"));
            Assertions.assertThat((long)this.collection.countDocuments()).isZero();
            this.collection.insertOne((Object)new Document("_id", (Object)i));
            this.collection.deleteOne((Bson)new Document("_id", (Object)i));
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
        this.collection.deleteOne((Bson)TestUtils.json("'doesnt exist': 1"));
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }

    @Test
    public void testInsertInSystemNamespace() throws Exception {
        try {
            this.getCollection("system.foobar").insertOne((Object)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(16459);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"attempt to insert in system namespace"});
        }
        try {
            this.getCollection("system.namespaces").insertOne((Object)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(16459);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"attempt to insert in system namespace"});
        }
    }

    @Test
    public void testListDatabaseNames() throws Exception {
        Assertions.assertThat(this.listDatabaseNames()).isEmpty();
        this.collection.insertOne((Object)TestUtils.json("{}"));
        Assertions.assertThat(this.listDatabaseNames()).containsExactly((Object[])new String[]{this.db.getName()});
        this.getDatabase().getCollection("some-collection").insertOne((Object)TestUtils.json("{}"));
        Assertions.assertThat(this.listDatabaseNames()).containsExactly((Object[])new String[]{OTHER_TEST_DATABASE_NAME, this.db.getName()});
    }

    private MongoDatabase getDatabase() {
        return this.syncClient.getDatabase(OTHER_TEST_DATABASE_NAME);
    }

    private List<String> listDatabaseNames() {
        ArrayList<String> databaseNames = new ArrayList<String>();
        for (String databaseName : this.syncClient.listDatabaseNames()) {
            databaseNames.add(databaseName);
        }
        return databaseNames;
    }

    @Test
    public void testMaxBsonSize() throws Exception {
        int maxBsonObjectSize = this.syncClient.getMaxBsonObjectSize();
        Assertions.assertThat((int)maxBsonObjectSize).isEqualTo(0x1000000);
    }

    @Test
    public void testQuery() throws Exception {
        Document obj = (Document)this.collection.find((Bson)TestUtils.json("_id: 1")).first();
        Assertions.assertThat((Map)obj).isNull();
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(0L);
    }

    @Test
    public void testQueryAll() throws Exception {
        ArrayList<Document> inserted = new ArrayList<Document>();
        for (int i = 0; i < 10; ++i) {
            Document obj = new Document("_id", (Object)i);
            this.collection.insertOne((Object)obj);
            inserted.add(obj);
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(10L);
        Assertions.assertThat(TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")))).isEqualTo(inserted);
    }

    @Test
    public void testQueryCount() throws Exception {
        for (int i = 0; i < 100; ++i) {
            this.collection.insertOne((Object)TestUtils.json("{}"));
        }
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(100L);
        Document obj = TestUtils.json("_id: 1");
        Assertions.assertThat((long)this.collection.countDocuments((Bson)obj)).isEqualTo(0L);
        this.collection.insertOne((Object)obj);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)obj)).isEqualTo(1L);
    }

    @Test
    public void testQueryLimitEmptyQuery() throws Exception {
        for (int i = 0; i < 5; ++i) {
            this.collection.insertOne((Object)TestUtils.json("{}"));
        }
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().limit(1))).isEqualTo(1L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().limit(-1))).isEqualTo(5L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"))).isEqualTo(5L);
    }

    @Test
    public void testQueryLimitSimpleQuery() throws Exception {
        for (int i = 0; i < 5; ++i) {
            this.collection.insertOne((Object)TestUtils.json("a:1"));
        }
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().limit(1))).isEqualTo(1L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().limit(-1))).isEqualTo(5L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"))).isEqualTo(5L);
    }

    @Test
    public void testQueryNull() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("foo: null")).first())).isEqualTo((Object)object);
    }

    @Test
    public void testQuerySkipLimitEmptyQuery() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().skip(3))).isEqualTo(0L);
        for (int i = 0; i < 10; ++i) {
            this.collection.insertOne((Object)TestUtils.json("{}"));
        }
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().skip(3))).isEqualTo(7L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().skip(15))).isEqualTo(0L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("{}"), new CountOptions().skip(3).limit(5))).isEqualTo(5L);
    }

    @Test
    public void testQuerySkipLimitSimpleQuery() throws Exception {
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().skip(3))).isEqualTo(0L);
        for (int i = 0; i < 10; ++i) {
            this.collection.insertOne((Object)TestUtils.json("a:1"));
        }
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().skip(3))).isEqualTo(7L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().skip(3).limit(5))).isEqualTo(5L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("a:1"), new CountOptions().skip(15).limit(5))).isEqualTo(0L);
    }

    @Test
    public void testQuerySort() throws Exception {
        double value;
        Random random = new Random(4711L);
        for (int i = 0; i < 10; ++i) {
            this.collection.insertOne((Object)new Document("_id", (Object)random.nextDouble()));
        }
        List<Object> objects = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: 1")));
        double before = Double.MIN_VALUE;
        for (Document obj : objects) {
            value = obj.getDouble((Object)"_id");
            Assertions.assertThat((double)value).isGreaterThanOrEqualTo(before);
            before = value;
        }
        objects = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("_id: -1")));
        before = Double.MAX_VALUE;
        for (Document obj : objects) {
            value = obj.getDouble((Object)"_id");
            Assertions.assertThat((double)value).isLessThanOrEqualTo(before);
            before = value;
        }
    }

    @Test
    public void testQueryWithFieldSelector() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("foo: 'bar'"));
        Document obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("foo: 1")).first();
        Assertions.assertThat((Iterable)obj.keySet()).containsOnly((Object[])new String[]{"_id", "foo"});
        obj = (Document)this.collection.find((Bson)TestUtils.json("foo:'bar'")).projection((Bson)TestUtils.json("_id: 1")).first();
        Assertions.assertThat((Iterable)obj.keySet()).containsOnly((Object[])new String[]{"_id"});
        obj = (Document)this.collection.find((Bson)TestUtils.json("foo: 'bar'")).projection((Bson)TestUtils.json("_id: 0, foo:1")).first();
        Assertions.assertThat((Iterable)obj.keySet()).containsOnly((Object[])new String[]{"foo"});
    }

    @Test
    public void testQueryWithDotNotationFieldSelector() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 123, index: false, foo: { a: 'a1', b: 0}"));
        Document obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("'foo.a': 1, 'foo.b': 1")).first();
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("_id: 123, foo: {a: 'a1', b: 0}"));
        obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("'foo.a': 1")).first();
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("_id: 123, foo: {a: 'a1'}"));
        obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("'foo.a': 1, index: 1, _id: 0")).first();
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("foo: {a: 'a1'}, index: false"));
        obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("foo: 1, _id: 0")).first();
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("foo: {a: 'a1', b: 0}"));
        obj = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("'foo.a.b.c.d': 1")).first();
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("_id: 123, foo: {}"));
    }

    @Test
    public void testQuerySystemNamespace() throws Exception {
        Assertions.assertThat((Map)((Map)this.getCollection("system.foobar").find().first())).isNull();
        Assertions.assertThat((Iterable)this.db.listCollectionNames()).isEmpty();
        this.collection.insertOne((Object)TestUtils.json("{}"));
        Document expectedObj = new Document("name", (Object)this.collection.getNamespace().getFullName());
        Document coll = (Document)this.getCollection("system.namespaces").find((Bson)expectedObj).first();
        Assertions.assertThat((Map)coll).isEqualTo((Object)expectedObj);
    }

    @Test
    public void testQueryAllExpression() throws Exception {
        this.collection.insertOne((Object)TestUtils.json(" _id : [ { x : 1 } , { x : 2  } ]"));
        this.collection.insertOne((Object)TestUtils.json(" _id : [ { x : 2 } , { x : 3  } ]"));
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("'_id.x':{$all:[1,2]}"))).isEqualTo(1L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)TestUtils.json("'_id.x':{$all:[2,3]}"))).isEqualTo(1L);
    }

    @Test
    public void testQueryWithSubdocumentIndex() throws Exception {
        this.collection.createIndex((Bson)TestUtils.json("action:{actionId:1}"), new IndexOptions().unique(true));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 1 }, value: 'a'"));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 2 }, value: 'b'"));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 3 }, value: 'c'"));
        Document foundWithNestedDocument = (Document)this.collection.find((Bson)TestUtils.json("action: { actionId: 2 }")).first();
        Assertions.assertThat((Object)foundWithNestedDocument.get((Object)"value")).isEqualTo((Object)"b");
        Document foundWithDotNotation = (Document)this.collection.find((Bson)TestUtils.json("'action.actionId': 2")).first();
        Assertions.assertThat((Object)foundWithDotNotation.get((Object)"value")).isEqualTo((Object)"b");
    }

    @Test
    public void testQueryBinaryData() throws Exception {
        byte[] firstBytes = new byte[]{1, 2, 3};
        byte[] secondBytes = new byte[]{3, 2, 1};
        this.collection.insertOne((Object)new Document("_id", (Object)1).append("test", (Object)firstBytes));
        this.collection.insertOne((Object)new Document("_id", (Object)2).append("test", (Object)secondBytes));
        Document first = (Document)this.collection.find((Bson)new Document("test", (Object)firstBytes)).first();
        Assertions.assertThat((Map)first).isNotNull();
        Assertions.assertThat((Object)first.get((Object)"_id")).isEqualTo((Object)1);
        Document second = (Document)this.collection.find((Bson)new Document("test", (Object)secondBytes)).first();
        Assertions.assertThat((Map)second).isNotNull();
        Assertions.assertThat((Object)second.get((Object)"_id")).isEqualTo((Object)2);
    }

    @Test
    public void testRemove() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 4"));
        this.collection.deleteOne((Bson)TestUtils.json("_id: 2"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 2")).first())).isNull();
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(3L);
        this.collection.deleteMany((Bson)TestUtils.json("_id: {$gte: 3}"));
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(1L);
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1"));
    }

    @Test
    public void testRemoveSingle() throws Exception {
        Document obj = new Document("_id", (Object)ObjectId.get());
        this.collection.insertOne((Object)obj);
        this.collection.deleteOne((Bson)obj);
    }

    @Test
    public void testRemoveReturnsModifiedDocumentCount() {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.collection.insertOne((Object)TestUtils.json("{}"));
        DeleteResult result = this.collection.deleteMany((Bson)TestUtils.json("{}"));
        Assertions.assertThat((long)result.getDeletedCount()).isEqualTo(2L);
        result = this.collection.deleteMany((Bson)TestUtils.json("{}"));
        Assertions.assertThat((long)result.getDeletedCount()).isEqualTo(0L);
    }

    @Test
    public void testReservedCollectionNames() throws Exception {
        try {
            this.getCollection("foo$bar").insertOne((Object)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot insert into reserved $ collection"});
        }
        String veryLongString = "verylongstring";
        for (int i = 0; i < 5; ++i) {
            veryLongString = veryLongString + veryLongString;
        }
        try {
            this.getCollection(veryLongString).insertOne((Object)TestUtils.json("{}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"name too long"});
        }
    }

    @Test
    public void testServerStatus() throws Exception {
        Date before = new Date();
        Document serverStatus = this.runCommand("serverStatus");
        Assertions.assertThat((Integer)serverStatus.getInteger((Object)"ok")).isEqualTo(1);
        Assertions.assertThat((Object)serverStatus.get((Object)"uptime")).isInstanceOf(Number.class);
        Assertions.assertThat((Object)serverStatus.get((Object)"uptimeMillis")).isInstanceOf(Long.class);
        Date serverTime = (Date)serverStatus.get((Object)"localTime");
        Assertions.assertThat((Date)serverTime).isNotNull();
        Assertions.assertThat((boolean)serverTime.after(new Date())).isFalse();
        Assertions.assertThat((boolean)before.after(serverTime)).isFalse();
        Document connections = (Document)serverStatus.get((Object)"connections");
        Assertions.assertThat((Object)connections.get((Object)"current")).isNotNull();
    }

    @Test
    public void testPing() throws Exception {
        Document response = this.runCommand("ping");
        Assertions.assertThat((Integer)response.getInteger((Object)"ok")).isEqualTo(1);
    }

    @Test
    public void testPingTrue() throws Exception {
        Document command = new Document("ping", (Object)Boolean.TRUE);
        Document response = this.runCommand(command);
        Assertions.assertThat((Integer)response.getInteger((Object)"ok")).isEqualTo(1);
    }

    @Test
    public void testReplSetGetStatus() throws Exception {
        try {
            this.runCommand("replSetGetStatus");
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getErrorMessage()).contains(new CharSequence[]{"not running with --replSet"});
        }
    }

    @Test
    public void testWhatsMyUri() throws Exception {
        for (String dbName : new String[]{"admin", "local", "test"}) {
            Document result = this.syncClient.getDatabase(dbName).runCommand((Bson)new Document("whatsmyuri", (Object)1));
            Assertions.assertThat((Object)result.get((Object)"you")).isNotNull();
            Assertions.assertThat((String)result.get((Object)"you").toString()).startsWith((CharSequence)"127.0.0.1:");
        }
    }

    @Test
    public void testSort() {
        this.collection.insertOne((Object)TestUtils.json("a:1, _id:1"));
        this.collection.insertOne((Object)TestUtils.json("a:2, _id:2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 5"));
        this.collection.insertOne((Object)TestUtils.json("a:3, _id:3"));
        this.collection.insertOne((Object)TestUtils.json("a:4, _id:4"));
        List objs = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("a: -1")));
        Assertions.assertThat(objs).containsExactly((Object[])new Document[]{TestUtils.json("a:4, _id:4"), TestUtils.json("a:3, _id:3"), TestUtils.json("a:2, _id:2"), TestUtils.json("a:1, _id:1"), TestUtils.json("_id: 5")});
    }

    @Test
    public void testSortByEmbeddedKey() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: { b:1 }"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, a: { b:2 }"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3, a: { b:3 }"));
        List results = TestUtils.toArray(this.collection.find().sort((Bson)TestUtils.json("'a.b': -1")));
        Assertions.assertThat(results).containsExactly((Object[])new Document[]{TestUtils.json("_id: 3, a: { b:3 }"), TestUtils.json("_id: 2, a: { b:2 }"), TestUtils.json("_id: 1, a: { b:1 }")});
    }

    @Test
    public void testUpdate() throws Exception {
        Document object = TestUtils.json("_id: 1");
        Document newObject = TestUtils.json("{_id: 1, foo: 'bar'}");
        this.collection.insertOne((Object)object);
        UpdateResult result = this.collection.replaceOne((Bson)object, (Object)newObject);
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(1L);
        Assertions.assertThat((Object)result.getUpsertedId()).isNull();
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)newObject);
    }

    @Test
    public void testUpdateNothing() throws Exception {
        Document object = TestUtils.json("_id: 1");
        UpdateResult result = this.collection.replaceOne((Bson)object, (Object)object);
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(0L);
        Assertions.assertThat((long)result.getMatchedCount()).isEqualTo(0L);
        Assertions.assertThat((Object)result.getUpsertedId()).isNull();
    }

    @Test
    public void testUpdateBlank() throws Exception {
        Document document = TestUtils.json("'': 1, _id: 2, a: 3, b: 4");
        this.collection.insertOne((Object)document);
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set: {c:5}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("'': 1, _id: 2, a: 3, b: 4, c:5"));
    }

    @Test
    public void testUpdateEmptyPositional() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set:{'a.$.b': 1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(16650);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply the positional operator without a corresponding query field containing an array."});
        }
    }

    @Test
    public void testUpdateMultiplePositional() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{a: {b: {c: 1}}}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("{'a.b.c':1}"), (Bson)TestUtils.json("$set:{'a.$.b.$.c': 1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(16650);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply the positional operator without a corresponding query field containing an array."});
        }
    }

    @Test
    public void testUpdateIllegalFieldName() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{x:1}"));
        this.collection.updateOne((Bson)TestUtils.json("{x:1}"), (Bson)TestUtils.json("$set: {y:1}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("{x:1}"), (Bson)TestUtils.json("$set: {$z:1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(15896);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Modified field name may not start with $"});
        }
        this.collection.updateOne((Bson)TestUtils.json("{x:1}"), (Bson)TestUtils.json("$unset: {$z:1}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("{x:1}"), (Bson)TestUtils.json("$inc: {$z:1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(15896);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Modified field name may not start with $"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("{x:1}"), (Bson)TestUtils.json("$pushAll: {$z:[1,2,3]}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(15896);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Modified field name may not start with $"});
        }
    }

    @Test
    public void testUpdateSubdocument() throws Exception {
        try {
            this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("'a.b.c': 123"));
            Assert.fail((String)"IllegalArgumentException expected");
        }
        catch (IllegalArgumentException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Invalid BSON field name a.b.c"});
        }
    }

    @Test
    public void testUpdateIdNoChange() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.replaceOne((Bson)TestUtils.json("_id: 1"), (Object)TestUtils.json("_id: 1, a: 5"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 5"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$set: {_id: 1, b: 3}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 5, b: 3"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$set: {_id: 1, a: 7}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 7, b: 3"));
    }

    @Test
    public void testUpdatePush() throws Exception {
        Document idObj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)idObj);
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$push: {'field.subfield.subsubfield': 'value'}"));
        Document expected = TestUtils.json("_id: 1, field:{subfield:{subsubfield: ['value']}}");
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)expected);
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$set: {field: 'value'}"));
        try {
            this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$push: {field: 'value'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10141);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply $push modifier to non-array"});
        }
        Document pushObj = TestUtils.json("$push: {field1: 'value', field2: 'value2'}");
        this.collection.updateOne((Bson)idObj, (Bson)pushObj);
        expected = TestUtils.json("_id: 1, field: 'value', field1: ['value'], field2: ['value2']");
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)expected);
        pushObj = TestUtils.json("$push: {field1: 'value'}");
        this.collection.updateOne((Bson)idObj, (Bson)pushObj);
        expected.put("field1", Arrays.asList("value", "value"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)expected);
    }

    @Test
    public void testUpdatePushAll() throws Exception {
        Document idObj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)idObj);
        try {
            this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$pushAll: {field: 'value'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10153);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Modifier $pushAll allowed for arrays only"});
        }
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$pushAll: {field: ['value', 'value2']}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)TestUtils.json("_id: 1, field: ['value', 'value2']"));
    }

    @Test
    public void testUpdateAddToSet() throws Exception {
        Document idObj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)idObj);
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$addToSet: {'field.subfield.subsubfield': 'value'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)TestUtils.json("_id: 1, field:{subfield:{subsubfield:['value']}}"));
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$set: {field: 'value'}"));
        try {
            this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$addToSet: {field: 'value'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10141);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply $addToSet modifier to non-array"});
        }
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$addToSet: {field1: 'value', field2: 'value2'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)TestUtils.json("_id: 1, field: 'value', field1: ['value'], field2: ['value2']"));
        this.collection.updateOne((Bson)idObj, (Bson)TestUtils.json("$addToSet: {field1: 'value'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)idObj).first())).isEqualTo((Object)TestUtils.json("_id: 1, field: 'value', field1: ['value'], field2: ['value2']"));
    }

    @Test
    public void testUpdateAddToSetEach() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.addEachToSet((String)"a", Arrays.asList(6, 5, 4)));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [6,5,4]"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.addEachToSet((String)"a", Arrays.asList(3, 2, 1)));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [6,5,4,3,2,1]"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.addEachToSet((String)"a", Arrays.asList(7, 7, 9, 2)));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [6,5,4,3,2,1,7,9]"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.addEachToSet((String)"a", Arrays.asList(12, 13, 12)));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [6,5,4,3,2,1,7,9,12,13]"));
    }

    @Test
    public void testUpdateDatasize() throws Exception {
        Document obj = TestUtils.json("{_id:1, a:{x:[1, 2, 3]}}");
        this.collection.insertOne((Object)obj);
        Long oldSize = this.getCollStats().getLong((Object)"size");
        this.collection.updateOne((Bson)TestUtils.json("_id:1"), Updates.set((String)"a.x.0", (Object)3));
        Assertions.assertThat((Object)((Document)this.collection.find().first()).get((Object)"a")).isEqualTo((Object)TestUtils.json("x:[3,2,3]"));
        Long newSize = this.getCollStats().getLong((Object)"size");
        Assertions.assertThat((Object)newSize).isEqualTo((Object)oldSize);
        this.collection.updateOne((Bson)TestUtils.json("_id:1"), Updates.set((String)"a.x.0", (Object)"abc"));
        Long yetNewSize = this.getCollStats().getLong((Object)"size");
        Assertions.assertThat((long)(yetNewSize - oldSize)).isEqualTo(4L);
    }

    @Test
    public void testUpdatePull() throws Exception {
        Document obj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)obj);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)obj).first())).isEqualTo((Object)obj);
        this.collection.updateOne((Bson)obj, Updates.set((String)"field", (Object)"value"));
        try {
            this.collection.updateOne((Bson)obj, Updates.pull((String)"field", (Object)"value"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10142);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply $pull modifier to non-array"});
        }
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$set: {field: ['value1', 'value2', 'value1']}"));
        this.collection.updateOne((Bson)obj, Updates.pull((String)"field", (Object)"value1"));
        Assertions.assertThat((Object)((Document)this.collection.find((Bson)obj).first()).get((Object)"field")).isEqualTo(Collections.singletonList("value2"));
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("{$set: {field1: ['value1', 'value2', 'value1']}}"));
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$set: {field2: ['value3', 'value3', 'value1']}"));
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$pull: {field1: 'value2', field2: 'value3'}"));
        Assertions.assertThat((Object)((Document)this.collection.find((Bson)obj).first()).get((Object)"field1")).isEqualTo(Arrays.asList("value1", "value1"));
        Assertions.assertThat((Object)((Document)this.collection.find((Bson)obj).first()).get((Object)"field2")).isEqualTo(Collections.singletonList("value1"));
    }

    @Test
    public void testUpdatePullValueWithCondition() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, votes: [ 3, 5, 6, 7, 7, 8 ]"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$pull: { votes: { $gte: 6 } }"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, votes: [ 3, 5 ]"));
    }

    @Test
    public void testUpdatePullDocuments() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, results: [{item: 'A', score: 5}, {item: 'B', score: 8, comment: 'foobar'}]"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, results: [{item: 'C', score: 8, comment: 'foobar'}, {item: 'B', score: 4}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$pull: { results: { score: 8 , item: 'B' } }"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, results: [{item: 'A', score: 5}]"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 2")).first())).isEqualTo((Object)TestUtils.json("_id: 2, results: [{item: 'C', score: 8, comment: 'foobar'}, {item: 'B', score: 4}]"));
    }

    @Test
    public void testUpdatePullLeavesEmptyArray() {
        Document obj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)obj);
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$set: {field: [{'key1': 'value1', 'key2': 'value2'}]}"));
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$pull: {field: {'key1': 'value1'}}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)obj).first())).isEqualTo((Object)TestUtils.json("_id: 1, field: []"));
    }

    @Test
    public void testUpdatePullAll() throws Exception {
        Document obj = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)obj);
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$set: {field: 'value'}"));
        try {
            this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$pullAll: {field: 'value'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10142);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot apply $pullAll modifier to non-array"});
        }
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$set: {field1: ['value1', 'value2', 'value1', 'value3', 'value4', 'value3']}"));
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$pullAll: {field1: ['value1', 'value3']}"));
        Assertions.assertThat((Object)((Document)this.collection.find((Bson)obj).first()).get((Object)"field1")).isEqualTo(Arrays.asList("value2", "value4"));
        try {
            this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$pullAll: {field1: 'bar'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10153);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Modifier $pullAll allowed for arrays only"});
        }
    }

    @Test
    public void testUpdateSet() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {foo: 'bar'}"));
        Document expected = TestUtils.json("{}");
        expected.putAll((Map)object);
        expected.put("foo", (Object)OTHER_TEST_DATABASE_NAME);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {bar: 'bla'}"));
        expected.put(OTHER_TEST_DATABASE_NAME, (Object)"bla");
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)expected);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {'foo.bar': 'bla'}"));
        expected.put("foo", (Object)TestUtils.json("bar: 'bla'"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)expected);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {'foo.foo': '123'}"));
        ((Document)expected.get((Object)"foo")).put("foo", (Object)"123");
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)expected);
    }

    @Test
    public void testUpdateSetOnInsert() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {b: 3}, $setOnInsert: {a: 3}"), new UpdateOptions().upsert(true));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, b: 3, a: 3"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {b: 4}, $setOnInsert: {a: 5}"), new UpdateOptions().upsert(true));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, b: 4, a: 3"));
    }

    @Test
    public void testUpdateSetWithArrayIndices() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: [{x:0}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set: {'a.0.x': 3}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [{x:3}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set: {'a.1.z': 17}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [{x:3}, {z:17}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set: {'a.0.y': 7}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [{x:3, y:7}, {z:17}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$set: {'a.1': 'test'}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: [{x:3, y:7}, 'test']"));
    }

    @Test
    public void testUpdateUnsetWithArrayIndices() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a:[{x:0}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$unset: {'a.0.x': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a:[{}]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$unset: {'a.0': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a:[null]"));
        this.collection.updateOne((Bson)TestUtils.json("{}"), (Bson)TestUtils.json("$unset: {'a.10': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a:[null]"));
    }

    @Test
    public void testUpdateMax() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 1}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 1}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': 10}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 10}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': -100}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 10}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': '1'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : '1'}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': null}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : '1'}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$max: {'foo.bar': '2', 'buz' : 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : '2'}, buz : 1"));
    }

    @Test
    public void testUpdateMin() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 'b'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 'b'}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 'a'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 'a'}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 10}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 10}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 10}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 10}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 1}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 100}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : 1}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': null}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : null}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$min: {'foo.bar': 'a'}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, foo : {bar : null}"));
    }

    @Test
    public void testUpdateMaxCompareNumbers() throws Exception {
        Document object = TestUtils.json("_id: 1, highScore: 800, lowScore: 200");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$max: { highScore: 950 }"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, highScore: 950, lowScore: 200"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$max: { highScore: 870 }"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, highScore: 950, lowScore: 200"));
    }

    @Test
    public void testUpdateMaxCompareDates() throws Exception {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.US);
        Document object = new Document("_id", (Object)1).append("desc", (Object)"crafts").append("dateEntered", (Object)df.parse("2013-10-01T05:00:00")).append("dateExpired", (Object)df.parse("2013-10-01T16:38:16"));
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)new Document("$max", (Object)new Document("dateExpired", (Object)df.parse("2013-09-30T00:00:00"))));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, desc: 'crafts'").append("dateEntered", (Object)df.parse("2013-10-01T05:00:00")).append("dateExpired", (Object)df.parse("2013-10-01T16:38:16")));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)new Document("$max", (Object)new Document("dateExpired", (Object)df.parse("2014-01-07T00:00:00"))));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, desc: 'crafts'").append("dateEntered", (Object)df.parse("2013-10-01T05:00:00")).append("dateExpired", (Object)df.parse("2014-01-07T00:00:00")));
    }

    @Test
    public void testUpdateMinCompareNumbers() throws Exception {
        Document object = TestUtils.json("_id: 1, highScore: 800, lowScore: 200");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$min: { lowScore: 150 }"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, highScore: 800, lowScore: 150"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$min: { lowScore: 250 }"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, highScore: 800, lowScore: 150"));
    }

    @Test
    public void testUpdateMinCompareDates() throws Exception {
        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.US);
        Document object = new Document("_id", (Object)1).append("desc", (Object)"crafts").append("dateEntered", (Object)df.parse("2013-10-01T05:00:00")).append("dateExpired", (Object)df.parse("2013-10-01T16:38:16"));
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)new Document("$min", (Object)new Document("dateEntered", (Object)df.parse("2013-09-25T00:00:00"))));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, desc: 'crafts'").append("dateEntered", (Object)df.parse("2013-09-25T00:00:00")).append("dateExpired", (Object)df.parse("2013-10-01T16:38:16")));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)new Document("$min", (Object)new Document("dateEntered", (Object)df.parse("2014-01-07T00:00:00"))));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id: 1, desc: 'crafts'").append("dateEntered", (Object)df.parse("2013-09-25T00:00:00")).append("dateExpired", (Object)df.parse("2013-10-01T16:38:16")));
    }

    @Test
    public void testUpdatePop() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$pop: {'foo.bar': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$set: {'foo.bar': [1,2,3]}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id:1, foo:{bar:[1,2,3]}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$pop: {'foo.bar': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id:1, foo:{bar:[1,2]}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$pop: {'foo.bar': -1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id:1, foo:{bar:[2]}"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$pop: {'foo.bar': null}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id:1, foo:{bar:[]}"));
    }

    @Test
    public void testUpdateUnset() throws Exception {
        Document obj = TestUtils.json("_id: 1, a: 1, b: null, c: 'value'");
        this.collection.insertOne((Object)obj);
        try {
            this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$unset: {_id: ''}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10148);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
        this.collection.updateOne((Bson)obj, (Bson)TestUtils.json("$unset: {a:'', b:''}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, c: 'value'"));
        this.collection.updateOne((Bson)obj, Updates.unset((String)"c.y"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, c: 'value'"));
        this.collection.replaceOne((Bson)TestUtils.json("_id: 1"), (Object)TestUtils.json("a: {b: 'foo', c: 'bar'}"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$unset: {'a.b':1}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: {c: 'bar'}"));
    }

    @Test
    public void testUpdateWithIdIn() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        Document update = TestUtils.json("$push: {n: {_id: 2, u:3}}, $inc: {c:4}");
        Document expected = TestUtils.json("_id: 1, n: [{_id: 2, u:3}], c:4");
        this.collection.updateOne((Bson)TestUtils.json("_id: {$in: [1]}"), (Bson)update);
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)expected);
    }

    @Test
    public void testUpdateMulti() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("a: 1"));
        this.collection.insertOne((Object)TestUtils.json("a: 1"));
        UpdateResult result = this.collection.updateOne((Bson)TestUtils.json("a: 1"), (Bson)TestUtils.json("$set: {b: 2}"));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(1L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)new Document("b", (Object)2))).isEqualTo(1L);
        result = this.collection.updateMany((Bson)TestUtils.json("a: 1"), (Bson)TestUtils.json("$set: {b: 3}"));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(2L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)new Document("b", (Object)2))).isEqualTo(0L);
        Assertions.assertThat((long)this.collection.countDocuments((Bson)new Document("b", (Object)3))).isEqualTo(2L);
    }

    @Test
    public void testUpdateIllegalInt() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: {x:1}"));
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {a: 1}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot increment value"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$inc: {'a.x': 'b'}"));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot increment with non-numeric value"});
        }
    }

    @Test
    public void testUpdateWithIdInMulti() {
        this.collection.insertMany(Arrays.asList(TestUtils.json("_id: 1"), TestUtils.json("_id: 2")));
        this.collection.updateMany((Bson)TestUtils.json("_id: {$in:[1,2]}"), (Bson)TestUtils.json("$set: {n:1}"));
        List results = TestUtils.toArray(this.collection.find());
        Assertions.assertThat(results).containsOnly((Object[])new Document[]{TestUtils.json("_id: 1, n:1"), TestUtils.json("_id: 2, n: 1")});
    }

    @Test
    public void testUpdateWithIdInMultiReturnModifiedDocumentCount() {
        this.collection.insertMany(Arrays.asList(TestUtils.json("_id: 1"), TestUtils.json("_id: 2")));
        UpdateResult result = this.collection.updateMany((Bson)TestUtils.json("_id: {$in:[1,2]}"), (Bson)TestUtils.json("$set:{n:1}"));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(2L);
    }

    @Test
    public void testUpdateWithIdQuery() {
        this.collection.insertMany(Arrays.asList(TestUtils.json("_id: 1"), TestUtils.json("_id: 2")));
        this.collection.updateMany((Bson)TestUtils.json("_id: {$gt:1}"), (Bson)TestUtils.json("$set: {n:1}"));
        List results = TestUtils.toArray(this.collection.find());
        Assertions.assertThat(results).containsOnly((Object[])new Document[]{TestUtils.json("_id: 1"), TestUtils.json("_id: 2, n:1")});
    }

    @Test
    public void testUpdateWithObjectId() {
        this.collection.insertOne((Object)TestUtils.json("_id: {n:1}"));
        UpdateResult result = this.collection.updateOne((Bson)TestUtils.json("_id: {n:1}"), (Bson)TestUtils.json("$set: {a:1}"));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(1L);
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: {n:1}, a:1"));
    }

    @Test
    public void testUpdateArrayMatch() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id:1, a:[{x:1,y:1}, {x:2,y:2}, {x:3,y:3}]"));
        this.collection.updateOne((Bson)TestUtils.json("'a.x': 2"), (Bson)TestUtils.json("$inc: {'a.$.y': 1}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("'a.x': 2")).first())).isEqualTo((Object)TestUtils.json("_id:1, a:[{x:1,y:1}, {x:2,y:3}, {x:3,y:3}]"));
        this.collection.insertOne((Object)TestUtils.json("{'array': [{'123a':{'name': 'old'}}]}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("{'array.123a.name': 'old'}")).first())).isNotNull();
        this.collection.updateOne((Bson)TestUtils.json("{'array.123a.name': 'old'}"), (Bson)TestUtils.json("{$set: {'array.$.123a.name': 'new'}}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("{'array.123a.name': 'new'}")).first())).isNotNull();
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("{'array.123a.name': 'old'}")).first())).isNull();
    }

    @Test
    public void testMultiUpdateArrayMatch() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.collection.insertOne((Object)TestUtils.json("x:[1,2,3]"));
        this.collection.insertOne((Object)TestUtils.json("x:99"));
        this.collection.updateMany((Bson)TestUtils.json("x:2"), (Bson)TestUtils.json("$inc:{'x.$': 1}"));
        Assertions.assertThat((Object)((Document)this.collection.find((Bson)TestUtils.json("x:1")).first()).get((Object)"x")).isEqualTo(Arrays.asList(1, 3, 3));
    }

    @Test
    public void testUpsert() {
        UpdateResult result = this.collection.updateMany((Bson)TestUtils.json("n:'jon'"), (Bson)TestUtils.json("$inc:{a:1}"), new UpdateOptions().upsert(true));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(0L);
        Document object = (Document)this.collection.find().first();
        Assertions.assertThat((Object)result.getUpsertedId()).isEqualTo((Object)new BsonObjectId(object.getObjectId((Object)"_id")));
        object.remove((Object)"_id");
        Assertions.assertThat((Map)object).isEqualTo((Object)TestUtils.json("n:'jon', a:1"));
        result = this.collection.updateOne((Bson)TestUtils.json("_id: 17, n:'jon'"), (Bson)TestUtils.json("$inc:{a:1}"), new UpdateOptions().upsert(true));
        Assertions.assertThat((Object)result.getUpsertedId()).isNull();
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id:17")).first())).isEqualTo((Object)TestUtils.json("_id: 17, n:'jon', a:1"));
    }

    @Test
    public void testUpsertFieldOrder() throws Exception {
        this.collection.updateOne((Bson)TestUtils.json("'x.y': 2"), (Bson)TestUtils.json("$inc: {a:7}"), new UpdateOptions().upsert(true));
        Document obj = (Document)this.collection.find().first();
        obj.remove((Object)"_id");
        Assertions.assertThat((Map)obj).isEqualTo((Object)TestUtils.json("x:{y:2}, a:7"));
    }

    @Test
    public void testUpsertWithoutId() {
        UpdateResult result = this.collection.updateOne(Filters.eq((String)"a", (Object)1), Updates.set((String)"a", (Object)2), new UpdateOptions().upsert(true));
        Assertions.assertThat((long)result.getModifiedCount()).isEqualTo(0L);
        Assertions.assertThat((Object)result.getUpsertedId()).isNotNull();
        Assertions.assertThat((Object)((Document)this.collection.find().first()).get((Object)"_id")).isInstanceOf(ObjectId.class);
        Assertions.assertThat((Object)((Document)this.collection.find().first()).get((Object)"a")).isEqualTo((Object)2);
    }

    @Test
    public void testUpsertOnIdWithPush() {
        Document update1 = TestUtils.json("$push: {c: {a:1, b:2} }");
        Document update2 = TestUtils.json("$push: {c: {a:3, b:4} }");
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)update1, new UpdateOptions().upsert(true));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)update2, new UpdateOptions().upsert(true));
        Document expected = TestUtils.json("_id: 1, c: [{a:1, b:2}, {a:3, b:4}]");
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("'c.a':3, 'c.b':4")).first())).isEqualTo((Object)expected);
    }

    @Test
    public void testUpsertWithConditional() {
        Document query = TestUtils.json("_id: 1, b: {$gt: 5}");
        Document update = TestUtils.json("$inc: {a: 1}");
        UpdateResult updateResult = this.collection.updateOne((Bson)query, (Bson)update, new UpdateOptions().upsert(true));
        Assertions.assertThat((long)updateResult.getModifiedCount()).isZero();
        Assertions.assertThat((long)updateResult.getMatchedCount()).isZero();
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 1"));
    }

    @Test
    public void testUpsertWithoutChange() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, a: 2, b: 3"));
        Document query = TestUtils.json("_id: 1");
        Document update = TestUtils.json("$set: {a: 2}");
        UpdateResult updateResult = this.collection.updateOne((Bson)query, (Bson)update, new UpdateOptions().upsert(true));
        Assertions.assertThat((long)updateResult.getModifiedCount()).isZero();
        Assertions.assertThat((long)updateResult.getMatchedCount()).isOne();
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 2, b: 3"));
    }

    @Test
    public void testUpsertWithEmbeddedQuery() {
        this.collection.updateOne((Bson)TestUtils.json("_id: 1, 'e.i': 1"), (Bson)TestUtils.json("$set: {a:1}"), new UpdateOptions().upsert(true));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)TestUtils.json("_id: 1")).first())).isEqualTo((Object)TestUtils.json("_id:1, e: {i:1}, a:1"));
    }

    @Test
    public void testUpsertWithIdIn() throws Exception {
        Document query = TestUtils.json("_id: {$in: [1]}");
        Document update = TestUtils.json("$push: {n: {_id: 2 ,u : 3}}, $inc: {c: 4}");
        Document expected = TestUtils.json("_id: 1, n: [{_id: 2 ,u : 3}], c: 4");
        UpdateResult updateResult = this.collection.updateOne((Bson)query, (Bson)update, new UpdateOptions().upsert(true));
        Assertions.assertThat((long)updateResult.getModifiedCount()).isZero();
        Assertions.assertThat((long)updateResult.getMatchedCount()).isZero();
        Document actual = (Document)this.collection.find().first();
        Assertions.assertThat((Map)actual).isEqualTo((Object)expected);
    }

    @Test
    public void testUpdateWithMultiplyOperator() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$mul: {a: 2}, $set: {b: 2}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 0, b: 2"));
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$mul: {b: 2.5}, $inc: {a: 0.5}"));
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)TestUtils.json("_id: 1, a: 0.5, b: 5.0"));
    }

    @Test
    public void testUpdateWithIllegalMultiplyFails() throws Exception {
        Document object = TestUtils.json("_id: 1, foo: 'x', bar: 1");
        this.collection.insertOne((Object)object);
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$mul: {_id: 2}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"_id"});
        }
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$mul: {foo: 2}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot multiply value 'x'"});
        }
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$mul: {bar: 'x'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"cannot multiply with non-numeric value"});
        }
    }

    @Test
    public void testIsMaster() throws Exception {
        Document isMaster = this.db.runCommand((Bson)new Document("isMaster", (Object)1));
        Assertions.assertThat((Boolean)isMaster.getBoolean((Object)"ismaster")).isTrue();
        Assertions.assertThat((Date)isMaster.getDate((Object)"localTime")).isInstanceOf(Date.class);
        Assertions.assertThat((Integer)isMaster.getInteger((Object)"maxBsonObjectSize")).isGreaterThan(1000);
        Assertions.assertThat((Integer)isMaster.getInteger((Object)"maxMessageSizeBytes")).isGreaterThan((Comparable)isMaster.getInteger((Object)"maxBsonObjectSize"));
    }

    @Test
    public void testFindWithNullOrNoFieldFilter() {
        this.collection.insertOne((Object)TestUtils.json("name: 'jon', group: 'group1'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'leo', group: 'group1'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'neil1', group: 'group2'"));
        this.collection.insertOne((Object)TestUtils.json("name: 'neil2', group: null"));
        this.collection.insertOne((Object)TestUtils.json("name: 'neil3'"));
        List objs = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("group: null")));
        ((ListAssert)Assertions.assertThat(objs).as("should have two neils (neil2, neil3)", new Object[0])).hasSize(2);
        objs = TestUtils.toArray(this.collection.find(Filters.exists((String)"group", (boolean)false)));
        ((ListAssert)Assertions.assertThat(objs).as("should have one neils (neil3)", new Object[0])).hasSize(1);
        objs = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("other: null")));
        ((ListAssert)Assertions.assertThat(objs).as("should return all documents", new Object[0])).hasSize(5);
        objs = TestUtils.toArray(this.collection.find(Filters.exists((String)"other", (boolean)false)));
        ((ListAssert)Assertions.assertThat(objs).as("should return all documents", new Object[0])).hasSize(5);
    }

    @Test
    public void testInsertsWithUniqueIndex() {
        this.collection.createIndex((Bson)new Document("uniqueKeyField", (Object)1), new IndexOptions().unique(true));
        this.collection.insertOne((Object)TestUtils.json("uniqueKeyField: 'abc1', afield: 'avalue'"));
        this.collection.insertOne((Object)TestUtils.json("uniqueKeyField: 'abc2', afield: 'avalue'"));
        this.collection.insertOne((Object)TestUtils.json("uniqueKeyField: 'abc3', afield: 'avalue'"));
        try {
            this.collection.insertOne((Object)TestUtils.json("uniqueKeyField: 'abc2', afield: 'avalue'"));
            Assert.fail((String)"MongoWriteException expected");
        }
        catch (MongoWriteException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"duplicate key error"});
        }
    }

    @Test
    public void testInsertBinaryData() throws Exception {
        this.collection.insertOne((Object)new Document("test", (Object)new byte[]{1, 2, 3}));
    }

    @Test
    public void testUniqueIndexWithSubdocument() {
        this.collection.createIndex((Bson)new Document("action.actionId", (Object)1), new IndexOptions().unique(true));
        this.collection.insertOne((Object)TestUtils.json("action: 'abc1'"));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 1 }"));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 2 }"));
        this.collection.insertOne((Object)TestUtils.json("action: { actionId: 3 }"));
        try {
            this.collection.insertOne((Object)TestUtils.json("action: { actionId: 1 }"));
            Assert.fail((String)"MongoWriteException expected");
        }
        catch (MongoWriteException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"duplicate key error"});
        }
    }

    @Test
    public void testAddNonUniqueIndexOnNonIdField() {
        this.collection.createIndex((Bson)new Document("someField", (Object)1), new IndexOptions().unique(false));
        this.collection.insertOne((Object)TestUtils.json("someField: 'abc'"));
        this.collection.insertOne((Object)TestUtils.json("someField: 'abc'"));
    }

    @Test
    public void testCompoundUniqueIndicesNotSupportedAndThrowsException() {
        try {
            this.collection.createIndex((Bson)new Document("a", (Object)1).append("b", (Object)1), new IndexOptions().unique(true));
            Assert.fail((String)"MongoException expected");
        }
        catch (MongoException mongoException) {
            // empty catch block
        }
    }

    @Test
    public void testCursorOptionNoTimeout() throws Exception {
        try (MongoCursor cursor = this.collection.find().noCursorTimeout(true).iterator();){
            Assert.assertFalse((boolean)cursor.hasNext());
        }
    }

    @Test
    public void testBulkInsert() throws Exception {
        ArrayList<InsertOneModel> inserts = new ArrayList<InsertOneModel>();
        inserts.add(new InsertOneModel((Object)TestUtils.json("_id: 1")));
        inserts.add(new InsertOneModel((Object)TestUtils.json("_id: 2")));
        inserts.add(new InsertOneModel((Object)TestUtils.json("_id: 3")));
        BulkWriteResult result = this.collection.bulkWrite(inserts);
        Assertions.assertThat((int)result.getInsertedCount()).isEqualTo(3);
    }

    @Test
    public void testBulkUpdateOrdered() throws Exception {
        this.testBulkUpdate(true);
    }

    @Test
    public void testBulkUpdateUnordered() throws Exception {
        this.testBulkUpdate(false);
    }

    private void testBulkUpdate(boolean ordered) {
        this.insertUpdateInBulk(ordered);
        this.removeInBulk(ordered);
        this.insertUpdateInBulkNoMatch(ordered);
    }

    @Test
    public void testUpdateCurrentDateIllegalTypeSpecification() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {lastModified: null}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(2);
            ((AbstractStringAssert)Assertions.assertThat((String)e.getErrorMessage()).startsWith((CharSequence)"NULL")).contains(new CharSequence[]{"is not a valid type"});
        }
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {lastModified: 123.456}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(2);
            ((AbstractStringAssert)Assertions.assertThat((String)e.getErrorMessage()).startsWith((CharSequence)"Double")).contains(new CharSequence[]{"is not a valid type"});
        }
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {lastModified: 'foo'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(2);
            ((AbstractStringAssert)Assertions.assertThat((String)e.getErrorMessage()).startsWith((CharSequence)"String")).contains(new CharSequence[]{"is not a valid type"});
        }
        try {
            this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {lastModified: {$type: 'foo'}}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(2);
            Assertions.assertThat((String)e.getErrorMessage()).startsWith((CharSequence)"The '$type' string field is required to be 'date' or 'timestamp'");
        }
        Assertions.assertThat((Map)((Map)this.collection.find((Bson)object).first())).isEqualTo((Object)object);
    }

    @Test
    public void testUpdateCurrentDate() throws Exception {
        Document object = TestUtils.json("_id: 1");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {'x.lastModified': true}"));
        Assertions.assertThat((Object)((Document)((Document)this.collection.find((Bson)object).first()).get((Object)"x")).get((Object)"lastModified")).isInstanceOf(Date.class);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {'x.lastModified': {$type: 'date'}}"));
        Assertions.assertThat((Object)((Document)((Document)this.collection.find((Bson)object).first()).get((Object)"x")).get((Object)"lastModified")).isInstanceOf(Date.class);
        this.collection.updateOne((Bson)object, (Bson)TestUtils.json("$currentDate: {'x.lastModified': {$type: 'timestamp'}}"));
        Assertions.assertThat((Object)((Document)((Document)this.collection.find((Bson)object).first()).get((Object)"x")).get((Object)"lastModified")).isInstanceOf(BsonTimestamp.class);
    }

    @Test
    public void testRenameField() throws Exception {
        Document object = TestUtils.json("_id: 1, foo: 'x', bar: 'y'");
        this.collection.insertOne((Object)object);
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {foo: 'foo2', bar: 'bar2'}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, foo2: 'x', bar2: 'y'"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {'bar2': 'foo', foo2: 'bar'}"));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, bar: 'x', foo: 'y'"));
    }

    @Test
    public void testRenameFieldIllegalValue() throws Exception {
        Document object = TestUtils.json("_id: 1, foo: 'x', bar: 'y'");
        this.collection.insertOne((Object)object);
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {foo: 12345}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"The 'to' field for $rename must be a string"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {'_id': 'id'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {foo: '_id'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Mod on _id not allowed"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {foo: 'bar', 'bar': 'bar2'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot update 'bar' and 'bar' at the same time"});
        }
        try {
            this.collection.updateOne((Bson)TestUtils.json("_id: 1"), (Bson)TestUtils.json("$rename: {bar: 'foo', bar2: 'foo'}"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"Cannot update 'foo' and 'foo' at the same time"});
        }
    }

    @Test
    public void testRenameCollection() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        this.collection.renameCollection(new MongoNamespace(this.collection.getNamespace().getDatabaseName(), "other-collection-name"));
        List collectionNames = TestUtils.toArray(this.db.listCollectionNames());
        Assertions.assertThat(collectionNames).containsOnly((Object[])new String[]{"system.indexes", "other-collection-name"});
        Assertions.assertThat((long)this.getCollection("other-collection-name").countDocuments()).isEqualTo(3L);
    }

    @Test
    public void testRenameCollection_targetAlreadyExists() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        MongoCollection otherCollection = this.db.getCollection("other-collection-name");
        otherCollection.insertOne((Object)TestUtils.json("_id: 1"));
        try {
            this.collection.renameCollection(new MongoNamespace(this.db.getName(), "other-collection-name"));
            Assert.fail((String)"MongoCommandException expected");
        }
        catch (MongoCommandException e) {
            Assertions.assertThat((String)e.getErrorMessage()).isEqualTo((Object)"target namespace already exists");
        }
        List collectionNames = TestUtils.toArray(this.db.listCollectionNames());
        Assertions.assertThat(collectionNames).containsOnly((Object[])new String[]{"system.indexes", this.collection.getNamespace().getCollectionName(), "other-collection-name"});
        Assertions.assertThat((long)this.collection.countDocuments()).isEqualTo(3L);
        Assertions.assertThat((long)this.getCollection("other-collection-name").countDocuments()).isEqualTo(1L);
    }

    @Test
    public void testRenameCollection_dropTarget() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3"));
        MongoCollection otherCollection = this.db.getCollection("other-collection-name");
        otherCollection.insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.renameCollection(new MongoNamespace(this.db.getName(), "other-collection-name"), new RenameCollectionOptions().dropTarget(true));
        List collectionNames = TestUtils.toArray(this.db.listCollectionNames());
        Assertions.assertThat(collectionNames).containsOnly((Object[])new String[]{"system.indexes", "other-collection-name"});
        Assertions.assertThat((long)this.getCollection("other-collection-name").countDocuments()).isEqualTo(3L);
    }

    @Test
    public void testListIndexes_empty() throws Exception {
        Assertions.assertThat((Iterable)this.collection.listIndexes()).isEmpty();
    }

    @Test
    public void testListIndexes() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        this.db.getCollection("other").insertOne((Object)TestUtils.json("_id: 1"));
        this.collection.createIndex((Bson)TestUtils.json("bla: 1"));
        List indexInfo = TestUtils.toArray(this.collection.listIndexes());
        Assertions.assertThat(indexInfo).containsOnly((Object[])new Document[]{TestUtils.json("name:'_id_', ns:'testdb.testcoll', key:{_id:1}"), TestUtils.json("name:'_id_', ns:'testdb.other', key:{_id:1}"), TestUtils.json("name:'bla_1', ns:'testdb.testcoll', key:{bla:1}")});
    }

    @Test
    public void testFieldSelection_deselectId() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, order:1, visits: 2"));
        Document document = (Document)this.collection.find((Bson)TestUtils.json("{}")).projection((Bson)TestUtils.json("_id: 0")).first();
        Assertions.assertThat((Map)document).isEqualTo((Object)TestUtils.json("order:1, visits:2"));
    }

    @Test
    public void testFieldSelection_deselectOneField() {
        Document obj = TestUtils.json("_id: 1, order:1, visits: 2, eid: 12345");
        this.collection.insertOne((Object)obj);
        Document document = (Document)this.collection.find((Bson)new Document()).projection((Bson)TestUtils.json("visits: 0")).first();
        Assertions.assertThat((Map)document).isEqualTo((Object)TestUtils.json("_id:1, order:1, eid: 12345"));
    }

    @Test
    public void testFieldSelection_deselectTwoFields() {
        Document obj = TestUtils.json("_id: 1, order:1, visits: 2, eid: 12345");
        this.collection.insertOne((Object)obj);
        Document document = (Document)this.collection.find((Bson)new Document()).projection((Bson)TestUtils.json("visits: 0, eid: 0")).first();
        Assertions.assertThat((Map)document).isEqualTo((Object)TestUtils.json("_id:1, order:1"));
    }

    @Test
    public void testFieldSelection_selectAndDeselectFields() {
        Document obj = TestUtils.json("_id: 1, order:1, visits: 2, eid: 12345");
        this.collection.insertOne((Object)obj);
        Document document = (Document)this.collection.find((Bson)new Document()).projection((Bson)TestUtils.json("visits: 0, eid: 1")).first();
        Assertions.assertThat((Map)document).isEqualTo((Object)TestUtils.json("_id:1, eid: 12345"));
    }

    @Test
    public void testPullWithInPattern() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, tags: ['aa', 'bb', 'ab', 'cc']"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.pullByFilter((Bson)Filters.in((String)"tags", (Object[])new Pattern[]{Pattern.compile("a+")})));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, tags: ['bb', 'cc']"));
    }

    @Test
    public void testPullWithInPatternAnchored() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, tags: ['aa', 'bb', 'ab', 'cc']"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.pullByFilter((Bson)Filters.in((String)"tags", (Object[])new Pattern[]{Pattern.compile("^a+$")})));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, tags: ['bb', 'ab', 'cc']"));
    }

    @Test
    public void testPullWithInNumbers() {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, values: [1, 2, 2.5, 3.0, 4]"));
        this.collection.updateOne((Bson)TestUtils.json("_id: 1"), Updates.pullByFilter((Bson)Filters.in((String)"values", Arrays.asList(2.0, 3, 4L))));
        Assertions.assertThat((Map)((Map)this.collection.find().first())).isEqualTo((Object)TestUtils.json("_id: 1, values: [1, 2.5]"));
    }

    @Test
    public void testDocumentWithHashMap() {
        HashMap<String, String> value = new HashMap<String, String>();
        value.put("foo", OTHER_TEST_DATABASE_NAME);
        this.collection.insertOne((Object)new Document("_id", (Object)1).append("map", value));
        Bson document = (Bson)this.collection.find().first();
        Assertions.assertThat((Object)document).isEqualTo((Object)TestUtils.json("{_id: 1, map: {foo: 'bar'}}"));
    }

    @Test
    public void testFindAndOfOrs() throws Exception {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        this.collection.insertOne((Object)new Document("_id", (Object)1).append("published", (Object)true).append("startDate", (Object)dateFormat.parse("2015-03-01 13:20:05")));
        this.collection.insertOne((Object)new Document("_id", (Object)2).append("published", (Object)true).append("expiration", (Object)dateFormat.parse("2020-12-31 18:00:00")));
        this.collection.insertOne((Object)new Document("_id", (Object)3).append("published", (Object)true));
        this.collection.insertOne((Object)new Document("_id", (Object)4).append("published", (Object)false));
        this.collection.insertOne((Object)new Document("_id", (Object)5).append("published", (Object)true).append("startDate", (Object)dateFormat.parse("2017-01-01 00:00:00")));
        this.collection.insertOne((Object)new Document("_id", (Object)6).append("published", (Object)true).append("expiration", (Object)dateFormat.parse("2016-01-01 00:00:00")));
        Date now = dateFormat.parse("2016-01-01 00:00:00");
        Bson query = Filters.and((Bson[])new Bson[]{Filters.ne((String)"published", (Object)false), Filters.or((Bson[])new Bson[]{Filters.exists((String)"startDate", (boolean)false), Filters.lt((String)"startDate", (Object)now)}), Filters.or((Bson[])new Bson[]{Filters.exists((String)"expiration", (boolean)false), Filters.gt((String)"expiration", (Object)now)})});
        List documents = TestUtils.toArray(this.collection.find(query).projection((Bson)TestUtils.json("_id: 1")));
        Assertions.assertThat(documents).containsOnly((Object[])new Document[]{TestUtils.json("_id: 1"), TestUtils.json("_id: 2"), TestUtils.json("_id: 3")});
    }

    @Test
    public void testInOperatorWithNullValue() {
        this.collection.insertMany(Arrays.asList(TestUtils.json("_id: 1, a: 1"), TestUtils.json("_id: 2, a: 2"), TestUtils.json("_id: 3, a: 3"), TestUtils.json("_id: 4, a: 4"), TestUtils.json("_id: 5")));
        Bson inQueryWithNull = Filters.in((String)"a", (Object[])new Integer[]{2, null, 3});
        List results = TestUtils.toArray(this.collection.find(inQueryWithNull).projection((Bson)TestUtils.json("_id: 1")));
        Assertions.assertThat(results).containsExactly((Object[])new Document[]{TestUtils.json("_id: 2"), TestUtils.json("_id: 3"), TestUtils.json("_id: 5")});
    }

    @Test
    public void testQueryWithReference() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        String collectionName = this.collection.getNamespace().getCollectionName();
        this.collection.insertOne((Object)new Document("_id", (Object)2).append("ref", (Object)new DBRef(collectionName, (Object)1)));
        this.collection.insertOne((Object)new Document("_id", (Object)3).append("ref", (Object)new DBRef(collectionName, (Object)2)));
        Document doc = (Document)this.collection.find((Bson)new Document("ref", (Object)new DBRef(collectionName, (Object)1))).projection((Bson)TestUtils.json("_id: 1")).first();
        Assertions.assertThat((Map)doc).isEqualTo((Object)TestUtils.json("_id: 2"));
    }

    @Test
    public void testQueryWithIllegalReference() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1"));
        String collectionName = this.collection.getNamespace().getCollectionName();
        this.collection.insertOne((Object)new Document("_id", (Object)2).append("ref", (Object)new DBRef(collectionName, (Object)1)));
        this.collection.insertOne((Object)new Document("_id", (Object)3).append("ref", (Object)new DBRef(collectionName, (Object)2)));
        try {
            this.collection.find((Bson)TestUtils.json("ref: {$ref: 'coll'}")).first();
            Assert.fail((String)"MongoQueryException expected");
        }
        catch (MongoQueryException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(10068);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"invalid operator: $ref"});
        }
    }

    @Test
    public void testAndOrNorWithEmptyArray() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("{}"));
        this.assertMongoQueryException(Filters.and((Bson[])new Bson[0]));
        this.assertMongoQueryException(Filters.nor((Bson[])new Bson[0]));
        this.assertMongoQueryException(Filters.or((Bson[])new Bson[0]));
    }

    @Test
    public void testInsertLargeDocument() throws Exception {
        this.insertAndFindLargeDocument(100, 1);
        this.insertAndFindLargeDocument(1000, 2);
        this.insertAndFindLargeDocument(10000, 3);
    }

    @Test
    public void testInsertAndUpdateAsynchronously() throws Exception {
        int numDocuments = 1000;
        final CountDownLatch latch = new CountDownLatch(numDocuments);
        final LinkedBlockingQueue errors = new LinkedBlockingQueue();
        final Semaphore concurrentOperationsOnTheFly = new Semaphore(50);
        for (int i = 1; i <= numDocuments; ++i) {
            final Document document = new Document("_id", (Object)i);
            for (int j = 0; j < 10; ++j) {
                document.append("key-" + i + "-" + j, (Object)("value-" + i + "-" + j));
            }
            concurrentOperationsOnTheFly.acquire();
            this.asyncCollection.insertOne((Object)document, (SingleResultCallback)new SingleResultCallback<Void>(){

                public void onResult(Void result, Throwable t) {
                    this.checkError("insert", t);
                    log.info("inserted {}", (Object)document);
                    final Document query = new Document("_id", (Object)document.getInteger((Object)"_id"));
                    AbstractBackendTest.this.asyncCollection.updateOne((Bson)query, Updates.set((String)"updated", (Object)true), (SingleResultCallback)new SingleResultCallback<UpdateResult>(){

                        public void onResult(UpdateResult result, Throwable t) {
                            this.checkError("update", t);
                            log.info("updated {}: {}", (Object)query, (Object)result);
                            this.release();
                        }
                    });
                }

                private void checkError(String operation, Throwable t) {
                    if (t != null) {
                        log.error(operation + " of {} failed", (Object)document, (Object)t);
                        RuntimeException exception = new RuntimeException("Failed to " + operation + " " + document, t);
                        errors.add(exception);
                        this.release();
                        throw exception;
                    }
                }

                private void release() {
                    latch.countDown();
                    concurrentOperationsOnTheFly.release();
                }
            });
        }
        boolean success = latch.await(30L, TimeUnit.SECONDS);
        Assert.assertTrue((boolean)success);
        if (!errors.isEmpty()) {
            throw (RuntimeException)errors.poll();
        }
        log.info("finished");
        for (int i = 1; i <= numDocuments; ++i) {
            Document query = new Document("_id", (Object)i);
            Document document = (Document)this.collection.find((Bson)query).first();
            ((MapAssert)Assertions.assertThat((Map)document).describedAs(query.toJson(), new Object[0])).isNotNull();
            ((AbstractBooleanAssert)Assertions.assertThat((Boolean)document.getBoolean((Object)"updated")).describedAs(document.toJson(), new Object[0])).isTrue();
        }
        long count = this.collection.countDocuments();
        Assertions.assertThat((long)count).isEqualTo((long)numDocuments);
    }

    @Test
    public void testAllQuery() throws Exception {
        this.collection.insertOne((Object)new Document("_id", (Object)new ObjectId("5234cc89687ea597eabee675")).append("code", (Object)"xyz").append("tags", Arrays.asList("school", "book", "bag", "headphone", "appliance")).append("qty", Arrays.asList(new Document().append("size", (Object)"S").append("num", (Object)10).append("color", (Object)"blue"), new Document().append("size", (Object)"M").append("num", (Object)45).append("color", (Object)"blue"), new Document().append("size", (Object)"L").append("num", (Object)100).append("color", (Object)"green"))));
        this.collection.insertOne((Object)new Document("_id", (Object)new ObjectId("5234cc8a687ea597eabee676")).append("code", (Object)"abc").append("tags", Arrays.asList("appliance", "school", "book")).append("qty", Arrays.asList(new Document().append("size", (Object)"6").append("num", (Object)100).append("color", (Object)"green"), new Document().append("size", (Object)"6").append("num", (Object)50).append("color", (Object)"blue"), new Document().append("size", (Object)"8").append("num", (Object)100).append("color", (Object)"brown"))));
        this.collection.insertOne((Object)new Document("_id", (Object)new ObjectId("5234ccb7687ea597eabee677")).append("code", (Object)"efg").append("tags", Arrays.asList("school", "book")).append("qty", Arrays.asList(new Document().append("size", (Object)"S").append("num", (Object)10).append("color", (Object)"blue"), new Document().append("size", (Object)"M").append("num", (Object)100).append("color", (Object)"blue"), new Document().append("size", (Object)"L").append("num", (Object)100).append("color", (Object)"green"))));
        this.collection.insertOne((Object)new Document("_id", (Object)new ObjectId("52350353b2eff1353b349de9")).append("code", (Object)"ijk").append("tags", Arrays.asList("electronics", "school")).append("qty", Collections.singletonList(new Document().append("size", (Object)"M").append("num", (Object)100).append("color", (Object)"green"))));
        List documents = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("{ tags: { $all: [ \"appliance\", \"school\", \"book\" ] } }")));
        Assertions.assertThat(documents).hasSize(2);
        Assertions.assertThat((Object)((Document)documents.get(0)).get((Object)"_id")).isEqualTo((Object)new ObjectId("5234cc89687ea597eabee675"));
        Assertions.assertThat((Object)((Document)documents.get(1)).get((Object)"_id")).isEqualTo((Object)new ObjectId("5234cc8a687ea597eabee676"));
    }

    @Test
    public void testMatchesElementQuery() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, results: [ 82, 85, 88 ]"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, results: [ 75, 88, 89 ]"));
        List results = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("results: { $elemMatch: { $gte: 80, $lt: 85 } }")));
        Assertions.assertThat(results).hasSize(1);
        Assertions.assertThat((Map)((Map)results.get(0))).isEqualTo((Object)TestUtils.json("\"_id\" : 1, \"results\" : [ 82, 85, 88 ]"));
    }

    @Test
    public void testIllegalElementMatchQuery() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, results: [ 82, 85, 88 ]"));
        try {
            this.collection.find((Bson)TestUtils.json("results: { $elemMatch: [ 85 ] }")).first();
            Assert.fail((String)"MongoQueryException expected");
        }
        catch (MongoQueryException e) {
            Assertions.assertThat((int)e.getErrorCode()).isEqualTo(2);
            Assertions.assertThat((String)e.getErrorMessage()).isEqualTo((Object)"$elemMatch needs an Object");
        }
        try {
            this.collection.find((Bson)TestUtils.json("results: { $elemMatch: 1 }")).first();
            Assert.fail((String)"MongoQueryException expected");
        }
        catch (MongoQueryException e) {
            Assertions.assertThat((int)e.getErrorCode()).isEqualTo(2);
            Assertions.assertThat((String)e.getErrorMessage()).isEqualTo((Object)"$elemMatch needs an Object");
        }
    }

    @Test
    public void testQueryWithComment() throws Exception {
        this.collection.insertOne((Object)TestUtils.json("_id: 1, x: 2"));
        this.collection.insertOne((Object)TestUtils.json("_id: 2, x: 3"));
        this.collection.insertOne((Object)TestUtils.json("_id: 3, x: 4"));
        List documents = TestUtils.toArray(this.collection.find((Bson)TestUtils.json("x: { $mod: [ 2, 0 ] }, $comment: \"Find even values.\"")));
        Assertions.assertThat(documents).hasSize(2);
        Assertions.assertThat((Object)((Document)documents.get(0)).get((Object)"_id")).isEqualTo((Object)1);
        Assertions.assertThat((Object)((Document)documents.get(1)).get((Object)"_id")).isEqualTo((Object)3);
    }

    private void insertAndFindLargeDocument(int numKeyValues, int id) {
        Document document = new Document("_id", (Object)id);
        for (int i = 0; i < numKeyValues; ++i) {
            document.put("key-" + i, (Object)("value-" + i));
        }
        this.collection.insertOne((Object)document);
        Document persistentDocument = (Document)this.collection.find((Bson)new Document("_id", (Object)id)).first();
        Assertions.assertThat((Iterable)persistentDocument.keySet()).hasSize(numKeyValues + 1);
    }

    private void assertMongoQueryException(Bson filter) {
        try {
            this.collection.find(filter).first();
            Assert.fail((String)"MongoQueryException expected");
        }
        catch (MongoQueryException e) {
            Assertions.assertThat((int)e.getCode()).isEqualTo(14816);
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{"nonempty array"});
        }
    }

    private void insertUpdateInBulk(boolean ordered) {
        ArrayList<Object> ops = new ArrayList<Object>();
        ops.add(new InsertOneModel((Object)TestUtils.json("_id: 1, field: 'x'")));
        ops.add(new InsertOneModel((Object)TestUtils.json("_id: 2, field: 'x'")));
        ops.add(new InsertOneModel((Object)TestUtils.json("_id: 3, field: 'x'")));
        ops.add(new UpdateManyModel((Bson)TestUtils.json("field: 'x'"), Updates.set((String)"field", (Object)"y")));
        BulkWriteResult result = this.collection.bulkWrite(ops, new BulkWriteOptions().ordered(ordered));
        Assertions.assertThat((int)result.getInsertedCount()).isEqualTo(3);
        Assertions.assertThat((int)result.getDeletedCount()).isEqualTo(0);
        Assertions.assertThat((int)result.getModifiedCount()).isEqualTo(3);
        Assertions.assertThat((int)result.getMatchedCount()).isEqualTo(3);
        long totalDocuments = this.collection.countDocuments();
        Assertions.assertThat((long)totalDocuments).isEqualTo(3L);
        long documentsWithY = this.collection.countDocuments((Bson)TestUtils.json("field: 'y'"));
        Assertions.assertThat((long)documentsWithY).isEqualTo(3L);
    }

    private void insertUpdateInBulkNoMatch(boolean ordered) {
        this.collection.insertOne((Object)TestUtils.json("foo: 'bar'"));
        ArrayList<UpdateOneModel> ops = new ArrayList<UpdateOneModel>();
        ops.add(new UpdateOneModel(Filters.ne((String)"foo", (Object)OTHER_TEST_DATABASE_NAME), Updates.set((String)"field", (Object)"y")));
        BulkWriteResult result = this.collection.bulkWrite(ops, new BulkWriteOptions().ordered(ordered));
        Assertions.assertThat((int)result.getInsertedCount()).isEqualTo(0);
        Assertions.assertThat((int)result.getDeletedCount()).isEqualTo(0);
        Assertions.assertThat((int)result.getModifiedCount()).isEqualTo(0);
        Assertions.assertThat((int)result.getMatchedCount()).isEqualTo(0);
    }

    private void removeInBulk(boolean ordered) {
        DeleteManyModel deleteOp = new DeleteManyModel((Bson)TestUtils.json("field: 'y'"));
        BulkWriteResult result = this.collection.bulkWrite(Collections.singletonList(deleteOp), new BulkWriteOptions().ordered(ordered));
        Assertions.assertThat((int)result.getDeletedCount()).isEqualTo(3);
        Assertions.assertThat((long)this.collection.countDocuments()).isZero();
    }
}

