/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.backend;

import com.mongodb.BasicDBObject;
import de.bwaldvogel.mongo.MongoBackend;
import de.bwaldvogel.mongo.MongoCollection;
import de.bwaldvogel.mongo.MongoDatabase;
import de.bwaldvogel.mongo.backend.Index;
import de.bwaldvogel.mongo.backend.LimitedList;
import de.bwaldvogel.mongo.backend.Utils;
import de.bwaldvogel.mongo.exception.MongoServerError;
import de.bwaldvogel.mongo.exception.MongoServerException;
import de.bwaldvogel.mongo.exception.MongoSilentServerException;
import de.bwaldvogel.mongo.exception.NoSuchCollectionException;
import de.bwaldvogel.mongo.exception.NoSuchCommandException;
import de.bwaldvogel.mongo.wire.message.MongoDelete;
import de.bwaldvogel.mongo.wire.message.MongoInsert;
import de.bwaldvogel.mongo.wire.message.MongoQuery;
import de.bwaldvogel.mongo.wire.message.MongoUpdate;
import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import org.bson.BSONObject;
import org.bson.BasicBSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractMongoDatabase<KEY>
implements MongoDatabase {
    private static final String NAMESPACES_COLLECTION_NAME = "system.namespaces";
    private static final String INDEXES_COLLECTION_NAME = "system.indexes";
    private static final Logger log = LoggerFactory.getLogger(AbstractMongoDatabase.class);
    protected final String databaseName;
    private final MongoBackend backend;
    private Map<String, MongoCollection<KEY>> collections = new ConcurrentHashMap<String, MongoCollection<KEY>>();
    private MongoCollection<KEY> namespaces;
    private AtomicReference<MongoCollection<KEY>> indexes = new AtomicReference();
    private Map<Channel, List<BSONObject>> lastResults = new ConcurrentHashMap<Channel, List<BSONObject>>();

    protected AbstractMongoDatabase(String databaseName, MongoBackend backend) {
        this.databaseName = databaseName;
        this.backend = backend;
    }

    protected void initializeNamespacesAndIndexes() throws MongoServerException {
        this.namespaces = this.openOrCreateCollection(NAMESPACES_COLLECTION_NAME, "name");
        this.collections.put(this.namespaces.getCollectionName(), this.namespaces);
        if (this.namespaces.count() > 0) {
            for (BSONObject namespace : this.namespaces.handleQuery((BSONObject)new BasicDBObject(), 0, 0, null)) {
                String name = namespace.get("name").toString();
                log.debug("opening {}", (Object)name);
                String collectionName = this.extractCollectionNameFromNamespace(name);
                MongoCollection<KEY> collection = this.openOrCreateCollection(collectionName, "_id");
                this.collections.put(collectionName, collection);
                log.debug("opened collection '{}'", (Object)collectionName);
            }
            MongoCollection<KEY> indexCollection = this.collections.get(INDEXES_COLLECTION_NAME);
            this.indexes.set(indexCollection);
            for (BSONObject indexDescription : indexCollection.handleQuery((BSONObject)new BasicDBObject(), 0, 0, null)) {
                this.openOrCreateIndex(indexDescription);
            }
        }
    }

    @Override
    public final String getDatabaseName() {
        return this.databaseName;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + this.getDatabaseName() + ")";
    }

    private BSONObject commandDropDatabase() throws MongoServerException {
        this.backend.dropDatabase(this.getDatabaseName());
        BasicBSONObject response = new BasicBSONObject("dropped", (Object)this.getDatabaseName());
        Utils.markOkay((BSONObject)response);
        return response;
    }

    @Override
    public BSONObject handleCommand(Channel channel, String command, BSONObject query) throws MongoServerException {
        if (command.equalsIgnoreCase("getlasterror")) {
            return this.commandGetLastError(channel, command, query);
        }
        if (command.equalsIgnoreCase("getpreverror")) {
            return this.commandGetPrevError(channel, command, query);
        }
        if (command.equalsIgnoreCase("reseterror")) {
            return this.commandResetError(channel, command, query);
        }
        this.clearLastStatus(channel);
        if (command.equalsIgnoreCase("insert")) {
            return this.commandInsert(channel, command, query);
        }
        if (command.equalsIgnoreCase("update")) {
            return this.commandUpdate(channel, command, query);
        }
        if (command.equalsIgnoreCase("delete")) {
            return this.commandDelete(channel, command, query);
        }
        if (command.equalsIgnoreCase("create")) {
            return this.commandCreate(channel, command, query);
        }
        if (command.equalsIgnoreCase("createIndexes")) {
            return this.commandCreateIndexes(channel, command, query);
        }
        if (command.equalsIgnoreCase("count")) {
            return this.commandCount(command, query);
        }
        if (command.equalsIgnoreCase("distinct")) {
            String collectionName = query.get(command).toString();
            MongoCollection<KEY> collection = this.resolveCollection(collectionName, true);
            return collection.handleDistinct(query);
        }
        if (command.equalsIgnoreCase("drop")) {
            return this.commandDrop(query);
        }
        if (command.equalsIgnoreCase("dropDatabase")) {
            return this.commandDropDatabase();
        }
        if (command.equalsIgnoreCase("dbstats")) {
            return this.commandDatabaseStats();
        }
        if (command.equalsIgnoreCase("collstats")) {
            String collectionName = query.get(command).toString();
            MongoCollection<KEY> collection = this.resolveCollection(collectionName, true);
            return collection.getStats();
        }
        if (command.equalsIgnoreCase("validate")) {
            String collectionName = query.get(command).toString();
            MongoCollection<KEY> collection = this.resolveCollection(collectionName, true);
            return collection.validate();
        }
        if (command.equalsIgnoreCase("findAndModify")) {
            String collectionName = query.get(command).toString();
            MongoCollection<KEY> collection = this.resolveOrCreateCollection(collectionName);
            return collection.findAndModify(query);
        }
        if (command.equalsIgnoreCase("listCollections")) {
            return this.listCollections();
        }
        if (command.equalsIgnoreCase("listIndexes")) {
            return this.listIndexes();
        }
        log.error("unknown query: {}", (Object)query);
        throw new NoSuchCommandException(command);
    }

    protected BSONObject listCollections() throws MongoServerException {
        BasicBSONObject cursor = new BasicBSONObject();
        cursor.put("id", (Object)0L);
        cursor.put("ns", (Object)(this.getDatabaseName() + ".$cmd.listCollections"));
        ArrayList<BasicBSONObject> firstBatch = new ArrayList<BasicBSONObject>();
        for (BSONObject collection : this.namespaces.handleQuery((BSONObject)new BasicBSONObject(), 0, 0, null)) {
            BasicBSONObject collectionDescription = new BasicBSONObject();
            BasicBSONObject collectionOptions = new BasicBSONObject();
            String namespace = (String)collection.get("name");
            String collectionName = this.extractCollectionNameFromNamespace(namespace);
            collectionDescription.put("name", (Object)collectionName);
            collectionDescription.put("options", (Object)collectionOptions);
            firstBatch.add(collectionDescription);
        }
        cursor.put("firstBatch", firstBatch);
        BasicBSONObject response = new BasicBSONObject();
        response.put("cursor", (Object)cursor);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected BSONObject listIndexes() throws MongoServerException {
        MongoCollection<KEY> indexes = this.resolveCollection(INDEXES_COLLECTION_NAME, true);
        BasicBSONObject cursor = new BasicBSONObject();
        cursor.put("id", (Object)0L);
        cursor.put("ns", (Object)(this.getDatabaseName() + ".$cmd.listIndexes"));
        ArrayList<BSONObject> firstBatch = new ArrayList<BSONObject>();
        for (BSONObject description : indexes.handleQuery((BSONObject)new BasicBSONObject(), 0, 0, null)) {
            firstBatch.add(description);
        }
        cursor.put("firstBatch", firstBatch);
        BasicBSONObject response = new BasicBSONObject();
        response.put("cursor", (Object)cursor);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected MongoCollection<KEY> resolveOrCreateCollection(String collectionName) throws MongoServerException {
        MongoCollection<KEY> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            return collection;
        }
        return this.createCollection(collectionName);
    }

    protected BSONObject commandInsert(Channel channel, String command, BSONObject query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List documents = (List)query.get("documents");
        ArrayList<BasicBSONObject> writeErrors = new ArrayList<BasicBSONObject>();
        int n = 0;
        for (BSONObject document : documents) {
            try {
                this.insertDocuments(channel, collectionName, Arrays.asList(document));
                ++n;
            }
            catch (MongoServerError e) {
                BasicBSONObject error = new BasicBSONObject();
                error.put("index", (Object)n);
                error.put("code", (Object)e.getCode());
                error.put("errmsg", (Object)e.getMessage());
                writeErrors.add(error);
            }
        }
        BasicBSONObject result = new BasicBSONObject();
        result.put("n", (Object)n);
        if (!writeErrors.isEmpty()) {
            result.put("writeErrors", writeErrors);
        }
        Utils.markOkay((BSONObject)result);
        return result;
    }

    protected BSONObject commandUpdate(Channel channel, String command, BSONObject query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List updates = (List)query.get("updates");
        int n = 0;
        boolean updatedExisting = false;
        ArrayList<BasicBSONObject> upserts = new ArrayList<BasicBSONObject>();
        for (BSONObject updateObj : updates) {
            BSONObject selector = (BSONObject)updateObj.get("q");
            BSONObject update = (BSONObject)updateObj.get("u");
            boolean multi = Utils.isTrue(updateObj.get("multi"));
            boolean upsert = Utils.isTrue(updateObj.get("upsert"));
            BSONObject result = this.updateDocuments(channel, collectionName, selector, update, multi, upsert);
            updatedExisting |= Utils.isTrue(result.get("updatedExisting"));
            if (result.containsField("upserted")) {
                Object id = result.get("upserted");
                BasicBSONObject upserted = new BasicBSONObject("index", (Object)upserts.size());
                upserted.put("_id", id);
                upserts.add(upserted);
            }
            Integer resultNumber = (Integer)result.get("n");
            n += resultNumber.intValue();
        }
        BasicBSONObject response = new BasicBSONObject("n", (Object)n);
        response.put("updatedExisting", (Object)updatedExisting);
        if (!upserts.isEmpty()) {
            response.put("upserted", upserts);
        }
        Utils.markOkay((BSONObject)response);
        this.putLastResult(channel, (BSONObject)response);
        return response;
    }

    protected BSONObject commandDelete(Channel channel, String command, BSONObject query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isOrdered = Utils.isTrue(query.get("ordered"));
        log.trace("ordered: {}", (Object)isOrdered);
        List deletes = (List)query.get("deletes");
        int n = 0;
        for (BSONObject delete : deletes) {
            BSONObject selector = (BSONObject)delete.get("q");
            int limit = ((Number)delete.get("limit")).intValue();
            BSONObject result = this.deleteDocuments(channel, collectionName, selector, limit);
            Integer resultNumber = (Integer)result.get("n");
            n += resultNumber.intValue();
        }
        BasicBSONObject response = new BasicBSONObject("n", (Object)n);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected BSONObject commandCreate(Channel channel, String command, BSONObject query) throws MongoServerException {
        String collectionName = query.get(command).toString();
        boolean isCapped = Utils.isTrue(query.get("capped"));
        if (isCapped) {
            throw new MongoServerException("Creating capped collections is not yet implemented");
        }
        Object autoIndexId = query.get("autoIndexId");
        if (autoIndexId != null && !Utils.isTrue(autoIndexId)) {
            throw new MongoServerException("Disabling autoIndexId is not yet implemented");
        }
        MongoCollection<KEY> collection = this.resolveCollection(collectionName, false);
        if (collection != null) {
            throw new MongoServerError(48, "collection already exists");
        }
        this.createCollection(collectionName);
        BasicBSONObject response = new BasicBSONObject();
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected BSONObject commandCreateIndexes(Channel channel, String command, BSONObject query) throws MongoServerException {
        int indexesBefore = this.countIndexes();
        Collection indexDescriptions = (Collection)query.get("indexes");
        for (BSONObject indexDescription : indexDescriptions) {
            this.addIndex(indexDescription);
        }
        int indexesAfter = this.countIndexes();
        BasicBSONObject response = new BasicBSONObject();
        response.put("numIndexesBefore", (Object)indexesBefore);
        response.put("numIndexesAfter", (Object)indexesAfter);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int countIndexes() {
        MongoCollection<KEY> indexesCollection;
        AtomicReference<MongoCollection<KEY>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            indexesCollection = this.indexes.get();
        }
        if (indexesCollection == null) {
            return 0;
        }
        return indexesCollection.count();
    }

    protected BSONObject commandDatabaseStats() throws MongoServerException {
        BasicBSONObject response = new BasicBSONObject("db", (Object)this.getDatabaseName());
        response.put("collections", (Object)this.namespaces.count());
        long storageSize = this.getStorageSize();
        long fileSize = this.getFileSize();
        long indexSize = 0L;
        long objects = 0L;
        long dataSize = 0L;
        double averageObjectSize = 0.0;
        for (MongoCollection<KEY> collection : this.collections.values()) {
            BSONObject stats = collection.getStats();
            objects += ((Number)stats.get("count")).longValue();
            dataSize += ((Number)stats.get("size")).longValue();
            BSONObject indexSizes = (BSONObject)stats.get("indexSize");
            for (String indexName : indexSizes.keySet()) {
                indexSize += ((Number)indexSizes.get(indexName)).longValue();
            }
        }
        if (objects > 0L) {
            averageObjectSize = (double)dataSize / (double)objects;
        }
        response.put("objects", (Object)objects);
        response.put("avgObjSize", (Object)averageObjectSize);
        response.put("dataSize", (Object)dataSize);
        response.put("storageSize", (Object)storageSize);
        response.put("numExtents", (Object)0);
        response.put("indexes", (Object)this.countIndexes());
        response.put("indexSize", (Object)indexSize);
        response.put("fileSize", (Object)fileSize);
        response.put("nsSizeMB", (Object)0);
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected abstract long getFileSize();

    protected abstract long getStorageSize();

    protected BSONObject commandDrop(BSONObject query) throws MongoServerException {
        String collectionName = query.get("drop").toString();
        MongoCollection<KEY> collection = this.collections.remove(collectionName);
        if (collection == null) {
            throw new MongoSilentServerException("ns not found");
        }
        BasicBSONObject response = new BasicBSONObject();
        this.namespaces.removeDocument((BSONObject)new BasicBSONObject("name", (Object)collection.getFullName()));
        response.put("nIndexesWas", (Object)collection.getNumIndexes());
        response.put("ns", (Object)collection.getFullName());
        Utils.markOkay((BSONObject)response);
        return response;
    }

    protected BSONObject commandGetLastError(Channel channel, String command, BSONObject query) throws MongoServerException {
        String subCommand;
        Iterator it = query.keySet().iterator();
        String cmd = (String)it.next();
        if (!cmd.equals(command)) {
            throw new IllegalStateException();
        }
        if (it.hasNext() && !(subCommand = (String)it.next()).equals("w") && !subCommand.equals("fsync")) {
            throw new MongoServerException("unknown subcommand: " + subCommand);
        }
        List<BSONObject> results = this.lastResults.get(channel);
        BasicBSONObject result = null;
        if (results != null && !results.isEmpty()) {
            result = results.get(results.size() - 1);
            if (result == null) {
                result = new BasicBSONObject();
            }
        } else {
            result = new BasicBSONObject();
            result.put("err", null);
        }
        Utils.markOkay((BSONObject)result);
        return result;
    }

    protected BSONObject commandGetPrevError(Channel channel, String command, BSONObject query) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results != null) {
            for (int i = 1; i < results.size(); ++i) {
                BSONObject result = results.get(results.size() - i);
                if (result == null) continue;
                boolean isRelevant = false;
                if (result.get("err") != null) {
                    isRelevant = true;
                } else if (((Number)result.get("n")).intValue() > 0) {
                    isRelevant = true;
                }
                if (!isRelevant) continue;
                result.put("nPrev", (Object)i);
                return result;
            }
        }
        BasicBSONObject result = new BasicBSONObject();
        result.put("nPrev", (Object)-1);
        Utils.markOkay((BSONObject)result);
        return result;
    }

    protected BSONObject commandResetError(Channel channel, String command, BSONObject query) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results != null) {
            results.clear();
        }
        BasicBSONObject result = new BasicBSONObject();
        Utils.markOkay((BSONObject)result);
        return result;
    }

    protected BSONObject commandCount(String command, BSONObject query) throws MongoServerException {
        String collection = query.get(command).toString();
        BasicBSONObject response = new BasicBSONObject();
        MongoCollection<KEY> coll = this.collections.get(collection);
        if (coll == null) {
            response.put("missing", (Object)Boolean.TRUE);
            response.put("n", (Object)0);
        } else {
            response.put("n", (Object)coll.count((BSONObject)query.get("query")));
        }
        Utils.markOkay((BSONObject)response);
        return response;
    }

    @Override
    public Iterable<BSONObject> handleQuery(MongoQuery query) throws MongoServerException {
        this.clearLastStatus(query.getChannel());
        String collectionName = query.getCollectionName();
        MongoCollection<KEY> collection = this.resolveCollection(collectionName, false);
        if (collection == null) {
            return Collections.emptyList();
        }
        return collection.handleQuery(query.getQuery(), query.getNumberToSkip(), query.getNumberToReturn(), query.getReturnFieldSelector());
    }

    @Override
    public void handleClose(Channel channel) {
        this.lastResults.remove(channel);
    }

    public synchronized void clearLastStatus(Channel channel) {
        List<BSONObject> results = this.lastResults.get(channel);
        if (results == null) {
            results = new LimitedList<BSONObject>(10);
            this.lastResults.put(channel, results);
        }
        results.add(null);
    }

    @Override
    public void handleInsert(MongoInsert insert) throws MongoServerException {
        Channel channel = insert.getChannel();
        String collectionName = insert.getCollectionName();
        List<BSONObject> documents = insert.getDocuments();
        if (collectionName.equals(INDEXES_COLLECTION_NAME)) {
            for (BSONObject indexDescription : documents) {
                this.addIndex(indexDescription);
            }
        } else {
            try {
                this.insertDocuments(channel, collectionName, documents);
            }
            catch (MongoServerException e) {
                log.error("failed to insert {}", (Object)insert, (Object)e);
            }
        }
    }

    public synchronized MongoCollection<KEY> resolveCollection(String collectionName, boolean throwIfNotFound) throws MongoServerException {
        this.checkCollectionName(collectionName);
        MongoCollection<KEY> collection = this.collections.get(collectionName);
        if (collection == null && throwIfNotFound) {
            throw new NoSuchCollectionException(collectionName);
        }
        return collection;
    }

    protected void checkCollectionName(String collectionName) throws MongoServerException {
        if (collectionName.length() > 128) {
            throw new MongoServerError(10080, "ns name too long, max size is 128");
        }
        if (collectionName.isEmpty()) {
            throw new MongoServerError(16256, "Invalid ns [" + collectionName + "]");
        }
    }

    @Override
    public boolean isEmpty() {
        return this.collections.isEmpty();
    }

    protected void addNamespace(MongoCollection<KEY> collection) throws MongoServerException {
        this.collections.put(collection.getCollectionName(), collection);
        this.namespaces.addDocument((BSONObject)new BasicDBObject("name", (Object)collection.getFullName()));
    }

    @Override
    public void handleDelete(MongoDelete delete) throws MongoServerException {
        Channel channel = delete.getChannel();
        String collectionName = delete.getCollectionName();
        BSONObject selector = delete.getSelector();
        int limit = delete.isSingleRemove() ? 1 : Integer.MAX_VALUE;
        try {
            this.deleteDocuments(channel, collectionName, selector, limit);
        }
        catch (MongoServerException e) {
            log.error("failed to delete {}", (Object)delete, (Object)e);
        }
    }

    @Override
    public void handleUpdate(MongoUpdate updateCommand) throws MongoServerException {
        Channel channel = updateCommand.getChannel();
        String collectionName = updateCommand.getCollectionName();
        BSONObject selector = updateCommand.getSelector();
        BSONObject update = updateCommand.getUpdate();
        boolean multi = updateCommand.isMulti();
        boolean upsert = updateCommand.isUpsert();
        try {
            BSONObject result = this.updateDocuments(channel, collectionName, selector, update, multi, upsert);
            this.putLastResult(channel, result);
        }
        catch (MongoServerException e) {
            log.error("failed to update {}", (Object)updateCommand, (Object)e);
        }
    }

    protected void addIndex(BSONObject indexDescription) throws MongoServerException {
        this.openOrCreateIndex(indexDescription);
        this.getOrCreateIndexesCollection().addDocument(indexDescription);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private MongoCollection<KEY> getOrCreateIndexesCollection() throws MongoServerException {
        AtomicReference<MongoCollection<KEY>> atomicReference = this.indexes;
        synchronized (atomicReference) {
            if (this.indexes.get() == null) {
                MongoCollection<KEY> indexCollection = this.openOrCreateCollection(INDEXES_COLLECTION_NAME, null);
                this.addNamespace(indexCollection);
                this.indexes.set(indexCollection);
            }
            return this.indexes.get();
        }
    }

    private String extractCollectionNameFromNamespace(String namespace) {
        if (!namespace.startsWith(this.databaseName)) {
            throw new IllegalArgumentException();
        }
        String collectionName = namespace.substring(this.databaseName.length() + 1);
        return collectionName;
    }

    private void openOrCreateIndex(BSONObject indexDescription) throws MongoServerException {
        String ns = indexDescription.get("ns").toString();
        String collectionName = this.extractCollectionNameFromNamespace(ns);
        MongoCollection<KEY> collection = this.resolveOrCreateCollection(collectionName);
        BSONObject key = (BSONObject)indexDescription.get("key");
        if (key.keySet().equals(Collections.singleton("_id"))) {
            boolean ascending = Utils.normalizeValue(key.get("_id")).equals(1.0);
            collection.addIndex(this.openOrCreateUniqueIndex(collectionName, "_id", ascending));
            log.info("adding unique _id index for collection {}", (Object)collectionName);
        } else if (Utils.isTrue(indexDescription.get("unique"))) {
            if (key.keySet().size() != 1) {
                throw new MongoServerException("Compound unique indices are not yet implemented");
            }
            log.info("adding unique index {} for collection {}", (Object)key.keySet(), (Object)collectionName);
            String field = (String)key.keySet().iterator().next();
            boolean ascending = Utils.normalizeValue(key.get(field)).equals(1.0);
            collection.addIndex(this.openOrCreateUniqueIndex(collectionName, field, ascending));
        } else {
            log.warn("adding non-unique non-id index with key {} is not yet implemented", (Object)key);
        }
    }

    protected abstract Index<KEY> openOrCreateUniqueIndex(String var1, String var2, boolean var3);

    protected BSONObject insertDocuments(Channel channel, String collectionName, List<BSONObject> documents) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(16459, "attempt to insert in system namespace");
            }
            MongoCollection<KEY> collection = this.resolveOrCreateCollection(collectionName);
            int n = collection.insertDocuments(documents);
            assert (n == documents.size());
            BasicBSONObject result = new BasicBSONObject("n", (Object)n);
            this.putLastResult(channel, (BSONObject)result);
            return result;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    protected BSONObject deleteDocuments(Channel channel, String collectionName, BSONObject selector, int limit) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(12050, "cannot delete from system namespace");
            }
            MongoCollection<KEY> collection = this.resolveCollection(collectionName, false);
            int n = collection == null ? 0 : collection.deleteDocuments(selector, limit);
            BasicBSONObject result = new BasicBSONObject("n", (Object)n);
            this.putLastResult(channel, (BSONObject)result);
            return result;
        }
        catch (MongoServerError e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    protected BSONObject updateDocuments(Channel channel, String collectionName, BSONObject selector, BSONObject update, boolean multi, boolean upsert) throws MongoServerException {
        this.clearLastStatus(channel);
        try {
            if (collectionName.startsWith("system.")) {
                throw new MongoServerError(10156, "cannot update system collection");
            }
            MongoCollection<KEY> collection = this.resolveOrCreateCollection(collectionName);
            return collection.updateDocuments(selector, update, multi, upsert);
        }
        catch (MongoServerException e) {
            this.putLastError(channel, e);
            throw e;
        }
    }

    protected void putLastError(Channel channel, MongoServerException ex) {
        BasicBSONObject error = new BasicBSONObject();
        if (ex instanceof MongoServerError) {
            MongoServerError err = (MongoServerError)ex;
            error.put("err", (Object)err.getMessage());
            error.put("code", (Object)err.getCode());
        } else {
            error.put("err", (Object)ex.getMessage());
        }
        error.put("connectionId", (Object)channel.hashCode());
        this.putLastResult(channel, (BSONObject)error);
    }

    protected synchronized void putLastResult(Channel channel, BSONObject result) {
        List<BSONObject> results = this.lastResults.get(channel);
        BSONObject last = results.get(results.size() - 1);
        if (last != null) {
            throw new IllegalStateException("last result already set: " + last);
        }
        results.set(results.size() - 1, result);
    }

    protected MongoCollection<KEY> createCollection(String collectionName) throws MongoServerException {
        this.checkCollectionName(collectionName);
        if (collectionName.contains("$")) {
            throw new MongoServerError(10093, "cannot insert into reserved $ collection");
        }
        MongoCollection<KEY> collection = this.openOrCreateCollection(collectionName, "_id");
        this.addNamespace(collection);
        BasicBSONObject indexDescription = new BasicBSONObject();
        indexDescription.put("name", (Object)"_id_");
        indexDescription.put("ns", (Object)collection.getFullName());
        indexDescription.put("key", (Object)new BasicBSONObject("_id", (Object)1));
        this.addIndex((BSONObject)indexDescription);
        log.info("created collection {}", (Object)collection.getFullName());
        return collection;
    }

    protected abstract MongoCollection<KEY> openOrCreateCollection(String var1, String var2);

    @Override
    public void drop() throws MongoServerException {
        log.debug("dropping {}", (Object)this);
        for (String collectionName : this.collections.keySet()) {
            this.dropCollection(collectionName);
        }
    }

    @Override
    public void dropCollection(String collectionName) throws MongoServerException {
        this.deregisterCollection(collectionName);
    }

    public MongoCollection<KEY> deregisterCollection(String collectionName) throws MongoServerException {
        MongoCollection<KEY> removedCollection = this.collections.remove(collectionName);
        this.namespaces.deleteDocuments((BSONObject)new BasicBSONObject("name", (Object)removedCollection.getFullName()), 1);
        return removedCollection;
    }

    @Override
    public void moveCollection(MongoDatabase oldDatabase, MongoCollection<?> collection, String newCollectionName) throws MongoServerException {
        oldDatabase.deregisterCollection(collection.getCollectionName());
        collection.renameTo(this.getDatabaseName(), newCollectionName);
        MongoCollection<?> newCollection = collection;
        this.collections.put(newCollectionName, newCollection);
        ArrayList<BSONObject> newDocuments = new ArrayList<BSONObject>();
        newDocuments.add((BSONObject)new BasicBSONObject("name", (Object)collection.getFullName()));
        this.namespaces.insertDocuments(newDocuments);
    }
}

