/**
 * 
 */
package de.boreddevblog.license;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.StringJoiner;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import lombok.NonNull;

/**
 * @author Tim Satke
 *
 */
public abstract class License {

	private static final String SIGNATURE_ALGORITHM = "SHA512withRSA";

	public static final String RESERVED_NAME_KEY = "key";

	private static final String RESERVED_NAMES[] = { RESERVED_NAME_KEY };
	private static final Set<String> reservedNames = new HashSet<>(Arrays.asList(RESERVED_NAMES));

	private final Map<String, LicensePropertyDescriptor> propertyDescriptors = new HashMap<>();
	private final Map<String, Object> properties = new HashMap<>();

	private final Supplier<KeyPairProvider> keyPairProviderSupplier = () -> {
		final KeyPairProvider keyPairProvider = getKeyPairProvider();

		if (keyPairProvider == null) {
			throw new NullPointerException("No KeyPairProvider defined");
		}

		if (keyPairProvider.getKeyPair() == null) {
			throw new NullPointerException("KeyPairProvider does not provide a KeyPair");
		}

		return keyPairProvider;
	};

	public License() {
		addPropertyDescriptor0(new LicensePropertyDescriptor(RESERVED_NAME_KEY, String.class));
	}

	protected abstract KeyPairProvider getKeyPairProvider();

	protected final void addPropertyDescriptor(@NonNull LicensePropertyDescriptor desc) {
		ensureNotReserved(desc.getPropertyName());
		addPropertyDescriptor0(desc);
	}

	private void addPropertyDescriptor0(@NonNull LicensePropertyDescriptor desc) {
		if (propertyDescriptors.get(desc.getPropertyName()) != null) {
			throw new DescriptorAlreadyExistsException(
					"A descriptor for property '" + desc.getPropertyName() + "' already exists");
		}
		propertyDescriptors.put(desc.getPropertyName(), desc);
	}

	public final void addProperty(@NonNull String key, Object value) {
		ensureNotReserved(key);
		addProperty0(key, value);
	}

	protected final void setKey(@NonNull String key) {
		addProperty0(RESERVED_NAME_KEY, key);
	}

	private final void addProperty0(@NonNull String key, Object value) {
		LicensePropertyDescriptor desc = propertyDescriptors.get(key);

		// no descriptor registered for property
		if (desc == null) {
			throw new NoDescriptorException("No descriptor for property " + key);
		}

		// value is not of type that is required by descriptor
		if (!desc.getType().isAssignableFrom(value.getClass())) {
			if (!Mapper.canMap(value.getClass(), desc.getType())) {
				throw new DescriptorTypeMismatchException("'" + key + "' is of type " + value.getClass().getName() + //
						", expected " + desc.getType().getName());
			} else {
				properties.put(key, Mapper.mapToClass(value, desc.getType()));
			}
		} else {
			// overwrite if necessary
			properties.put(key, value);
		}
	}

	// Derivates MUST implement toString, since it is used to save the license
	public abstract String toString();

	public abstract void load(@NonNull File file) throws IOException;

	public final void sign() throws NoSuchAlgorithmException, SignatureException, InvalidKeyException {
		properties.keySet().stream().map(propertyDescriptors::get).filter(LicensePropertyDescriptor::isDeprecated)
				.map(LicensePropertyDescriptor::getPropertyName).forEach(name -> {
					System.err.println("[WARN] '" + name
							+ "' is deprecated and should not be used. It will be removed in a future build.");
				});

		// actually sign
		final KeyPairProvider keyPairProvider = keyPairProviderSupplier.get();

		KeyPair keyPair = keyPairProvider.getKeyPair();
		if (keyPair.getPrivate() == null) {
			throw new NullPointerException("KeyPairProvider does not provide a private key to sign the license");
		}

		Signature sig = Signature.getInstance(SIGNATURE_ALGORITHM);
		sig.initSign(keyPair.getPrivate());
		sig.update(getSignableString().getBytes(Charset.forName("UTF-8")));

		setKey(Base64.getEncoder().encodeToString(sig.sign()));
	}

	private String getSignableString() {
		List<Map.Entry<String, Object>> sorted = getPropertiesSorted();
		StringJoiner sj = new StringJoiner("#");
		sorted.forEach(e -> {
			if (e.getValue().getClass().isArray()) {
				sj.add(e.getKey() + "=" + Arrays.deepToString((Object[]) e.getValue()));
			} else {
				sj.add(e.getKey() + "=" + e.getValue());
			}
		});

		return sj.toString();
	}

	private List<Entry<String, Object>> getPropertiesSorted() {
		return properties.entrySet().stream() //
				// do not consider the signature key
				.filter(e -> e.getKey() != RESERVED_NAME_KEY) //
				.sorted((fst, snd) -> {
					return fst.getKey().compareTo(snd.getKey());
				}) //
				.collect(Collectors.toList());
	}

	public final void verify()
			throws VerificationException, NoSuchAlgorithmException, InvalidKeyException, SignatureException {
		// required properties verification
		final Set<String> required = propertyDescriptors.values().stream().filter(desc -> {
			return desc.isRequired() || (desc.getRequiredIf() != null ? desc.getRequiredIf().test(this) : false);
		}).map(LicensePropertyDescriptor::getPropertyName).collect(Collectors.toSet());
		properties.keySet().forEach(required::remove);
		if (!required.isEmpty()) {
			throw new VerificationException("Missing required properties: " + required);
		}

		// key verification

		final KeyPairProvider keyPairProvider = keyPairProviderSupplier.get();

		KeyPair keyPair = keyPairProvider.getKeyPair();
		if (keyPair.getPublic() == null) {
			throw new NullPointerException("KeyPairProvider does not provide a public key to verify the license");
		}

		String key = (String) properties.get(RESERVED_NAME_KEY);
		if (key == null) {
			throw new VerificationException("No key is present. The license has not been signed yet.");
		}

		Signature sig = Signature.getInstance(SIGNATURE_ALGORITHM);
		sig.initVerify(keyPair.getPublic());
		sig.update(getSignableString().getBytes(Charset.forName("UTF-8")));
		boolean isValid = sig.verify(Base64.getDecoder().decode(key));
		if (!isValid) {
			throw new VerificationException("License key is invalid");
		}
	}

	public void saveTo(@NonNull OutputStream destination) throws IOException {
		destination.write(toString().getBytes(Charset.forName("UTF-8")));
	}

	private final void ensureNotReserved(@NonNull String key) {
		if (reservedNames.contains(key)) {
			throw new ReservedNameException(key + " is a reserved property name");
		}
	}

	public Map<String, Object> getProperties() {
		return Collections.unmodifiableMap(properties);
	}

	public Map<String, LicensePropertyDescriptor> getPropertyDescriptors() {
		return Collections.unmodifiableMap(propertyDescriptors);
	}

	@SuppressWarnings("unchecked")
	public <T> T get(@NonNull String key) {
		return (T) properties.get(key);
	}

}
