/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.reflection

import de.bixilon.kutil.cast.CastUtil.unsafeCast
import de.bixilon.kutil.exception.ExceptionUtil.ignoreAll
import de.bixilon.kutil.observer.ObserveUtil.jOwner
import de.bixilon.kutil.unsafe.UnsafeUtil.setUnsafeAccessible
import java.lang.reflect.Field
import kotlin.jvm.internal.CallableReference
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty0
import kotlin.reflect.jvm.javaField

object ReflectionUtil {
    private val ANY = Any::class.java
    private val PRIVATE_GET_DECLARED_FIELDS = ignoreAll { Class::class.java.getDeclaredMethod("privateGetDeclaredFields", Boolean::class.java).apply { setUnsafeAccessible() } }

    fun Field.forceSet(instance: Any, value: Any?) {
        if (!isAccessible) {
            this.setUnsafeAccessible()
        }

        this.set(instance, value)
    }


    val Class<*>.realName: String
        get() = this.name.removePrefix(this.packageName).removePrefix(".")


    fun Class<*>.forceInit() {
        Class.forName(this.name, true, this.classLoader)
    }

    fun <T> KProperty0<T>.forceSet(value: T) {
        if (this !is CallableReference) throw IllegalStateException("Not a CallableReference")
        val field = this.jvmField
        if (!field.isAccessible) {
            field.setUnsafeAccessible()
        }

        field.set(boundReceiver, value)
    }

    val KProperty<*>.jvmField: Field
        get() {
            if (this !is CallableReference) throw IllegalStateException("Not a CallableReference")
            val field = jOwner.getFieldOrNull(this.name) ?: javaField ?: throw NoSuchFieldException("This property has no backing field!")

            if (!field.isAccessible) {
                field.setUnsafeAccessible()
            }

            return field
        }

    fun Class<*>.getUnsafeDeclaredFields(): Array<Field> {
        ignoreAll { PRIVATE_GET_DECLARED_FIELDS?.invoke(this, false)?.unsafeCast<Array<Field>>()?.let { return it } }

        return declaredFields
    }


    fun Class<*>.getFieldOrNull(name: String, supertypes: Boolean = true): Field? {
        val fields = getUnsafeDeclaredFields()
        val field = fields.find { it.name == name }
        if (field != null) {
            field.setUnsafeAccessible()
            return field
        }

        if (!supertypes) return null

        val superclass = superclass
        if (superclass == ANY) return null
        return superclass.getFieldOrNull(name, true)
    }
}
