/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.queue

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.time.TimeUtil.millis

class Queue(initialCapacity: Int = 100) {
    private val queue: MutableList<Runnable> = ArrayList(initialCapacity)
    private var lock = SimpleLock()

    val size: Int
        get() = queue.size

    fun add(runnable: Runnable) {
        lock.lock()
        queue += runnable
        lock.unlock()
    }

    operator fun plusAssign(runnable: Runnable) {
        add(runnable)
    }

    private fun runFirst() {
        val item = queue.removeAt(0)
        lock.unlock()
        try {
            item.run()
        } catch (exception: Throwable) {
            exception.printStackTrace()
        }
    }

    fun work(maxJobs: Int = Int.MAX_VALUE) {
        lock.lock()
        var jobsDone = 0
        var cachedSize = queue.size
        while (cachedSize-- > 0 || queue.size > 0) {
            runFirst()
            lock.lock()

            if (++jobsDone == maxJobs) {
                break
            }
        }
        lock.unlock()
    }

    fun timeWork(time: Long) {
        check(time > 0L) { "Can not have <= 0 time to do jobs!" }
        val start = millis()
        lock.lock()

        var cachedSize = queue.size
        while (cachedSize-- > 0 || queue.size > 0) {
            runFirst()
            lock.lock()

            if (millis() - start >= time) {
                break
            }
        }
        lock.unlock()
    }

    fun clear() {
        lock.lock()
        queue.clear()
        lock.unlock()
    }
}
