/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.lock.thread

import de.bixilon.kutil.concurrent.lock.Lock
import de.bixilon.kutil.concurrent.lock.NobodyIsReadingException
import de.bixilon.kutil.concurrent.lock.NobodyIsWritingException

class ThreadLock : Lock {
    private val lock = Object()
    override var readers: Int = 0
        private set
    private var thread: Thread? = null
    private var writers = 0
    override val locked: Boolean get() = writers > 0 && thread != null

    fun isOwner(): Boolean {
        return thread == Thread.currentThread()
    }

    override fun acquire() {
        val thread = Thread.currentThread()
        synchronized(lock) {
            while (locked && this.thread !== thread) {
                lock.wait()
            }
            readers++
            lock.notifyAll()
        }
    }

    override fun release() {
        synchronized(lock) {
            if (readers <= 0) throw NobodyIsReadingException()
            readers--
            lock.notifyAll()
        }
    }

    override fun lock() {
        val thread = Thread.currentThread()
        synchronized(lock) {
            while (readers > 0) {
                lock.wait()
            }
            if (this.thread !== thread) {
                while (writers > 0) {
                    lock.wait()
                }
                this.thread = thread
            }

            writers++
            lock.notifyAll()
        }
    }

    override fun unlock() {
        val thread = this.thread
        val currentThread = Thread.currentThread()
        synchronized(lock) {
            if (writers <= 0 || thread == null) throw NobodyIsWritingException()
            if (this.thread !== currentThread) throw ThreadMissmatchException(thread, currentThread)
            writers--
            if (writers <= 0) {
                this.thread = null
            }
            lock.notifyAll()
        }
    }
}
