/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.pool

import de.bixilon.kutil.collections.CollectionUtil.synchronizedListOf
import de.bixilon.kutil.collections.CollectionUtil.toSynchronizedList
import de.bixilon.kutil.concurrent.pool.runnable.InterruptableRunnable
import de.bixilon.kutil.concurrent.pool.runnable.SimplePoolRunnable
import de.bixilon.kutil.concurrent.pool.runnable.ThreadPoolRunnable
import java.util.concurrent.*
import java.util.concurrent.atomic.AtomicInteger

open class ThreadPool(
    val threadCount: Int = Runtime.getRuntime().availableProcessors(),
    private val name: String = "Worker#%d",
    val priority: Int = Thread.MIN_PRIORITY,
) : ExecutorService {
    private var running = AtomicInteger()
    private var state = ThreadPoolStates.STARTING
    private var threads: MutableList<Thread> = synchronizedListOf()
    private var queue: PriorityBlockingQueue<ThreadPoolRunnable> = PriorityBlockingQueue()
    private var nextThreadId = 0

    init {
        check(threadCount >= 1) { "Can not have < 1 thread!" }
        checkThreads()
        state = ThreadPoolStates.STARTED
    }

    val queueSize: Int
        get() = queue.size

    private fun _run(runnable: ThreadPoolRunnable) {
        runnable.runnable?.run()
    }

    private fun _run(runnable: InterruptableRunnable) {
        try {
            runnable.thread = Thread.currentThread()
            runnable.runnable?.run()
            runnable.thread = null
        } catch (exception: InterruptedException) {
            runnable.interrupted = true
            queue += runnable
        } finally {
            runnable.thread = null
        }
    }

    private fun run(runnable: ThreadPoolRunnable) {
        try {
            running.incrementAndGet()
            if (runnable is InterruptableRunnable) {
                _run(runnable)
            } else {
                _run(runnable)
            }
        } catch (exception: Throwable) {
            exception.printStackTrace()
        } finally {
            running.decrementAndGet()
        }
    }

    private fun startThreadLoop() {
        var runnable: ThreadPoolRunnable
        while (true) {
            if (state == ThreadPoolStates.STOPPING) {
                break
            }
            try {
                runnable = queue.take()
            } catch (exception: InterruptedException) {
                continue
            }
            run(runnable)

            if (state == ThreadPoolStates.STOPPING) {
                break
            }
        }
    }

    @Synchronized
    private fun checkThreads() {
        for (i in 0 until threadCount - threads.size) {
            val thread = Thread {
                try {
                    startThreadLoop()
                } catch (exception: Exception) {
                    checkThreads()
                    throw exception
                } finally {
                    threads -= Thread.currentThread()
                }
            }
            thread.name = name.replace("%d", (nextThreadId++).toString())
            thread.priority = priority
            thread.start()
            threads += thread
        }
    }

    fun execute(runnable: ThreadPoolRunnable) {
        if (!runnable.forcePool && (isBusy() || isBacklog())) {
            run(runnable)
            return
        }
        queue += runnable
    }

    override fun execute(runnable: Runnable) {
        execute(SimplePoolRunnable(runnable = runnable))
    }

    operator fun plusAssign(runnable: ThreadPoolRunnable) = execute(runnable)
    operator fun plusAssign(runnable: Runnable) = execute(runnable)

    override fun shutdown() {
        state = ThreadPoolStates.STOPPING
        synchronized(threads) {
            for (thread in threads.toSynchronizedList()) {
                thread.interrupt()
            }
        }
        while (threads.isNotEmpty()) {
            Thread.sleep(1L)
        }
        state = ThreadPoolStates.STOPPED
    }

    override fun shutdownNow(): MutableList<Runnable> {
        state = ThreadPoolStates.STOPPING
        synchronized(threads) {
            for (thread in threads.toSynchronizedList()) {
                thread.interrupt()
            }
        }
        state = ThreadPoolStates.STOPPED
        return mutableListOf()
    }

    override fun isShutdown(): Boolean {
        return state == ThreadPoolStates.STOPPING
    }

    override fun isTerminated(): Boolean {
        return state == ThreadPoolStates.STOPPED
    }

    override fun awaitTermination(p0: Long, p1: TimeUnit): Boolean {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> submit(p0: Callable<T>): Future<T> {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> submit(p0: Runnable, p1: T): Future<T> {
        TODO("Not yet implemented")
    }

    override fun submit(p0: Runnable): Future<*> {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> invokeAll(p0: MutableCollection<out Callable<T>>): MutableList<Future<T>> {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> invokeAll(p0: MutableCollection<out Callable<T>>, p1: Long, p2: TimeUnit): MutableList<Future<T>> {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> invokeAny(p0: MutableCollection<out Callable<T>>): T {
        TODO("Not yet implemented")
    }

    override fun <T : Any?> invokeAny(p0: MutableCollection<out Callable<T>>, p1: Long, p2: TimeUnit): T {
        TODO("Not yet implemented")
    }

    fun isBusy(): Boolean {
        return running.get() >= threads.size
    }

    fun isBacklog(): Boolean {
        return queue.size >= threads.size
    }

    companion object Priorities {
        const val HIGHEST = Int.MAX_VALUE
        const val HIGHER = 500
        const val HIGH = 100
        const val NORMAL = 0
        const val LOW = -HIGH
        const val LOWER = -HIGHER
        const val LOWEST = Int.MIN_VALUE
    }
}
