/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */
package de.bixilon.kutil.buffer.bytes.out

import de.bixilon.kutil.collections.primitive.bytes.HeapArrayByteList
import java.nio.ByteBuffer
import java.nio.charset.StandardCharsets
import java.util.*

open class OutByteBuffer() {
    private val bytes = HeapArrayByteList()

    constructor(buffer: OutByteBuffer) : this() {
        bytes.addAll(buffer.bytes)
    }

    fun writeShort(short: Short) {
        writeShort(short.toInt())
    }

    fun writeShort(short: Int) {
        writeByte(short ushr Byte.SIZE_BITS)
        writeByte(short)
    }

    fun writeInt(int: Int) {
        writeShort(int shr Short.SIZE_BITS)
        writeShort(int)
    }

    open fun writeBareByteArray(data: ByteArray) {
        bytes.addAll(data)
    }

    open fun writeByteArray(data: ByteArray) {
        writeVarInt(data.size)
        bytes.addAll(data)
    }

    fun writeLong(value: Long) {
        writeInt((value shr Int.SIZE_BITS).toInt())
        writeInt(value.toInt())
    }

    open fun writeString(string: String) {
        val bytes = string.toByteArray(StandardCharsets.UTF_8)
        writeVarInt(bytes.size)
        writeBareByteArray(bytes)
    }

    fun writeVarLong(long: Long) {
        var value = long
        do {
            var temp = value and 0x7F
            value = value ushr 7
            if (value != 0L) {
                temp = temp or 0x80
            }
            writeByte(temp)
        } while (value != 0L)
    }

    fun writeByte(byte: Byte) {
        bytes.add(byte)
    }

    fun writeByte(byte: Int) {
        writeByte((byte and 0xFF).toByte())
    }

    fun writeByte(long: Long) {
        writeByte((long and 0xFF).toByte())
    }

    open fun writeFloat(float: Float) {
        writeInt(float.toBits())
    }

    fun writeFloat(float: Double) {
        writeFloat(float.toFloat())
    }

    open fun writeDouble(double: Double) {
        writeLong(double.toBits())
    }

    fun writeDouble(float: Float) {
        writeDouble(float.toDouble())
    }

    fun writeUUID(uuid: UUID) {
        writeLong(uuid.mostSignificantBits)
        writeLong(uuid.leastSignificantBits)
    }

    fun writeVarInt(int: Int) {
        // thanks https://wiki.vg/Protocol#VarInt_and_VarLong
        var value = int
        do {
            var temp = value and 0x7F
            value = value ushr 7
            if (value != 0) {
                temp = temp or 0x80
            }
            writeByte(temp)
        } while (value != 0)
    }

    fun writeBoolean(value: Boolean) {
        writeByte(
            if (value) {
                0x01
            } else {
                0x00
            }
        )
    }

    open fun writeBareString(string: String) {
        writeBareByteArray(string.toByteArray(StandardCharsets.UTF_8))
    }

    fun toArray(): ByteArray {
        return bytes.toArray()
    }

    fun writeBareIntArray(data: IntArray) {
        for (i in data) {
            writeInt(i)
        }
    }

    fun writeIntArray(data: IntArray) {
        writeVarInt(data.size)
        writeBareIntArray(data)
    }

    fun writeBareLongArray(data: LongArray) {
        for (l in data) {
            writeLong(l)
        }
    }

    fun writeLongArray(data: LongArray) {
        writeVarInt(data.size)
        writeBareLongArray(data)
    }

    fun writeTo(buffer: ByteBuffer) {
        buffer.put(toArray())
    }

    fun <T> writeArray(array: Array<T>, writer: (T) -> Unit) {
        writeVarInt(array.size)
        for (entry in array) {
            writer(entry)
        }
    }

    fun <T> writeArray(collection: Collection<T>, writer: (T) -> Unit) {
        writeVarInt(collection.size)
        for (entry in collection) {
            writer(entry)
        }
    }

    fun <T> writeOptional(value: T?, writer: (T) -> Unit) {
        writeBoolean(value != null)
        value?.let(writer)
    }
}
