/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.observer

import de.bixilon.kutil.cast.CastUtil.unsafeCast
import de.bixilon.kutil.collections.map.bi.AbstractBiMap
import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.observer.ObserveUtil.delegate
import de.bixilon.kutil.observer.ObserveUtil.invalid
import java.lang.ref.WeakReference
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty0
import kotlin.reflect.jvm.isAccessible

open class DataObserver<V>(protected var value: V) : ReadWriteProperty<Any, V> {
    protected val observers: MutableList<Pair<WeakReference<*>, (V) -> Unit>> = mutableListOf()
    protected val lock = SimpleLock()

    override fun getValue(thisRef: Any, property: KProperty<*>): V {
        return value
    }

    override fun setValue(thisRef: Any, property: KProperty<*>, value: V) {
        if (this.value == value) {
            return
        }
        this.value = value
        val toRemove: MutableSet<Pair<WeakReference<*>, (V) -> Unit>> = mutableSetOf()
        lock.acquire()
        for (pair in observers) {
            val (reference, observer) = pair
            if (reference.invalid) {
                toRemove += pair
                continue
            }
            try {
                observer(value)
            } catch (exception: Throwable) {
                exception.printStackTrace()
            }
        }
        lock.release()
        if (toRemove.isNotEmpty()) {
            lock.lock()
            observers -= toRemove
            lock.unlock()
        }
    }

    open operator fun plusAssign(observer: Pair<WeakReference<*>, (V) -> Unit>) {
        lock.lock()
        this.observers += observer
        lock.unlock()
    }

    companion object {

        fun <V> observed(value: V): DataObserver<V> {
            if (value is AbstractBiMap<*, *>) {
                throw IllegalArgumentException("Use observedBiMap")
            }
            if (value is Map<*, *>) {
                throw IllegalArgumentException("Use observedMap")
            }
            if (value is List<*>) {
                throw IllegalArgumentException("Use observedList")
            }
            if (value is Set<*>) {
                throw IllegalArgumentException("Use observedSet")
            }
            return DataObserver(value)
        }

        @JvmName("watched1")
        fun <V> V.observed(): DataObserver<V> = observed(this)

        fun <V> KProperty0<V>.observe(owner: Any, instant: Boolean = false, observer: (V) -> Unit) {
            isAccessible = true
            val delegate = delegate.unsafeCast<DataObserver<V>>()
            val reference = WeakReference(owner)
            delegate += Pair(reference, observer)
            if (instant) {
                observer(delegate.value)
            }
        }
    }
}
