/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.worker.unconditional

import de.bixilon.kutil.collections.CollectionUtil.synchronizedListOf
import de.bixilon.kutil.concurrent.pool.DefaultThreadPool
import de.bixilon.kutil.concurrent.pool.ThreadPool
import de.bixilon.kutil.concurrent.pool.ThreadPoolRunnable
import de.bixilon.kutil.concurrent.worker.WorkerExecutor
import de.bixilon.kutil.concurrent.worker.WorkerStates
import de.bixilon.kutil.latch.CountUpAndDownLatch

class UnconditionalWorker(
    val pool: ThreadPool = DefaultThreadPool,
    val forcePool: Boolean = false,
    val threadSafe: Boolean = false,
) {
    private val tasks: MutableList<UnconditionalTask> = if (threadSafe) synchronizedListOf() else mutableListOf()
    var state: WorkerStates = WorkerStates.PREPARING
        private set

    fun add(task: UnconditionalTask) {
        if (state != WorkerStates.PREPARING) {
            throw IllegalStateException("Invalid state: $state")
        }
        tasks += task
    }

    operator fun plusAssign(task: UnconditionalTask) = add(task)

    fun add(executor: WorkerExecutor) = add(UnconditionalTask(executor = executor))

    operator fun plusAssign(executor: WorkerExecutor) = add(executor)

    fun work(latch: CountUpAndDownLatch? = null) {
        if (state != WorkerStates.PREPARING) {
            throw IllegalStateException("Invalid state: $state")
        }
        state = WorkerStates.WORKING

        val innerLatch = latch?.let { CountUpAndDownLatch(1, it) } ?: CountUpAndDownLatch(1)

        tasks.sortWith { a, b -> b.priority - a.priority }
        innerLatch.plus(tasks.size)

        val running = CountUpAndDownLatch(0)

        fun run(executor: (CountUpAndDownLatch) -> Unit) {
            try {
                executor.invoke(innerLatch)
            } catch (error: Throwable) {
                error.printStackTrace()
            }
            innerLatch.dec()
            running.dec()
        }

        for (task in tasks) {
            running.inc()
            if (!forcePool && pool.isBusy()) {
                run(task.executor)
            } else {
                pool += ThreadPoolRunnable(task.priority) { run(task.executor) }
            }
            running.waitIfGreater(pool.threadCount)
        }


        innerLatch.dec()
        innerLatch.await()

        state = WorkerStates.FINISHED
    }

    fun reset() {
        tasks.clear()
        state = WorkerStates.PREPARING
    }
}
