/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.worker.task

import de.bixilon.kutil.collections.CollectionUtil.synchronizedListOf
import de.bixilon.kutil.collections.CollectionUtil.synchronizedMapOf
import de.bixilon.kutil.collections.CollectionUtil.toSynchronizedList
import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.concurrent.pool.DefaultThreadPool
import de.bixilon.kutil.concurrent.pool.ThreadPool
import de.bixilon.kutil.concurrent.pool.ThreadPoolRunnable
import de.bixilon.kutil.concurrent.worker.WorkerExecutor
import de.bixilon.kutil.concurrent.worker.WorkerStates
import de.bixilon.kutil.latch.CountUpAndDownLatch

class TaskWorker(
    var errorHandler: (task: WorkerTask, exception: Throwable) -> Unit = { _, error -> error.printStackTrace() },
    var criticalErrorHandler: (task: WorkerTask, exception: Throwable) -> Unit = { _, error -> error.printStackTrace() },
    private val pool: ThreadPool = DefaultThreadPool,
    private val forcePool: Boolean = false,
) {
    private val todo: MutableMap<Any, WorkerTask> = synchronizedMapOf()
    private val otherTodo: MutableList<WorkerTask> = synchronizedListOf()
    var state: WorkerStates = WorkerStates.PREPARING
        private set

    operator fun plusAssign(task: WorkerTask) {
        check(state == WorkerStates.PREPARING) { "Task worker is already working!" }
        if (task.dependencies.contains(task.identifier)) {
            throw IllegalArgumentException("Task can not depend on itself!")
        }
        if (task.identifier != null) {
            val previous = todo.put(task.identifier, task)
            if (previous != null) {
                System.err.println("Task ${task.identifier} replaced existing task!")
            }
        } else {
            otherTodo += task
        }
    }

    operator fun plusAssign(runnable: WorkerExecutor) {
        this += WorkerTask(executor = runnable)
    }

    @Synchronized
    fun work(progress: CountUpAndDownLatch = CountUpAndDownLatch(0)) {
        if (state != WorkerStates.PREPARING) throw IllegalStateException("Invalid state: $state")
        state = WorkerStates.WORKING
        val todo = this.todo.values.toMutableList()
        todo += otherTodo
        todo.sortWith { a, b -> b.priority - a.priority }
        val done: MutableSet<Any> = mutableSetOf()
        val doneLock = SimpleLock()
        var exit = false

        val workerProgress = CountUpAndDownLatch(1, progress)

        while (todo.isNotEmpty()) {
            var changed = false
            task@ for (task in todo.toSynchronizedList()) {
                if (task.dependencies.isNotEmpty()) {
                    doneLock.acquire()
                    for (dependency in task.dependencies) {
                        if (!done.contains(dependency)) {
                            doneLock.release()
                            continue@task
                        }
                    }
                    doneLock.release()
                }

                val taskProgress = CountUpAndDownLatch(2, workerProgress)
                todo -= task
                val runnable = Runnable {
                    taskProgress.dec()

                    try {
                        task.executor(taskProgress)
                        task.identifier?.let {
                            doneLock.lock()
                            done += it
                            doneLock.unlock()
                        }
                        changed = true

                        taskProgress.dec()
                    } catch (exception: Throwable) {
                        if (exception !is InterruptedException) {
                            exception.printStackTrace()
                        }
                        taskProgress.count = 0
                        changed = true
                        if (task.optional) {
                            errorHandler.invoke(task, exception)
                        } else {
                            criticalErrorHandler.invoke(task, exception)
                            exit = true
                        }
                    }
                }

                pool += ThreadPoolRunnable(priority = task.priority, runnable = runnable, forcePool = forcePool)
            }
            if (exit || todo.isEmpty()) break
            if (changed) continue
            workerProgress.waitForChange()
            if (exit) break
        }

        workerProgress.dec()
        workerProgress.await()
        state = WorkerStates.FINISHED
    }
}
