/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.time

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.concurrent.pool.DefaultThreadPool
import de.bixilon.kutil.concurrent.pool.ThreadPoolRunnable
import de.bixilon.kutil.time.TimeUtil.millis
import java.util.concurrent.TimeUnit

object TimeWorker {
    private val TASKS: MutableSet<TimeWorkerTask> = mutableSetOf()
    private val taskLock = SimpleLock()

    init {
        Thread({
            while (true) {
                val currentTime = millis()
                taskLock.acquire()
                val toRemove: MutableSet<TimeWorkerTask> = mutableSetOf()
                for (task in TASKS) {
                    if (task.executing) {
                        continue
                    }
                    if (currentTime - task.lastExecution <= task.interval) {
                        continue
                    }
                    DefaultThreadPool += ThreadPoolRunnable(priority = task.executionPriority) {
                        if (!task.lock.tryLock(1L, TimeUnit.MILLISECONDS)) {
                            return@ThreadPoolRunnable
                        }
                        if (task.executing) {
                            task.lock.unlock()
                            return@ThreadPoolRunnable
                        }
                        if (millis() - currentTime >= task.maxDelayTime) {
                            task.lock.unlock()
                            return@ThreadPoolRunnable
                        }
                        try {
                            try {
                                task.thread = Thread.currentThread()
                                task.runnable.run()
                            } catch (exception: Exception) {
                                exception.printStackTrace()
                            }
                            task.thread = null
                        } catch (exception: InterruptedException) {
                            task.thread = null
                        }
                        task.lastExecution = currentTime
                        task.lock.unlock()
                    }
                    if (task.runOnce) {
                        toRemove += task
                    }
                }
                taskLock.release()
                if (toRemove.isNotEmpty()) {
                    taskLock.lock()
                    TASKS -= toRemove
                    taskLock.unlock()
                }
                Thread.sleep(1)
            }
        }, "TimeWorkerThread").start()
    }

    fun runLater(millis: Int, runnable: Runnable): TimeWorkerTask {
        val task = TimeWorkerTask(
            interval = millis,
            runnable = runnable,
            runOnce = true,
        )
        task.lastExecution = millis()
        addTask(task)
        return task
    }

    fun addTask(task: TimeWorkerTask) {
        taskLock.lock()
        TASKS += task
        taskLock.unlock()
    }

    operator fun plusAssign(task: TimeWorkerTask) = addTask(task)

    fun removeTask(task: TimeWorkerTask) {
        taskLock.lock()
        TASKS -= task
        taskLock.unlock()
    }

    operator fun minusAssign(task: TimeWorkerTask) = removeTask(task)
}
