/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.pool

import de.bixilon.kutil.time.TimeUtil
import java.util.concurrent.locks.ReentrantLock

class ThreadPoolRunnable(
    val priority: Int = ThreadPool.NORMAL,
    interruptable: Boolean = false,
    val forcePool: Boolean = true,
    var runnable: Runnable? = null,
) : Comparable<ThreadPoolRunnable> {
    private val interruptLock = ReentrantLock()
    val creation = TimeUtil.nanos()
    var wasInterrupted = false
    var thread: Thread? = null
        set(value) {
            if (value == null) {
                Thread.interrupted() // clear interrupted state
            }
            interruptLock.lock()
            field = value
            interruptLock.unlock()
        }


    var interruptable: Boolean = interruptable
        set(value) {
            interruptLock.lock()
            if (field == value) {
                interruptLock.unlock()
                return
            }
            field = value
            interruptLock.unlock()
        }

    init {
        if (priority == 1 || priority == -1) {
            throw IllegalArgumentException("Priority of 1 or -1 is unsupported!")
        }
    }

    fun interrupt() {
        interruptLock.lock()
        try {
            val thread = this.thread
            if (!interruptable || thread == null || wasInterrupted) {
                return
            }
            thread.interrupt()
            wasInterrupted = true
        } finally {
            interruptLock.unlock()
        }
    }

    override fun compareTo(other: ThreadPoolRunnable): Int {
        val difference = other.priority - priority
        if (difference != 0) {
            return difference
        }
        if (creation - other.creation < 0) {
            return -1
        }
        return 1
    }
}
