/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.observer.set

import de.bixilon.kutil.cast.CastUtil.unsafeCast
import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.observer.ObserveUtil.delegate
import de.bixilon.kutil.observer.ObserveUtil.invalid
import java.lang.ref.WeakReference
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty0
import kotlin.reflect.jvm.isAccessible

open class SetObserver<V>(value: MutableSet<V>) : ReadWriteProperty<Any, MutableSet<V>> {
    protected val observers: MutableList<Pair<WeakReference<*>, (SetChange<V>) -> Unit>> = mutableListOf()
    protected val value = ObservedSet(value)
    protected val lock = SimpleLock()

    init {
        this.value.addObserver {
            val toRemove: MutableSet<Pair<WeakReference<*>, (SetChange<V>) -> Unit>> = mutableSetOf()
            lock.acquire()
            for (pair in observers) {
                val (reference, observer) = pair
                if (reference.invalid) {
                    toRemove += pair
                    continue
                }
                try {
                    observer(it)
                } catch (exception: Throwable) {
                    exception.printStackTrace()
                }
            }
            lock.release()
            if (toRemove.isNotEmpty()) {
                lock.lock()
                observers -= toRemove
                lock.unlock()
            }
        }
    }

    override fun getValue(thisRef: Any, property: KProperty<*>): MutableSet<V> {
        return value
    }

    override fun setValue(thisRef: Any, property: KProperty<*>, value: MutableSet<V>) {
        if (this.value == value) {
            return
        }
        this.value.unsafe = value
    }

    open operator fun plusAssign(observer: Pair<WeakReference<*>, (SetChange<V>) -> Unit>) {
        lock.lock()
        this.observers += observer
        lock.unlock()
    }

    companion object {
        fun <V> observedSet(value: MutableSet<V>): SetObserver<V> = SetObserver(value)

        @JvmName("observedSet1")
        fun <V> MutableSet<V>.observedSet(): SetObserver<V> = observedSet(this)

        fun <V> KProperty0<Set<V>>.observeSet(owner: Any, instant: Boolean = false, observer: (SetChange<V>) -> Unit) {
            isAccessible = true
            val delegate = delegate.unsafeCast<SetObserver<V>>()
            val reference = WeakReference(owner)
            delegate += Pair(reference, observer)
            if (instant) {
                observer(SetChange(adds = delegate.value, emptyList()))
            }
        }
    }
}
