/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.observer

import de.bixilon.kutil.cast.CastUtil.unsafeCast
import java.lang.ref.WeakReference
import java.lang.reflect.Field
import kotlin.reflect.KClass
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty0
import kotlin.reflect.KProperty1
import kotlin.reflect.full.memberProperties
import kotlin.reflect.jvm.isAccessible

object ObserveUtil {
    // [field][<receiver>][reference][listener]
    private val CLASS = Class.forName("kotlin.jvm.internal.CallableReference")
    private val RECEIVER_FIELD = CLASS.getDeclaredField("receiver")

    private val PROPERTY_1_CLASS = Class.forName("kotlin.reflect.jvm.internal.KPropertyImpl")
    private val CONTAINER_FIELD = PROPERTY_1_CLASS.getDeclaredField("container")

    private val MUTABLE_PROPERTY_1_CLASS = Class.forName("kotlin.reflect.jvm.internal.KProperty1Impl")
    private val DELEGATE_SOURCE_FIELD = MUTABLE_PROPERTY_1_CLASS.getDeclaredField("delegateSource")

    private val KCLASS_CLASS = Class.forName("kotlin.reflect.jvm.internal.KClassImpl")
    private val JCLASS_FIELD = KCLASS_CLASS.getDeclaredField("jClass")

    init {
        RECEIVER_FIELD.isAccessible = true
        CONTAINER_FIELD.isAccessible = true
        DELEGATE_SOURCE_FIELD.isAccessible = true
        JCLASS_FIELD.isAccessible = true
    }

    val WeakReference<*>.invalid: Boolean
        get() {
            val referenceValue = get() ?: return true
            if (referenceValue is ObservedReference<*> && referenceValue.unsafeCast<ObservedReference<Any>>().isValid(referenceValue)) {
                return false
            }
            // ToDO if (referenceValue == null || (referenceValue is LifecycleOwner && referenceValue.lifecycle.currentState == Lifecycle.State.DESTROYED)) {
            return false
        }

    val KProperty<*>.receiver: Any
        get() = RECEIVER_FIELD.get(this)

    val KProperty<*>.container: KClass<*>
        get() = CONTAINER_FIELD.get(this).unsafeCast()

    val KClass<*>.jClass: Class<*>
        get() = JCLASS_FIELD.get(this).unsafeCast()

    val KProperty<*>.identifier: String
        get() = when (this) {
            is KProperty0<*> -> this.receiver::class.java.name + ":" + this.name
            is KProperty1<*, *> -> this.container.jClass.name + ":" + this.name
            else -> TODO("Can not identify $this")
        }

    val KProperty<*>.delegateSourceField: Field
        get() {
            val lazy = DELEGATE_SOURCE_FIELD.get(this)
            return lazy.unsafeCast<Lazy<Any>>().value.unsafeCast()
        }

    val KProperty0<*>.delegate: Any?
        get() {
            getDelegate()?.let { return it }

            try {
                val receiver = this.receiver
                val property = receiver::class.memberProperties.find { it.name == this.name }.unsafeCast<KProperty1<Any, Any>>()
                property.isAccessible = true
                val delegateSourceField = property.delegateSourceField
                delegateSourceField.isAccessible = true

                return property.getDelegate(receiver)
            } catch (error: Throwable) {
                error.printStackTrace()
            }
            System.err.println("Can not get delegate for $this. Is it an observable property?")
            return null
        }
}
