/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.watcher.map


class ObservedMap<K, V>(private val original: MutableMap<K, V>) : MutableMap<K, V> {
    private val watchers: MutableList<(MapChange<K, V>) -> Unit> = mutableListOf()
    private val emptySet: Set<Map.Entry<K, V>> = setOf()
    override val size: Int by original::size

    fun addWatcher(watcher: (MapChange<K, V>) -> Unit) {
        watchers += watcher
    }

    private fun notifyWatchers(change: MapChange<K, V>) {
        for (watcher in watchers) {
            watcher.invoke(change)
        }
    }

    override fun containsKey(key: K): Boolean {
        return original.containsKey(key)
    }

    override fun containsValue(value: V): Boolean {
        return original.containsValue(value)
    }

    override fun get(key: K): V? {
        return original[key]
    }

    override fun isEmpty(): Boolean {
        return original.isEmpty()
    }

    override val entries: MutableSet<MutableMap.MutableEntry<K, V>> by original::entries
    override val keys: MutableSet<K> by original::keys
    override val values: MutableCollection<V> by original::values

    override fun clear() {
        notifyWatchers(MapChange(emptySet, entries))
        original.clear()
    }

    override fun put(key: K, value: V): V? {
        val original = original.put(key, value)
        val removed = original?.let { return@let setOf(MapEntry(key, it)) } ?: emptySet

        notifyWatchers(MapChange(setOf(MapEntry(key, value)), removed))
        return original
    }

    override fun putAll(from: Map<out K, V>) {
        val removed: MutableSet<MapEntry<K, V>> = mutableSetOf()
        for ((key, value) in from) {
            removed += MapEntry(key, original.put(key, value) ?: continue)
        }

        notifyWatchers(MapChange(from.entries, removed))
    }

    override fun remove(key: K): V? {
        val removed = original.remove(key) ?: return null
        notifyWatchers(MapChange(emptySet, setOf(MapEntry(key, removed))))
        return removed
    }
}
