/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.watcher.list

class ObservedList<V>(private val original: MutableList<V>) : MutableList<V> {
    private val watchers: MutableList<(ListChange<V>) -> Unit> = mutableListOf()
    private val emptyList: List<V> = listOf()
    override val size: Int by original::size

    fun addWatcher(watcher: (ListChange<V>) -> Unit) {
        watchers += watcher
    }

    private fun notifyWatchers(change: ListChange<V>) {
        for (watcher in watchers) {
            watcher.invoke(change)
        }
    }

    override fun contains(element: V): Boolean {
        return original.contains(element)
    }

    override fun containsAll(elements: Collection<V>): Boolean {
        return original.containsAll(elements)
    }

    override fun get(index: Int): V {
        return original[index]
    }

    override fun indexOf(element: V): Int {
        return original.indexOf(element)
    }

    override fun isEmpty(): Boolean {
        return original.isEmpty()
    }

    override fun iterator(): MutableIterator<V> {
        return original.iterator()
    }

    override fun lastIndexOf(element: V): Int {
        return original.lastIndexOf(element)
    }

    override fun add(element: V): Boolean {
        notifyWatchers(ListChange(listOf(element), emptyList))
        return original.add(element)
    }

    override fun add(index: Int, element: V) {
        notifyWatchers(ListChange(listOf(element), emptyList))
        return original.add(index, element)
    }

    override fun addAll(index: Int, elements: Collection<V>): Boolean {
        notifyWatchers(ListChange(elements, emptyList))
        return original.addAll(index, elements)
    }

    override fun addAll(elements: Collection<V>): Boolean {
        notifyWatchers(ListChange(elements, emptyList))
        return original.addAll(elements)
    }

    override fun clear() {
        notifyWatchers(ListChange(emptyList, original.toList()))
        original.clear()
    }

    override fun listIterator(): MutableListIterator<V> {
        return original.listIterator()
    }

    override fun listIterator(index: Int): MutableListIterator<V> {
        return original.listIterator(index)
    }

    override fun remove(element: V): Boolean {
        if (!original.remove(element)) {
            return false
        }
        notifyWatchers(ListChange(emptyList, listOf(element)))
        return true
    }

    override fun removeAll(elements: Collection<V>): Boolean {
        val contained: MutableList<V> = mutableListOf()
        for (element in elements) {
            if (element !in original) {
                continue
            }
            contained += element
        }
        if (!original.removeAll(elements)) {
            return false
        }
        if (contained.isNotEmpty()) {
            notifyWatchers(ListChange(emptyList, contained))
        }
        return true
    }

    override fun removeAt(index: Int): V {
        val removed = original.removeAt(index)
        notifyWatchers(ListChange(emptyList, listOf(removed)))
        return removed
    }

    override fun retainAll(elements: Collection<V>): Boolean {
        val removed = original.retainAll(elements)
        notifyWatchers(ListChange(emptyList, elements)) // ToDo
        return removed
    }

    override fun set(index: Int, element: V): V {
        val previous = original.set(index, element)
        notifyWatchers(ListChange(listOf(element), listOf(element)))
        return previous
    }

    override fun subList(fromIndex: Int, toIndex: Int): MutableList<V> {
        return original.subList(fromIndex, toIndex)
    }
}
