/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.queue

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.time.TimeUtil

class Queue(initialCapacity: Int = 100) {
    private val queue: MutableList<Runnable> = ArrayList(initialCapacity)
    private var removeLock = SimpleLock()

    val size: Int
        get() = queue.size

    fun add(runnable: Runnable): Boolean {
        queue += runnable
        return true
    }

    operator fun plusAssign(runnable: Runnable) {
        add(runnable)
    }

    private fun runFirst() {
        try {
            queue.removeAt(0).run()
        } catch (exception: Throwable) {
            exception.printStackTrace()
        }
    }

    fun work(maxJobs: Int = Int.MAX_VALUE) {
        removeLock.lock()
        var jobsDone = 0
        var cachedSize = queue.size
        while (cachedSize-- > 0 || queue.size > 0) {
            runFirst()

            if (jobsDone++ == maxJobs) {
                break
            }
        }
        removeLock.unlock()
    }

    fun timeWork(time: Long) {
        check(time > 0L) { "Can not have <= 0 time to do jobs!" }
        val start = TimeUtil.millis
        removeLock.lock()

        var cachedSize = queue.size
        while (cachedSize-- > 0 || queue.size > 0) {
            runFirst()

            if (TimeUtil.millis - start >= time) {
                break
            }
        }
        removeLock.unlock()
    }

    fun clear() {
        removeLock.lock()
        queue.clear()
        removeLock.unlock()
    }
}
