/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.watcher.set

class ObservedSet<V>(private val original: MutableSet<V>) : MutableSet<V> {
    private val watchers: MutableList<(SetChange<V>) -> Unit> = mutableListOf()
    private val emptySet: Set<V> = setOf()
    override val size: Int by original::size

    fun addWatcher(watcher: (SetChange<V>) -> Unit) {
        watchers += watcher
    }

    private fun notifyWatchers(change: SetChange<V>) {
        for (watcher in watchers) {
            watcher.invoke(change)
        }
    }

    override fun add(element: V): Boolean {
        val added = original.add(element)
        if (added) {
            notifyWatchers(SetChange(setOf(element), emptySet))
        }
        return added
    }

    override fun addAll(elements: Collection<V>): Boolean {
        val add = original.addAll(elements)
        if (!add) {
            return false
        }
        notifyWatchers(SetChange(elements, emptySet))
        return true
    }

    override fun clear() {
        notifyWatchers(SetChange(emptySet, original.toSet()))
        original.clear()
    }

    override fun iterator(): MutableIterator<V> {
        return original.iterator()
    }

    override fun remove(element: V): Boolean {
        val removed = original.remove(element)
        if (!removed) {
            return false
        }
        notifyWatchers(SetChange(emptySet, setOf(element)))
        return true
    }

    override fun removeAll(elements: Collection<V>): Boolean {
        val removed = original.removeAll(elements)
        if (!removed) {
            return false
        }
        notifyWatchers(SetChange(emptySet, elements))
        return true
    }

    override fun retainAll(elements: Collection<V>): Boolean {
        val retained = original.retainAll(elements)
        if (!retained) {
            return false
        }
        println("Kutil: Set::retainAll is not observed yet!")
        return true
    }

    override fun contains(element: V): Boolean {
        return original.contains(element)
    }

    override fun containsAll(elements: Collection<V>): Boolean {
        return original.containsAll(elements)
    }

    override fun isEmpty(): Boolean {
        return original.isEmpty()
    }
}
