/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.watcher.map.bi

import de.bixilon.kutil.cast.CastUtil.unsafeCast
import de.bixilon.kutil.collections.map.bi.AbstractBiMap
import de.bixilon.kutil.collections.map.bi.AbstractMutableBiMap
import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.watcher.WatchUtil.delegate
import de.bixilon.kutil.watcher.WatchUtil.invalid
import de.bixilon.kutil.watcher.map.MapChange
import java.lang.ref.WeakReference
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty
import kotlin.reflect.KProperty0
import kotlin.reflect.jvm.isAccessible

class BiMapDataWatcher<K, V>(value: AbstractMutableBiMap<K, V>) : ReadWriteProperty<Any, AbstractMutableBiMap<K, V>> {
    val watchers: MutableList<Pair<WeakReference<*>, (MapChange<K, V>) -> Unit>> = mutableListOf()
    private val value = ObservedBiMap(value)
    val lock = SimpleLock()

    init {
        this.value.addWatcher {
            val toRemove: MutableSet<Pair<WeakReference<*>, (MapChange<K, V>) -> Unit>> = mutableSetOf()
            lock.acquire()
            for (pair in watchers) {
                val (reference, observer) = pair
                if (reference.invalid) {
                    toRemove += pair
                    continue
                }
                try {
                    observer(it)
                } catch (exception: Throwable) {
                    exception.printStackTrace()
                }
            }
            lock.release()
            if (toRemove.isNotEmpty()) {
                lock.lock()
                watchers -= toRemove
                lock.unlock()
            }
        }
    }

    override fun getValue(thisRef: Any, property: KProperty<*>): AbstractMutableBiMap<K, V> {
        return value
    }

    override fun setValue(thisRef: Any, property: KProperty<*>, value: AbstractMutableBiMap<K, V>) {
        if (this.value == value) {
            return
        }
        this.value.clear()
        this.value.putAll(value)
    }

    companion object {
        fun <K, V> watchedBiMap(value: AbstractMutableBiMap<K, V>): BiMapDataWatcher<K, V> = BiMapDataWatcher(value)

        @JvmName("watchedBiMap1")
        fun <K, V> AbstractMutableBiMap<K, V>.watchedBiMap(): BiMapDataWatcher<K, V> = watchedBiMap(this)

        @Suppress("NON_PUBLIC_CALL_FROM_PUBLIC_INLINE")
        fun <K, V> KProperty0<AbstractBiMap<K, V>>.observeBiMap(owner: Any, observer: (MapChange<K, V>) -> Unit) {
            isAccessible = true
            val delegate = delegate.unsafeCast<BiMapDataWatcher<K, V>>()
            val reference = WeakReference(owner)
            delegate.lock.lock()
            delegate.watchers += Pair(reference, observer)
            delegate.lock.unlock()
        }
    }
}
