/*
 * KUtil
 * Copyright (C) 2021 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.time

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.concurrent.pool.DefaultThreadPool
import de.bixilon.kutil.time.TimeUtil
import java.util.concurrent.TimeUnit

object TimeWorker {
    private val TASKS: MutableSet<TimeWorkerTask> = mutableSetOf()
    private val taskLock = SimpleLock()

    init {
        Thread({
            while (true) {
                val currentTime = TimeUtil.millis
                taskLock.acquire()
                for (task in TASKS) {
                    if (task.executing) {
                        continue
                    }
                    if (currentTime - task.lastExecution <= task.interval) {
                        continue
                    }
                    DefaultThreadPool += execute@{
                        if (!task.lock.tryLock(1L, TimeUnit.MILLISECONDS)) {
                            return@execute
                        }
                        if (task.executing) {
                            task.lock.unlock()
                            return@execute
                        }
                        if (TimeUtil.millis - currentTime >= task.maxDelayTime) {
                            task.lock.unlock()
                            return@execute
                        }
                        task.executing = true
                        try {
                            task.runnable.run()
                        } catch (exception: Exception) {
                            exception.printStackTrace()
                        }
                        task.lastExecution = currentTime
                        task.executing = false
                        task.lock.unlock()
                    }
                    if (task.runOnce) {
                        TASKS -= task
                    }
                }
                taskLock.release()
                Thread.sleep(1)
            }
        }, "TimeWorkerThread").start()
    }

    fun runIn(millis: Int, runnable: Runnable) {
        val task = TimeWorkerTask(
            interval = millis,
            runnable = runnable,
            runOnce = true,
        )
        task.lastExecution = TimeUtil.millis
        taskLock.lock()
        TASKS += task
        taskLock.unlock()
    }

    fun addTask(task: TimeWorkerTask) {
        taskLock.lock()
        TASKS += task
        taskLock.unlock()
    }

    operator fun plusAssign(task: TimeWorkerTask) = addTask(task)

    fun removeTask(task: TimeWorkerTask) {
        taskLock.lock()
        TASKS -= task
        taskLock.unlock()
    }

    operator fun minusAssign(task: TimeWorkerTask) = removeTask(task)
}
