package de.bixilon.kotlinglm.vec2

import de.bixilon.kotlinglm.*
import de.bixilon.kotlinglm.vec1.Vec1bool
import de.bixilon.kotlinglm.vec1.Vec1t
import de.bixilon.kotlinglm.vec2.operators.op_Vec2ui
import de.bixilon.kotlinglm.vec3.Vec3bool
import de.bixilon.kotlinglm.vec3.Vec3t
import de.bixilon.kotlinglm.vec4.Vec4bool
import de.bixilon.kotlinglm.vec4.Vec4t
import de.bixilon.kotlinkool.*
import org.lwjgl.system.MemoryStack
import org.lwjgl.system.MemoryUtil.memGetInt
import org.lwjgl.system.MemoryUtil.memPutInt
import unsigned.Uint
import unsigned.UintArray
import unsigned.toUint
import java.io.PrintStream
import java.nio.*

/**
 * Created by elect on 08/10/16.
 */

class Vec2ui(@JvmField var ofs: Int, var array: UintArray) : Vec2t<Uint>, ToBuffer {

    inline var x: Uint
        get() = Uint(array[ofs])
        set(value) = array.set(ofs, value)
    inline var y: Uint
        get() = Uint(array[ofs + 1])
        set(value) = array.set(ofs + 1, value)

    inline var vX: Int
        get() = array[ofs].toInt()
        set(value) = array.set(ofs, value.toUint())
    inline var vY: Int
        get() = array[ofs + 1].toInt()
        set(value) = array.set(ofs + 1, value.toUint())

    // -- Implicit basic constructors --

    constructor() : this(0.ui)
    constructor(v: Vec2ui) : this(v.x, v.y)

    // -- Explicit basic constructors --

    @JvmOverloads
    constructor(x: Int, y: Int = x) : this(x.ui, y.ui)

    @JvmOverloads
    constructor(x: Uint, y: Uint = x) : this(0, UintArray(intArrayOf(x.v, y.v)))

    // -- Conversion constructors --


    constructor(v: Number) : this(v.ui)
    constructor(x: Number, y: Number) : this(x.ui, y.ui)

    // Explicit conversions (From section 5.4.1 Conversion and scalar constructors of GLSL 1.30.08 specification)

    @JvmOverloads
    constructor(x: Vec1t<out Number>, y: Number = x._x) : this(x._x, y)

    constructor(x: Number, v: Vec1t<out Number>) : this(x, v._x)
    constructor(x: Vec1t<out Number>, y: Vec1t<out Number>) : this(x._x, y._x)

    constructor(v: Vec2t<out Number>) : this(v._x, v._y)
    constructor(v: Vec3t<out Number>) : this(v._x, v._y)
    constructor(v: Vec4t<out Number>) : this(v._x, v._y)

    @JvmOverloads
    constructor(x: Boolean, y: Boolean = x) : this(x.ui, y.ui)

    constructor(x: Boolean, v: Vec1bool) : this(x.ui, v.x.ui)

    @JvmOverloads
    constructor(v: Vec1bool, y: Boolean = v.x) : this(v.x.ui, y.ui)

    constructor(v: Vec1bool, y: Vec1bool) : this(v.x.ui, y.x.ui)

    constructor(v: Vec2bool) : this(v.x.ui, v.y.ui)
    constructor(v: Vec3bool) : this(v.x.ui, v.y.ui)
    constructor(v: Vec4bool) : this(v.x.ui, v.y.ui)

    constructor(bytes: ByteArray, index: Int = 0, oneByteOneUint: Boolean = false, bigEndian: Boolean = true) : this(
            if (oneByteOneUint) bytes[index].ui else bytes.getUint(index, bigEndian),
            if (oneByteOneUint) bytes[index + 1].ui else bytes.getUint(index + Uint.BYTES, bigEndian))

    constructor(chars: CharArray, index: Int = 0) : this(chars[index].ui, chars[index + 1].ui)
    constructor(shorts: ShortArray, index: Int = 0) : this(shorts[index], shorts[index + 1])
    constructor(ints: IntArray, index: Int = 0) : this(ints[index], ints[index + 1])
    constructor(longs: LongArray, index: Int = 0) : this(longs[index], longs[index + 1])
    constructor(floats: FloatArray, index: Int = 0) : this(floats[index], floats[index + 1])
    constructor(doubles: DoubleArray, index: Int = 0) : this(doubles[index], doubles[index + 1])
    constructor(booleans: BooleanArray, index: Int = 0) : this(booleans[index].ui, booleans[index + 1].ui)

    constructor(numbers: Array<out Number>, index: Int = 0) : this(numbers[index], numbers[index + 1])
    constructor(chars: Array<Char>, index: Int = 0) : this(chars[index].ui, chars[index + 1].ui)
    constructor(booleans: Array<Boolean>, index: Int = 0) : this(booleans[index].ui, booleans[index + 1].ui)

    constructor(list: Iterable<*>, index: Int = 0) : this(list.elementAt(index)!!.toInt, list.elementAt(index + 1)!!.toInt)

    constructor(bytes: ByteBuffer, index: Int = bytes.pos, oneByteOneUint: Boolean = false) : this(
            if (oneByteOneUint) bytes[index].ui else bytes.getInt(index).ui,
            if (oneByteOneUint) bytes[index + 1].ui else bytes.getInt(index + Uint.BYTES).ui)

    constructor(chars: CharBuffer, index: Int = chars.pos) : this(chars[index].ui, chars[index + 1].ui)
    constructor(shorts: ShortBuffer, index: Int = shorts.pos) : this(shorts[index], shorts[index + 1])
    constructor(ints: IntBuffer, index: Int = ints.pos) : this(ints[index], ints[index + 1])
    constructor(longs: LongBuffer, index: Int = longs.pos) : this(longs[index], longs[index + 1])
    constructor(floats: FloatBuffer, index: Int = floats.pos) : this(floats[index], floats[index + 1])
    constructor(doubles: DoubleBuffer, index: Int = doubles.pos) : this(doubles[index], doubles[index + 1])

    constructor(block: (Int) -> Uint) : this(block(0), block(1))


    fun set(bytes: ByteArray, index: Int = 0, oneByteOneUint: Boolean = false, bigEndian: Boolean = true) {
        x.v = if (oneByteOneUint) bytes[index].i else bytes.getInt(index, bigEndian)
        y.v = if (oneByteOneUint) bytes[index + 1].i else bytes.getInt(index + Uint.BYTES, bigEndian)
    }

    fun set(bytes: ByteBuffer, index: Int = bytes.pos, oneByteOneUint: Boolean = false) {
        x.v = if (oneByteOneUint) bytes[index].i else bytes.getInt(index)
        y.v = if (oneByteOneUint) bytes[index + 1].i else bytes.getInt(index + Uint.BYTES)
    }


    fun put(x: Uint, y: Uint) {
        this.x = x
        this.y = y
    }

    operator fun invoke(x: Uint, y: Uint): Vec2ui {
        this.x = x
        this.y = y
        return this
    }

    fun put(x: Int, y: Int) {
        this.x.v = x
        this.y.v = y
    }

    operator fun invoke(x: Int, y: Int): Vec2ui {
        this.x.v = x
        this.y.v = y
        return this
    }

    override fun put(x: Number, y: Number) {
        this.x = x.ui
        this.y = y.ui
    }

    override operator fun invoke(x: Number, y: Number): Vec2ui {
        this.x = x.ui
        this.y = y.ui
        return this
    }

    fun to(bytes: ByteArray, index: Int) = to(bytes, index, true)
    override fun to(bytes: ByteArray, index: Int, bigEndian: Boolean): ByteArray {
        bytes.putInt(index, x.v)
        bytes.putInt(index + Int.BYTES, y.v)
        return bytes
    }

    fun toIntArray(): IntArray = to(IntArray(length), 0)
    infix fun to(ints: IntArray): IntArray = to(ints, 0)
    fun to(ints: IntArray, index: Int): IntArray {
        System.arraycopy(array, ofs, ints, index, length)
        return ints
    }

    override fun to(buf: ByteBuffer, offset: Int): ByteBuffer {
        buf.putInt(offset, x.v)
        buf.putInt(offset + Int.BYTES, y.v)
        return buf
    }

    fun toIntBufferStack(): IntBuffer = to(MemoryStack.stackGet().mallocInt(length), 0)
    infix fun toIntBuffer(stack: MemoryStack): IntBuffer = to(stack.mallocInt(length), 0)
    fun toIntBuffer(): IntBuffer = to(IntBuffer(length), 0)
    infix fun to(buf: IntBuffer): IntBuffer = to(buf, buf.pos)
    fun to(buf: IntBuffer, index: Int): IntBuffer {
        buf[index] = x.v
        buf[index + 1] = y.v
        return buf
    }

    infix fun to(ptr: Ptr) {
        memPutInt(ptr, x.v)
        memPutInt(ptr + Int.BYTES, y.v)
    }

    // -- Component accesses --

    override operator fun set(index: Int, value: Number) = when (index) {
        0 -> x = value.ui
        1 -> y = value.ui
        else -> throw ArrayIndexOutOfBoundsException()
    }

    // -- Unary arithmetic operators --

    operator fun unaryPlus() = this

    // no unaryMinus operator, only signed

    // -- Increment main.and decrement operators --

    operator fun inc(res: Vec2ui = Vec2ui()) = plus(res, this, 1, 1)
    fun incAssign() = plus(this, this, 1, 1)


    operator fun dec(res: Vec2ui = Vec2ui()) = minus(res, this, 1, 1)
    fun decAssign() = minus(this, this, 1, 1)


    // -- Specific binary arithmetic operators --

    infix operator fun plus(b: Uint) = plus(Vec2ui(), this, b, b)
    infix operator fun plus(b: Int) = plus(Vec2ui(), this, b, b)
    infix operator fun plus(b: Vec2ui) = plus(Vec2ui(), this, b.x, b.y)

    @JvmOverloads
    fun plus(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = plus(res, this, bX, bY)

    @JvmOverloads
    fun plus(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = plus(res, this, bX, bY)

    fun plus(b: Uint, res: Vec2ui) = plus(res, this, b, b)
    fun plus(b: Int, res: Vec2ui) = plus(res, this, b, b)
    fun plus(b: Vec2ui, res: Vec2ui) = plus(res, this, b.x, b.y)

    fun plusAssign(bX: Uint, bY: Uint) = plus(this, this, bX, bY)
    fun plusAssign(bX: Int, bY: Int) = plus(this, this, bX, bY)
    infix operator fun plusAssign(b: Uint) {
        plus(this, this, b, b)
    }

    infix operator fun plusAssign(b: Int) {
        plus(this, this, b, b)
    }

    infix operator fun plusAssign(b: Vec2ui) {
        plus(this, this, b.x, b.y)
    }


    infix operator fun minus(b: Uint) = minus(Vec2ui(), this, b, b)
    infix operator fun minus(b: Int) = minus(Vec2ui(), this, b, b)
    infix operator fun minus(b: Vec2ui) = minus(Vec2ui(), this, b.x, b.y)

    @JvmOverloads
    fun minus(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = minus(res, this, bX, bY)

    @JvmOverloads
    fun minus(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = minus(res, this, bX, bY)

    fun minus(b: Uint, res: Vec2ui) = minus(res, this, b, b)
    fun minus(b: Int, res: Vec2ui) = minus(res, this, b, b)
    fun minus(b: Vec2ui, res: Vec2ui) = minus(res, this, b.x, b.y)

    fun minusAssign(bX: Uint, bY: Uint) = minus(this, this, bX, bY)
    fun minusAssign(bX: Int, bY: Int) = minus(this, this, bX, bY)
    infix operator fun minusAssign(b: Uint) {
        minus(this, this, b, b)
    }

    infix operator fun minusAssign(b: Int) {
        minus(this, this, b, b)
    }

    infix operator fun minusAssign(b: Vec2ui) {
        minus(this, this, b.x, b.y)
    }


    infix operator fun times(b: Uint) = times(Vec2ui(), this, b, b)
    infix operator fun times(b: Int) = times(Vec2ui(), this, b, b)
    infix operator fun times(b: Vec2ui) = times(Vec2ui(), this, b.x, b.y)

    @JvmOverloads
    fun times(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = times(res, this, bX, bY)

    @JvmOverloads
    fun times(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = times(res, this, bX, bY)

    fun times(b: Uint, res: Vec2ui) = times(res, this, b, b)
    fun times(b: Int, res: Vec2ui) = times(res, this, b, b)
    fun times(b: Vec2ui, res: Vec2ui) = times(res, this, b.x, b.y)

    fun timesAssign(bX: Uint, bY: Uint) = times(this, this, bX, bY)
    fun timesAssign(bX: Int, bY: Int) = times(this, this, bX, bY)
    infix operator fun timesAssign(b: Uint) {
        times(this, this, b, b)
    }

    infix operator fun timesAssign(b: Int) {
        times(this, this, b, b)
    }

    infix operator fun timesAssign(b: Vec2ui) {
        times(this, this, b.x, b.y)
    }


    infix operator fun div(b: Uint) = div(Vec2ui(), this, b, b)
    infix operator fun div(b: Int) = div(Vec2ui(), this, b, b)
    infix operator fun div(b: Vec2ui) = div(Vec2ui(), this, b.x, b.y)

    @JvmOverloads
    fun div(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = div(res, this, bX, bY)

    @JvmOverloads
    fun div(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = div(res, this, bX, bY)

    fun div(b: Uint, res: Vec2ui) = div(res, this, b, b)
    fun div(b: Int, res: Vec2ui) = div(res, this, b, b)
    fun div(b: Vec2ui, res: Vec2ui) = div(res, this, b.x, b.y)

    fun divAssign(bX: Uint, bY: Uint) = div(this, this, bX, bY)
    fun divAssign(bX: Int, bY: Int) = div(this, this, bX, bY)
    infix operator fun divAssign(b: Uint) {
        div(this, this, b, b)
    }

    infix operator fun divAssign(b: Int) {
        div(this, this, b, b)
    }

    infix operator fun divAssign(b: Vec2ui) {
        div(this, this, b.x, b.y)
    }


    infix operator fun rem(b: Uint) = rem(Vec2ui(), this, b, b)
    infix operator fun rem(b: Int) = rem(Vec2ui(), this, b, b)
    infix operator fun rem(b: Vec2ui) = rem(Vec2ui(), this, b.x, b.y)

    @JvmOverloads
    fun rem(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = rem(res, this, bX, bY)

    @JvmOverloads
    fun rem(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = rem(res, this, bX, bY)

    fun rem(b: Uint, res: Vec2ui) = rem(res, this, b, b)
    fun rem(b: Int, res: Vec2ui) = rem(res, this, b, b)
    fun rem(b: Vec2ui, res: Vec2ui) = rem(res, this, b.x, b.y)

    fun remAssign(bX: Uint, bY: Uint) = rem(this, this, bX, bY)
    fun remAssign(bX: Int, bY: Int) = rem(this, this, bX, bY)
    infix operator fun remAssign(b: Uint) {
        rem(this, this, b, b)
    }

    infix operator fun remAssign(b: Int) {
        rem(this, this, b, b)
    }

    infix operator fun remAssign(b: Vec2ui) {
        rem(this, this, b.x, b.y)
    }


    // -- Generic binary arithmetic operators --

    infix operator fun plus(b: Number) = plus(Vec2ui(), this, b.i, b.i)
    infix operator fun plus(b: Vec2t<out Number>) = plus(Vec2ui(), this, b._x.i, b._y.i)

    @JvmOverloads
    fun plus(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = plus(res, this, bX.i, bY.i)

    fun plus(b: Number, res: Vec2ui) = plus(res, this, b.i, b.i)
    fun plus(b: Vec2t<out Number>, res: Vec2ui) = plus(res, this, b._x.i, b._y.i)

    fun plusAssign(bX: Number, bY: Number) = plus(this, this, bX.i, bY.i)
    infix operator fun plusAssign(b: Number) {
        plus(this, this, b.i, b.i)
    }

    infix operator fun plusAssign(b: Vec2t<out Number>) {
        plus(this, this, b._x.i, b._y.i)
    }


    infix operator fun minus(b: Number) = minus(Vec2ui(), this, b.i, b.i)
    infix operator fun minus(b: Vec2t<out Number>) = minus(Vec2ui(), this, b._x.i, b._y.i)

    @JvmOverloads
    fun minus(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = minus(res, this, bX.i, bY.i)

    fun minus(b: Number, res: Vec2ui) = minus(res, this, b.i, b.i)
    fun minus(b: Vec2t<out Number>, res: Vec2ui) = minus(res, this, b._x.i, b._y.i)

    fun minusAssign(bX: Number, bY: Number) = minus(this, this, bX.i, bY.i)
    infix operator fun minusAssign(b: Number) {
        minus(this, this, b.i, b.i)
    }

    infix operator fun minusAssign(b: Vec2t<out Number>) {
        minus(this, this, b._x.i, b._y.i)
    }


    infix operator fun times(b: Number) = times(Vec2ui(), this, b.i, b.i)
    infix operator fun times(b: Vec2t<out Number>) = times(Vec2ui(), this, b._x.i, b._y.i)

    @JvmOverloads
    fun times(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = times(res, this, bX.i, bY.i)

    fun times(b: Number, res: Vec2ui) = times(res, this, b.i, b.i)
    fun times(b: Vec2t<out Number>, res: Vec2ui) = times(res, this, b._x.i, b._y.i)

    fun timesAssign(bX: Number, bY: Number) = times(this, this, bX.i, bY.i)
    infix operator fun timesAssign(b: Number) {
        times(this, this, b.i, b.i)
    }

    infix operator fun timesAssign(b: Vec2t<out Number>) {
        times(this, this, b._x.i, b._y.i)
    }


    infix operator fun div(b: Number) = div(Vec2ui(), this, b.i, b.i)
    infix operator fun div(b: Vec2t<out Number>) = div(Vec2ui(), this, b._x.i, b._y.i)

    @JvmOverloads
    fun div(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = div(res, this, bX.i, bY.i)

    fun div(b: Number, res: Vec2ui) = div(res, this, b.i, b.i)
    fun div(b: Vec2t<out Number>, res: Vec2ui) = div(res, this, b._x.i, b._y.i)

    fun divAssign(bX: Number, bY: Number) = div(this, this, bX.i, bY.i)
    infix operator fun divAssign(b: Number) {
        div(this, this, b.i, b.i)
    }

    infix operator fun divAssign(b: Vec2t<out Number>) {
        div(this, this, b._x.i, b._y.i)
    }


    infix operator fun rem(b: Number) = rem(Vec2ui(), this, b.i, b.i)
    infix operator fun rem(b: Vec2t<out Number>) = rem(Vec2ui(), this, b._x.i, b._y.i)

    @JvmOverloads
    fun rem(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = rem(res, this, bX.i, bY.i)

    fun rem(b: Number, res: Vec2ui) = rem(res, this, b.i, b.i)
    fun rem(b: Vec2t<out Number>, res: Vec2ui) = rem(res, this, b._x.i, b._y.i)

    fun remAssign(bX: Number, bY: Number) = rem(this, this, bX.i, bY.i)
    infix operator fun remAssign(b: Number) {
        rem(this, this, b.i, b.i)
    }

    infix operator fun remAssign(b: Vec2t<out Number>) {
        rem(this, this, b._x.i, b._y.i)
    }


    // -- Specific bitwise operators --

    infix fun and(b: Uint) = and(Vec2ui(), this, b, b)
    infix fun and(b: Int) = and(Vec2ui(), this, b, b)
    infix fun and(b: Vec2ui) = and(Vec2ui(), this, b.x, b.y)

    fun and(b: Uint, res: Vec2ui) = and(res, this, b, b)
    fun and(b: Int, res: Vec2ui) = and(res, this, b, b)
    fun and(b: Vec2ui, res: Vec2ui) = and(res, this, b.x, b.y)
    @JvmOverloads
    fun and(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = and(res, this, bX, bY)

    @JvmOverloads
    fun and(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = and(res, this, bX, bY)

    infix fun andAssign(b: Uint) = and(this, this, b, b)
    infix fun andAssign(b: Int) = and(this, this, b, b)
    infix fun andAssign(b: Vec2ui) = and(this, this, b.x, b.y)
    fun andAssign(bX: Uint, bY: Uint) = and(this, this, bX, bY)
    fun andAssign(bX: Int, bY: Int) = and(this, this, bX, bY)


    infix fun or(b: Uint) = or(Vec2ui(), this, b, b)
    infix fun or(b: Int) = or(Vec2ui(), this, b, b)
    infix fun or(b: Vec2ui) = or(Vec2ui(), this, b.x, b.y)

    fun or(b: Uint, res: Vec2ui) = or(res, this, b, b)
    fun or(b: Int, res: Vec2ui) = or(res, this, b, b)
    fun or(b: Vec2ui, res: Vec2ui) = or(res, this, b.x, b.y)
    @JvmOverloads
    fun or(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = or(res, this, bX, bY)

    @JvmOverloads
    fun or(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = or(res, this, bX, bY)

    infix fun orAssign(b: Uint) = or(this, this, b, b)
    infix fun orAssign(b: Int) = or(this, this, b, b)
    infix fun orAssign(b: Vec2ui) = or(this, this, b.x, b.y)
    fun orAssign(bX: Uint, bY: Uint) = or(this, this, bX, bY)
    fun orAssign(bX: Int, bY: Int) = or(this, this, bX, bY)


    infix fun xor(b: Uint) = xor(Vec2ui(), this, b, b)
    infix fun xor(b: Int) = xor(Vec2ui(), this, b, b)
    infix fun xor(b: Vec2ui) = xor(Vec2ui(), this, b.x, b.y)

    fun xor(b: Uint, res: Vec2ui) = xor(res, this, b, b)
    fun xor(b: Int, res: Vec2ui) = xor(res, this, b, b)
    fun xor(b: Vec2ui, res: Vec2ui) = xor(res, this, b.x, b.y)
    @JvmOverloads
    fun xor(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = xor(res, this, bX, bY)

    @JvmOverloads
    fun xor(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = xor(res, this, bX, bY)

    infix fun xorAssign(b: Uint) = xor(this, this, b, b)
    infix fun xorAssign(b: Int) = xor(this, this, b, b)
    infix fun xorAssign(b: Vec2ui) = xor(this, this, b.x, b.y)
    fun xorAssign(bX: Uint, bY: Uint) = xor(this, this, bX, bY)
    fun xorAssign(bX: Int, bY: Int) = xor(this, this, bX, bY)


    infix fun shl(b: Uint) = shl(Vec2ui(), this, b, b)
    infix fun shl(b: Int) = shl(Vec2ui(), this, b, b)
    infix fun shl(b: Vec2ui) = shl(Vec2ui(), this, b.x, b.y)

    fun shl(b: Uint, res: Vec2ui) = shl(res, this, b, b)
    fun shl(b: Int, res: Vec2ui) = shl(res, this, b, b)
    fun shl(b: Vec2ui, res: Vec2ui) = shl(res, this, b.x, b.y)
    @JvmOverloads
    fun shl(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = shl(res, this, bX, bY)

    @JvmOverloads
    fun shl(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = shl(res, this, bX, bY)

    infix fun shlAssign(b: Uint) = shl(this, this, b, b)
    infix fun shlAssign(b: Int) = shl(this, this, b, b)
    infix fun shlAssign(b: Vec2ui) = shl(this, this, b.x, b.y)
    fun shlAssign(bX: Uint, bY: Uint) = shl(this, this, bX, bY)
    fun shlAssign(bX: Int, bY: Int) = shl(this, this, bX, bY)


    infix fun shr(b: Uint) = shr(Vec2ui(), this, b, b)
    infix fun shr(b: Int) = shr(Vec2ui(), this, b, b)
    infix fun shr(b: Vec2ui) = shr(Vec2ui(), this, b.x, b.y)

    fun shr(b: Uint, res: Vec2ui) = shr(res, this, b, b)
    fun shr(b: Int, res: Vec2ui) = shr(res, this, b, b)
    fun shr(b: Vec2ui, res: Vec2ui) = shr(res, this, b.x, b.y)
    @JvmOverloads
    fun shr(bX: Uint, bY: Uint, res: Vec2ui = Vec2ui()) = shr(res, this, bX, bY)

    @JvmOverloads
    fun shr(bX: Int, bY: Int, res: Vec2ui = Vec2ui()) = shr(res, this, bX, bY)

    infix fun shrAssign(b: Uint) = shr(this, this, b, b)
    infix fun shrAssign(b: Int) = shr(this, this, b, b)
    infix fun shrAssign(b: Vec2ui) = shr(this, this, b.x, b.y)
    fun shrAssign(bX: Uint, bY: Uint) = shr(this, this, bX, bY)
    fun shrAssign(bX: Int, bY: Int) = shr(this, this, bX, bY)


    @JvmOverloads
    fun inv(res: Vec2ui = Vec2ui()) = inv(res, this)

    fun invAssign() = inv(this, this)


    // -- Generic bitwise operators --

    infix fun and(b: Number) = and(Vec2ui(), this, b.i, b.i)
    infix fun and(b: Vec2t<out Number>) = and(Vec2ui(), this, b._x.i, b._y.i)

    fun and(b: Number, res: Vec2ui) = and(res, this, b.i, b.i)
    fun and(b: Vec2t<out Number>, res: Vec2ui) = and(res, this, b._x.i, b._y.i)

    @JvmOverloads
    fun and(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = and(res, this, bX.i, bY.i)

    infix fun andAssign(b: Number) = and(this, this, b.i, b.i)
    infix fun andAssign(b: Vec2t<out Number>) = and(this, this, b._x.i, b._y.i)
    fun andAssign(bX: Number, bY: Number) = and(this, this, bX.i, bY.i)


    infix fun or(b: Number) = or(Vec2ui(), this, b.i, b.i)
    infix fun or(b: Vec2t<out Number>) = or(Vec2ui(), this, b._x.i, b._y.i)

    fun or(b: Number, res: Vec2ui) = or(res, this, b.i, b.i)
    fun or(b: Vec2t<out Number>, res: Vec2ui) = or(res, this, b._x.i, b._y.i)

    @JvmOverloads
    fun or(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = or(res, this, bX.i, bY.i)

    infix fun orAssign(b: Number) = or(this, this, b.i, b.i)
    infix fun orAssign(b: Vec2t<out Number>) = or(this, this, b._x.i, b._y.i)
    fun orAssign(bX: Number, bY: Number) = or(this, this, bX.i, bY.i)


    infix fun xor(b: Number) = xor(Vec2ui(), this, b.i, b.i)
    infix fun xor(b: Vec2t<out Number>) = xor(Vec2ui(), this, b._x.i, b._y.i)

    fun xor(b: Number, res: Vec2ui) = xor(res, this, b.i, b.i)
    fun xor(b: Vec2t<out Number>, res: Vec2ui) = xor(res, this, b._x.i, b._y.i)

    @JvmOverloads
    fun xor(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = xor(res, this, bX.i, bY.i)

    infix fun xorAssign(b: Number) = xor(this, this, b.i, b.i)
    infix fun xorAssign(b: Vec2t<out Number>) = xor(this, this, b._x.i, b._y.i)
    fun xorAssign(bX: Number, bY: Number) = xor(this, this, bX.i, bY.i)


    infix fun shl(b: Number) = shl(Vec2ui(), this, b.i, b.i)
    infix fun shl(b: Vec2t<out Number>) = shl(Vec2ui(), this, b._x.i, b._y.i)

    fun shl(b: Number, res: Vec2ui) = shl(res, this, b.i, b.i)
    fun shl(b: Vec2t<out Number>, res: Vec2ui) = shl(res, this, b._x.i, b._y.i)

    @JvmOverloads
    fun shl(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = shl(res, this, bX.i, bY.i)

    infix fun shlAssign(b: Number) = shl(this, this, b.i, b.i)
    infix fun shlAssign(b: Vec2t<out Number>) = shl(this, this, b._x.i, b._y.i)
    fun shlAssign(bX: Number, bY: Number) = shl(this, this, bX.i, bY.i)


    infix fun shr(b: Number) = shr(Vec2ui(), this, b.i, b.i)
    infix fun shr(b: Vec2t<out Number>) = shr(Vec2ui(), this, b._x.i, b._y.i)

    fun shr(b: Number, res: Vec2ui) = shr(res, this, b.i, b.i)
    fun shr(b: Vec2t<out Number>, res: Vec2ui) = shr(res, this, b._x.i, b._y.i)

    @JvmOverloads
    fun shr(bX: Number, bY: Number, res: Vec2ui = Vec2ui()) = shr(res, this, bX.i, bY.i)

    infix fun shrAssign(b: Number) = shr(this, this, b.i, b.i)
    infix fun shrAssign(b: Vec2t<out Number>) = shr(this, this, b._x.i, b._y.i)
    fun shrAssign(bX: Number, bY: Number) = shr(this, this, bX.i, bY.i)


    infix fun allLessThan(ui: Uint): Boolean = x < ui && y < ui
    infix fun anyLessThan(ui: Uint): Boolean = x < ui || y < ui
    infix fun lessThan(ui: Uint): Vec2bool = Vec2bool { get(it) < ui }

    infix fun allLessThanEqual(ui: Uint): Boolean = x <= ui && y <= ui
    infix fun anyLessThanEqual(ui: Uint): Boolean = x <= ui || y <= ui
    infix fun lessThanEqual(ui: Uint): Vec2bool = Vec2bool { get(it) <= ui }

    infix fun allEqual(ui: Uint): Boolean = x == ui && y == ui
    infix fun anyEqual(ui: Uint): Boolean = x == ui || y == ui
    infix fun equal(ui: Uint): Vec2bool = Vec2bool { get(it) == ui }

    infix fun allNotEqual(ui: Uint): Boolean = x != ui && y != ui
    infix fun anyNotEqual(ui: Uint): Boolean = x != ui || y != ui
    infix fun notEqual(ui: Uint): Vec2bool = Vec2bool { get(it) != ui }

    infix fun allGreaterThan(ui: Uint): Boolean = x > ui && y > ui
    infix fun anyGreaterThan(ui: Uint): Boolean = x > ui || y > ui
    infix fun greaterThan(ui: Uint): Vec2bool = Vec2bool { get(it) > ui }

    infix fun allGreaterThanEqual(ui: Uint): Boolean = x >= ui && y >= ui
    infix fun anyGreaterThanEqual(ui: Uint): Boolean = x >= ui || y >= ui
    infix fun greaterThanEqual(ui: Uint): Vec2bool = Vec2bool { get(it) >= ui }


    infix fun allLessThan(v: Vec2ui): Boolean = x < v.x && y < v.y
    infix fun anyLessThan(v: Vec2ui): Boolean = x < v.x || y < v.y
    infix fun lessThan(v: Vec2ui): Vec2bool = Vec2bool { get(it) < v[it] }

    infix fun allLessThanEqual(v: Vec2ui): Boolean = x <= v.x && y <= v.y
    infix fun anyLessThanEqual(v: Vec2ui): Boolean = x <= v.x || y <= v.y
    infix fun lessThanEqual(v: Vec2ui): Vec2bool = Vec2bool { get(it) <= v[it] }

    infix fun allEqual(v: Vec2ui): Boolean = x == v.x && y == v.y
    infix fun anyEqual(v: Vec2ui): Boolean = x == v.x || y == v.y
    infix fun equal(v: Vec2ui): Vec2bool = Vec2bool { get(it) == v[it] }

    infix fun allNotEqual(v: Vec2ui): Boolean = x != v.x && y != v.y
    infix fun anyNotEqual(v: Vec2ui): Boolean = x != v.x || y != v.y
    infix fun notEqual(v: Vec2ui): Vec2bool = Vec2bool { get(it) != v[it] }

    infix fun allGreaterThan(v: Vec2ui): Boolean = x > v.x && y > v.y
    infix fun anyGreaterThan(v: Vec2ui): Boolean = x > v.x || y > v.y
    infix fun greaterThan(v: Vec2ui): Vec2bool = Vec2bool { get(it) > v[it] }

    infix fun allGreaterThanEqual(v: Vec2ui): Boolean = x >= v.x && y >= v.y
    infix fun anyGreaterThanEqual(v: Vec2ui): Boolean = x >= v.x || y >= v.y
    infix fun greaterThanEqual(v: Vec2ui): Vec2bool = Vec2bool { get(it) >= v[it] }


    companion object : op_Vec2ui {
        const val length = 2
        @JvmField
        val size = length * Uint.BYTES

        @JvmStatic
        fun fromPointer(ptr: Ptr) = Vec2ui(memGetInt(ptr), memGetInt(ptr + Int.BYTES))
    }

    override fun size() = size

    override fun equals(other: Any?) = other is Vec2ui && this[0] == other[0] && this[1] == other[1]

    override fun hashCode() = 31 * x.v.hashCode() + y.v.hashCode()

    @JvmOverloads
    fun print(name: String = "", stream: PrintStream = System.out) = stream.print("$name$this")

    @JvmOverloads
    fun println(name: String = "", stream: PrintStream = System.out) = stream.println("$name$this")


    //@formatter:off
    override inline var _x get() = x; set(value) { x = value }
    override inline var r get() = x; set(value) { x = value }
    override inline var s get() = x; set(value) { x = value }

    override inline var _y get() = y; set(value) { y = value }
    override inline var g get() = y; set(value) { y = value }
    override inline var t get() = y; set(value) { y = value }
    //@formatter:on

    override inline operator fun get(index: Int) = array[ofs + index]

    inline operator fun set(index: Int, value: Uint) {
        array[ofs + index] = value
    }

    override inline operator fun component1() = x
    override inline operator fun component2() = y


    override fun toString(): String = "($x, $y)"
}
