/**
 * 
 */
package excel.functions.utils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Vector;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import javax.imageio.ImageIO;
import javax.swing.JOptionPane;

import org.apache.poi.EncryptedDocumentException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;

import ij.ImagePlus;
import ij.measure.ResultsTable;



/**
 * @author BioVoxxel
 *
 */
public class ExcelUtils {
	
	//Logger parameters
	private static final String LOGGER_FORMAT = "[%1$tY-%1$tm-%1$td] [%1$tH:%1$tM:%1$tS] [%4$s] %5$s%6$s%n";
	protected static final Logger logger = Logger.getLogger(ExcelUtils.class.getName());
	private static Level LOGGER_LEVEL = Level.OFF;
	private static Level CONSOLE_LOGGER_LEVEL = Level.OFF;
	protected static boolean LOGGER_SETUP_MISSING = true;
	
	protected static int EXCEL_SHEET_NAME_LIMIT = 30;
	private static final String FIRST_SHEET_INDICATOR = "FIRST";
	private static final String LAST_SHEET_INDICATOR = "LAST";
	
	
	public ExcelUtils() {
		
	}
	
	
		
	public static void setLoggingParameters(Level logLevel, Level consoleLevel) {
		LOGGER_LEVEL = logLevel;
		CONSOLE_LOGGER_LEVEL = consoleLevel;
	}
	
	public static void setupLogger() {
		
			System.setProperty("java.util.logging.SimpleFormatter.format", LOGGER_FORMAT);			
			
			logger.setLevel(LOGGER_LEVEL);
			
			logger.setUseParentHandlers(false);
			
			Handler[] existingHandlers = logger.getHandlers();
			
			for (int handler = 0; handler <	existingHandlers.length; handler++) {
				logger.removeHandler(existingHandlers[handler]);			
			}
			
			ConsoleHandler consoleHandler = new ConsoleHandler();
			consoleHandler.setLevel(CONSOLE_LOGGER_LEVEL);
			consoleHandler.setFormatter(new SimpleFormatter());
			
			logger.addHandler(consoleHandler);	
	}
	
	
	
	
	public static int getLastColumnNum(Sheet sheet) {
		int lastRowIndex = sheet.getLastRowNum();
		logger.info("Last row = " + lastRowIndex);
		
		int lastColumnIndex = 0;
		for (int rowIndex = 0; rowIndex < lastRowIndex; rowIndex++) {
			lastColumnIndex = Math.max(lastColumnIndex, sheet.getRow(rowIndex).getLastCellNum());
		}
		
		logger.info("Last column = " + lastColumnIndex);
		return lastColumnIndex;
	}
	
	
	public static Boolean readBooleanMacroInput(Object parameter) {
		
		String param = parameter.toString();
		Boolean outputValue = false;
		
		if (param.equals("1") || param.equalsIgnoreCase("true")) {
			outputValue = true;
		} 
		
		return outputValue;
	}

	
	
	public static void setColumnDataFormat(Sheet sheet, Integer columnNumber, String dataFormatString) {
		
		if (dataFormatString.equals("")) {
			dataFormatString = "General";
		}
		
		Workbook workbook = sheet.getWorkbook();
		
		DataFormat dataFormat = workbook.createDataFormat();
    	
    	CellStyle cellStyle = workbook.createCellStyle();
    	
    	cellStyle.setDataFormat(dataFormat.getFormat(dataFormatString));
    	logger.info("Used data format = " + dataFormat.getFormat(dataFormatString));
    	
    	int lastRowNumber = sheet.getLastRowNum();
    	logger.info("Last row = " + lastRowNumber);
    	
    	for (int rowIndex = 0; rowIndex <= lastRowNumber; rowIndex++) {
			try {
				sheet.getRow(rowIndex).getCell(columnNumber).setCellStyle(cellStyle);
				logger.fine("Processing Row = " + rowIndex);				
			} catch (NullPointerException e) {
				logger.info("NullPointerException at row = " + rowIndex);
			}
    		
		}
	}
	
	
	
	
	
		
	protected String addMissingFileExtension(String filePath) {
		if (!filePath.endsWith(".xlsx")) {
			filePath = filePath + ".xlsx";
		}
		return filePath;
	}
	
	public ResultsTable createResultsTable(Vector<String[]> tableVector) {
		
		ResultsTable resultsTable = new ResultsTable(tableVector.get(0).length - 1);
		
		for (int col = 0; col < tableVector.size(); col++) {
			
			String[] columnArray = tableVector.get(col);
			
			logger.info("Column size (rows) = " + columnArray.length);
			if (!columnArray[0].equals("") && !columnArray[0].equals(" ") && columnArray[0] != null) {
				
				for (int row = 1; row < columnArray.length; row++) {
					
					logger.fine("Processing cell = col: " + col + " / row: " + row);
					
					if (isNumeric(columnArray[row])) {
						
						logger.fine("Value is numeric");
						resultsTable.setValue(columnArray[0], row-1, Double.parseDouble(columnArray[row]));
						
					} else {
						
						logger.fine("Value is String");
						
						if (columnArray[row].equals("") || columnArray[row] == null) {
							columnArray[row] = "NaN";
						}
						resultsTable.setValue(columnArray[0], row-1, columnArray[row]);	
					}
				}
			}
		}
		
		return resultsTable;
	}
	
	
	
	public String fixFilePath(String filePath) {
						
		if (filePath == null || filePath.equals("")) {
			filePath = System.getProperty("user.home") + File.separator + "NoFileNameGivenForTable.xlsx";
		} else {		
			filePath = addMissingFileExtension(filePath);
		}
		
		return filePath;
	}
	
	
	public Vector<String> getAllSheetNames(Workbook workbook) {
		
		Vector<String> allSheetNames = new Vector<String>();
		
		for (int sheet = 0; sheet < workbook.getNumberOfSheets(); sheet++) {
			allSheetNames.add(workbook.getSheetName(sheet));
		}
		
		return allSheetNames;
		
	}
	
	
	
	public Sheet[] getAllSheets(Workbook workbook) {
				
		Sheet[] allSheets = new Sheet[workbook.getNumberOfSheets()];
		
		for (int sheet = 0; sheet < allSheets.length; sheet++) {
			allSheets[sheet] = workbook.getSheetAt(sheet);
		}
		
		return allSheets;
		
	}
	
	
	
	public ResultsTable getIJResultsTable(String ijTableName) {
		
		logger.info("Given results table name = " + ijTableName);
		ResultsTable table = null;
		
		if (ijTableName.equals("")) {
			
			table = ResultsTable.getActiveTable();
			
		} else {
			
			table = ResultsTable.getResultsTable(ijTableName);
			
		}
		logger.info("Returning table = " + table);
		return table;
	}	
	
	
	public byte[] getImageAsByteArray(ImagePlus imagePlus) {
		byte [] imageData = null;
		
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
	    
		try {
			ImageIO.write(imagePlus.getBufferedImage(), "png", bos);
			imageData = bos.toByteArray();
			bos.close();
			
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		
		return imageData;
	}
	
	
	
	public Vector<String[]> getImageJTableColumnsAsVector(ResultsTable rt) {
		Vector<String[]> columnVector = new Vector<String[]>();
		String columnHeadingsAsString = rt.getColumnHeadings();
		logger.info("table columns headings: " + columnHeadingsAsString);
		String[] columnHeadingsArray = columnHeadingsAsString.split("\t");
		
		for (int col = 0; col < columnHeadingsArray.length; col++) {
			if (!columnHeadingsArray[col].equals(" ")) {
				String[] columnArray = rt.getColumnAsStrings(columnHeadingsArray[col]);
				columnVector.add(columnArray);				
			}
		}
		
		
		return columnVector;
	}
	
	

	public Vector<String[]> getImageJTableRowsAsVector(ResultsTable rt) {
		
		int rowCount = rt.getCounter();
		Vector<String[]> rowVector = new Vector<String[]>();
		String columnHeadingsAsString = rt.getColumnHeadings();
		logger.info("table columns headings: " + columnHeadingsAsString);
		String[] columnHeadingsArray = columnHeadingsAsString.split("\t");
		
		rowVector.add(columnHeadingsArray);
		
		for (int row = 0; row < rowCount; row++) {
			String[] rowArray = rt.getRowAsString(row).split("\t");
			rowVector.add(rowArray);
		}
		
		
		return rowVector;
	}
	
	
	public Sheet getSheet(Workbook workbook, Object sheetObject, Boolean appendData) {
		
		Sheet sheet = null;
		
		int sheetIndex = getSheetIndex(workbook, sheetObject);
				
		if (sheetIndex == -1) {
			String sheetName = getSheetName(workbook, sheetObject);
			sheet = workbook.createSheet(sheetName);	//sheet not existing --> create new sheet
		} else {
			
			try {
				sheet = workbook.getSheetAt(sheetIndex);	//get existing sheet
			} catch (Exception e) {
				logger.info(e.getMessage());
				
			}

			if (!appendData) {
				String uniqueSheetName = getUniqueSheetName(workbook, sheet.getSheetName());
				sheet = workbook.createSheet(uniqueSheetName);
				logger.info("New sheet " + uniqueSheetName + " created");
			}
		}
		
		
		logger.info("Sheet = " + sheet);
		return sheet;
	}


	public int getSheetIndex(Workbook workbook, Object sheetObject) {
		
		int sheetIndex = -1;
						
		String sheetObjectAsString = getSheetObjectAsString(sheetObject);
				
		
		if (sheetObjectAsString.equals("")) {
			
			ResultsTable activeTable = ResultsTable.getActiveTable();
			
			if (activeTable != null) {
				String activeTableName = activeTable.getTitle();
				sheetIndex = workbook.getSheetIndex(activeTableName);
			} else {
				noTableFoundMessage();
			}
			
		} else if (isNumeric(sheetObjectAsString)) {
			
			sheetIndex = Integer.parseInt(sheetObjectAsString);
						
			try {
				workbook.getSheetAt(sheetIndex);
			} catch (IllegalArgumentException e) {
				logger.info(e.getMessage());
				sheetIndex = -1;
			}
		
		} else if (sheetObjectAsString.equalsIgnoreCase(FIRST_SHEET_INDICATOR)) {
			
			sheetIndex = 0;
		
		} else if (sheetObjectAsString.equalsIgnoreCase(LAST_SHEET_INDICATOR)) {
		
			sheetIndex = workbook.getNumberOfSheets() - 1;
		
		} else {
				
			sheetIndex = workbook.getSheetIndex(sheetObjectAsString);
			
		}
		
		logger.info("Sheet index = " + sheetIndex);
		return sheetIndex;
	}


	public String getSheetName(Workbook workbook, Object sheetObject) {
		
		String sheetObjectAsString = getSheetObjectAsString(sheetObject);
		String sheetName = "";
		
		if (sheetObjectAsString.equals("")) {
			
			if (ResultsTable.getActiveTable() != null ) {
				sheetName = ResultsTable.getActiveTable().getTitle();
			} else {
				sheetName = "Sheet";
			}
			
		} else {
			sheetName = sheetObjectAsString;
		}
		
		logger.info("Sheet name = " + sheetName);
		return sheetName;
	}


	public String getSheetObjectAsString(Object sheetObject) {
		String sheetObjectAsString;
		
		if (sheetObject == null) {
			sheetObjectAsString = "";
		} else {
			sheetObjectAsString = sheetObject.toString();			
		}
		
		logger.info("Sheet object as string = \"" + sheetObjectAsString + "\"");
		return sheetObjectAsString;
	}


	public String getUniqueSheetName(Workbook workbook, String sheetName) {
		
		String uniqueSheetName = sheetName;
		
		Vector<String> allSheetNames = getAllSheetNames(workbook);
		
		int uniqueSheetIndex = 1;
		
		while (allSheetNames.contains(uniqueSheetName)) {
			uniqueSheetName = sheetName + "-" + uniqueSheetIndex;
			uniqueSheetIndex++;
		}
		return uniqueSheetName;
	}
	
	

	public int getVectorContentSize(Vector<String[]> vector) {
		return vector.get(0).length;
	}


	public Workbook getWorkbook(File outputFile) {
		
		Workbook workbook = null;
		
		logger.info("File " + outputFile + " exists = " + outputFile.exists());
		
		if (outputFile.exists()) {
			
			workbook = readWorkbook(outputFile);
			
		} else {
			
			try {
				
				workbook = WorkbookFactory.create(true);
				
			} catch (IOException e) {
				e.printStackTrace();
			}
		}	
		return workbook;	
	}
	
	
	public Workbook getWorkbook(String filePath) {
		
		File outputFile = new File(filePath);
				
		return getWorkbook(outputFile);
	}
	
	
	public Vector<String[]> getXlsxSheetAsVectorOfColumns(Sheet sheet) {
		
		//the vector holds the columns in form of a string array
		Vector<String[]> tableVector = new Vector<String[]>();
		
		int rowCount = sheet.getLastRowNum() + 1;
		logger.info("Row count = " + rowCount);
		int columnCount = 0;
		Row firstRow;	
		
		if (rowCount > 0) {
			firstRow = sheet.getRow(0);
			columnCount = firstRow.getLastCellNum();
		}
		logger.info("Column count = " + columnCount);
		
		for (int colIndex = 0; colIndex < columnCount; colIndex++) {

			String[] column = new String[rowCount];

			for (int rowIndex = 0; rowIndex < rowCount; rowIndex++) {
				
				Row currentRow = sheet.getRow(rowIndex);
				if (currentRow != null) {
					Cell currentCell = currentRow.getCell(colIndex);
					logger.fine("Current cell = " + currentCell);
					
					if (currentCell != null) {
						
						CellType currentCellType = currentCell.getCellType();
						logger.fine("Cell type = " + currentCellType);
						
						if (currentCellType == CellType.BLANK) {
							column[rowIndex] = "";
						} else if (currentCellType == CellType.NUMERIC) {
							column[rowIndex] = "" + currentCell.getNumericCellValue();
						} else if (currentCellType == CellType.STRING) {
							column[rowIndex] = currentCell.getStringCellValue();
						}
					} else {
						column[rowIndex] = "";
					}
				} else {
					column[rowIndex] = "";
				}
			}
			
			tableVector.add(column);
			
		}
		
		return tableVector;
	}
	
	
	
	public boolean isNumeric(Object object) {
		
		return isNumeric(object.toString());
		
	}


	
	@SuppressWarnings("unused")
	public boolean isNumeric(String text) {
		if (text == null) {
			return false;
		}
		
		try {
		    double d = Double.parseDouble(text);
		} catch (NumberFormatException nfe) {
		    return false;
		}
		return true;
	}

	
	
	public void noTableFoundMessage() {
		JOptionPane.showMessageDialog(null, "No results table found", "No table found", JOptionPane.WARNING_MESSAGE);
	}
	
	
	public Workbook readWorkbook(File workbookFile) {
		
		logger.info("Loading workbook from file = " + workbookFile.getAbsolutePath());
		
		Workbook workbook = null;
		
		try {
			workbook = WorkbookFactory.create(workbookFile);
		} catch (EncryptedDocumentException | IOException e) {
			try {
				workbook = WorkbookFactory.create(true);
			} catch (IOException e1) {
				e1.printStackTrace();
			}
			e.printStackTrace();
		} 
		
		logger.info("Returning loaded workbook = " + workbook);
		return workbook;
	}
	
	
	public Workbook readWorkbook(String filepath) {
		File workbookFile = new File(filepath);
		
		return readWorkbook(workbookFile);
	}
	
	
	public boolean saveWorkbook(Workbook workbook, File outputFile, boolean closeWorkbookAfterSaving) {
		
		boolean successfullySaved = false;
		
		if (outputFile.exists()) {
			outputFile.delete();
		}
		
		try {
			
			FileOutputStream fos = new FileOutputStream(outputFile);
			workbook.write(fos);
			successfullySaved = true;
			
			logger.info("Closing FileOutputStream " + fos);
			fos.close();
			
			logger.info("Closing Workbook " + workbook);
			if (closeWorkbookAfterSaving) {
				workbook.close();				
			}
			return successfullySaved;
			
		} catch (FileNotFoundException e) {
			JOptionPane.showMessageDialog(null, "Specified file cannot be accessed!\n"
					+ "Check if the file is open in another application.", "File not accessible", JOptionPane.WARNING_MESSAGE);
			//e.printStackTrace();
		} catch (IOException e) {
			logger.info("Original file cannot be deleted if existing or current file cannot be saved. File could be open in another application!");
			e.printStackTrace();
		}
		return successfullySaved;
		
	}
	
	
	public boolean saveWorkbook(Workbook workbook, String filePath, boolean closeWorkbookAfterSaving) {
		try {
			File outputFile = new File(filePath);
			return saveWorkbook(workbook, outputFile, closeWorkbookAfterSaving);
			
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}
	
}
