/**
 * 
 */
package excel.functions.plugins;

import java.io.File;
import java.util.logging.Logger;

import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.scijava.command.Command;
import org.scijava.plugin.Parameter;
import org.scijava.plugin.Plugin;

import excel.functions.macro.BioVoxxelMacroExtensionDescriptor;
import excel.functions.utils.ExcelUtils;
import ij.macro.MacroExtension;

/**
 * @author BioVoxxel
 *
 */
@Plugin(type = Command.class, menuPath = "Plugins>BioVoxxel>Set column data format")
public class SetColumnDataFormat implements Command, BioVoxxelMacroExtensionDescriptor {

	protected static final Logger logger = Logger.getLogger(ExcelUtils.class.getName());
	protected static boolean CLOSE_WORKBOOK_AFTER_SAVING = true;
	
	@Parameter(required = false)
	private File file;
	 
    @Parameter(label = "Sheet name or index", description = "if empty the sheet will receive the name of the table used truncated at 30 characters if the title is >30 char", required = false)
    private String sheetNameOrIndex;
    
    @Parameter(label="Column index (0-based)")
    private int columnIndex;
    
    @Parameter(label="Data format", choices = {"General", "0", "0.00", "0%", "0.00%", "h:mm:ss", "0.00E+00"})
    private String dataFormat;
	

	
	protected static void setColumnDataFormat(String filePathToExcelWorkbook, Object sheetObject, Integer columnNumber, String dataFormatString) {
		
    	ExcelUtils excelUtils = new ExcelUtils();
    	    	
    	String filePath = excelUtils.fixFilePath(filePathToExcelWorkbook);
		logger.info("Output file path  = " + filePath);
		
    	Workbook workbook = excelUtils.getWorkbook(filePath);
    	
    	Sheet sheet = excelUtils.getSheet(workbook, sheetObject, true);
    	logger.info("Current sheet = " + sheet);

    	
    	ExcelUtils.setColumnDataFormat(sheet, columnNumber, dataFormatString);
    	
    	excelUtils.saveWorkbook(workbook, filePath, CLOSE_WORKBOOK_AFTER_SAVING);
    	
	}

	
	
	@Override
	public void run() {
		
		String filePath = (file == null) ? "" : file.getAbsolutePath();
		
		setColumnDataFormat(filePath, sheetNameOrIndex, columnIndex, dataFormat);
		
	}
	
	@Override
	public void runFromMacro(Object[] parameters) {
		
		String filePath = (String)parameters[0];
		
		if (filePath.equals("") || filePath == null) {
			filePath = System.getProperty("user.home");
		}
		
		setColumnDataFormat(filePath, parameters[1].toString(), ((Double)parameters[2]).intValue(), parameters[3].toString());
	}

	@Override
	public int[] parameterTypes() {
		 return new int[] { MacroExtension.ARG_STRING, MacroExtension.ARG_STRING, MacroExtension.ARG_NUMBER, MacroExtension.ARG_STRING };
	}

	@Override
	public String description() {
		
		
		return "The column data format can de set for a defined workbook and sheet\n"
				+ "Possible formats are described under\n"
				+ "<a href=https://poi.apache.org/apidocs/dev/org/apache/poi/ss/usermodel/BuiltinFormats.html>BuiltinFormats</a>";
	}

	@Override
	public String parameters() {
		
		return "filePathToExcelWorkbook, sheetNameOrIndex, columnNumber, dataFormatString";
	}

	
	
}
