/**
 * 
 */
package excel.functions.plugins;

import java.io.File;
import java.util.Vector;
import java.util.logging.Logger;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.scijava.command.Command;
import org.scijava.plugin.Parameter;
import org.scijava.plugin.Plugin;

import excel.functions.macro.BioVoxxelMacroExtensionDescriptor;
import excel.functions.utils.ExcelUtils;
import ij.macro.MacroExtension;
import ij.measure.ResultsTable;


/**
 * @author BioVoxxel
 *
 */

@Plugin(type = Command.class, menuPath = "Plugins>BioVoxxel>Save table as worksheet")
public class SaveTableAsWorksheet implements Command, BioVoxxelMacroExtensionDescriptor {

	protected static final Logger logger = Logger.getLogger(ExcelUtils.class.getName());
	protected static boolean CLOSE_WORKBOOK_AFTER_SAVING = true;
    

	@Parameter(label = "Results table name", description = "if this field is empty the active table will be automatically used", required = false)
    private ResultsTable selectedTable;
    
	@Parameter(required = false)
	private File file;
	 
    @Parameter(label = "Sheet name or index", description = "if empty the sheet will receive the name of the table used truncated at 30 characters if the title is >30 char", required = false)
    private String sheetNameOrIndex;
    
    @Parameter(label = "Append data to specified sheet")
    private Boolean append = false;
    
    @Parameter(label = "Skip headings while appending")
    private Boolean skipAppendedHeadings = false;
    
   
      
    @Override
	public void run() {
    	
    	String filePath = (file == null) ? "" : file.getAbsolutePath();
    	
    	saveTableAsWorkbookSheet(selectedTable, filePath, sheetNameOrIndex, append, skipAppendedHeadings);			
	}
	
	
	protected static void saveTableAsWorkbookSheet(ResultsTable ijResultsTable, String filePathToExcelWorkbook, Object sheetObject, Boolean appendToSheet, Boolean skipHeadings) {
		
		ExcelUtils.setupLogger();
		
		ExcelUtils excelUtils = new ExcelUtils();
		
		filePathToExcelWorkbook = excelUtils.fixFilePath(filePathToExcelWorkbook);
		
		if (ijResultsTable == null) {
			excelUtils.noTableFoundMessage();
			return;	//stop if an ImageJ table cannot be determined 
		}
		logger.info("Current results table = " + ijResultsTable);
		
		Vector<String[]> tableVector = excelUtils.getImageJTableRowsAsVector(ijResultsTable);
	
		Workbook workbook = excelUtils.getWorkbook(filePathToExcelWorkbook);
		logger.info("Workbook = " + workbook);
		
		if (workbook == null) {
			logger.info("Aborting due to nonexisting workbook");
			return;	//stop processing if no workbook created or access blocked
		}
		
		String sheetObjectAsString = excelUtils.getSheetObjectAsString(sheetObject);
		
		if (sheetObjectAsString.equals("") && ijResultsTable != null) {
			sheetObjectAsString = ijResultsTable.getTitle();
			if (sheetObjectAsString.equals("")) {
				sheetObjectAsString = "Sheet";
			}
		}
		
		Sheet sheet = excelUtils.getSheet(workbook, sheetObjectAsString, appendToSheet);
		logger.info("Current sheet = " + sheet);
		logger.info("Current sheet name = " + sheet.getSheetName());
		

		logger.info("Append data = " + appendToSheet);
		logger.info("Skip headings = " + skipHeadings);
		

		int excelRowStartingIndex = 0;
		int excelColumnStartingIndex = 0;
		
		if (appendToSheet) {
			excelRowStartingIndex = (sheet.getLastRowNum() < 0) ? 0 : sheet.getLastRowNum() + 1;
			
			skipHeadings = (skipHeadings && excelRowStartingIndex > 0) ? true : false;
			
		}			
		logger.info("Row starting index = " + excelRowStartingIndex);
		logger.info("Skip data headings = " + skipHeadings);
		
		
		int rowIndexOffset = 0;
		for (int excelRowIndex = excelRowStartingIndex; excelRowIndex < (excelRowStartingIndex + tableVector.size()); excelRowIndex++) {
			
			int ijTableRowIndex = excelRowIndex - excelRowStartingIndex;
			
			if (skipHeadings && ijTableRowIndex == 0) {
				//skip headings when appending data
				rowIndexOffset = -1;
			} else {
				
				Row row = sheet.createRow(excelRowIndex + rowIndexOffset);

				for (int excelColumnIndex = excelColumnStartingIndex; excelColumnIndex < excelUtils.getVectorContentSize(tableVector); excelColumnIndex++) {
					
					Cell cell = row.createCell(excelColumnIndex);
					
					String value = tableVector.get(ijTableRowIndex)[excelColumnIndex];
					
					if (value.equals("NaN")) {
						value = "";
					}
					
					try {
						cell.setCellValue(Double.parseDouble(value));
					} catch (Exception e) {
						cell.setCellValue(value);
					}	
				}
			}
		}
						
		logger.info("Save " + workbook + " at " + filePathToExcelWorkbook);
		excelUtils.saveWorkbook(workbook, filePathToExcelWorkbook, CLOSE_WORKBOOK_AFTER_SAVING);	
	}


	
	
	
	
	@Override
	public void runFromMacro(Object[] parameters) {
		
		ExcelUtils.setupLogger();
		
		String filePath = (String)parameters[0];
		
		if (filePath.equals("") || filePath == null) {
			filePath = System.getProperty("user.home");
		}
		
		parameters[3] = ExcelUtils.readBooleanMacroInput(parameters[3]);
		parameters[4] = ExcelUtils.readBooleanMacroInput(parameters[4]);
		
		for (int p = 0; p < parameters.length; p++) {
			logger.info("Macro parameter " + p +" = " + parameters[p]);
		}
		
		saveTableAsWorkbookSheet(ResultsTable.getResultsTable(parameters[0].toString()), parameters[1].toString(), parameters[2], Boolean.valueOf(parameters[3].toString()), Boolean.valueOf(parameters[4].toString()));
		
	}



	/**
     * We need to define a list of parameter types
     * @return int array with parameter types as defined in the MacroExtension class.
     */
    @Override
    public int[] parameterTypes() {
        return new int[] { MacroExtension.ARG_STRING, MacroExtension.ARG_STRING, MacroExtension.ARG_STRING, MacroExtension.ARG_STRING, MacroExtension.ARG_STRING };
    }

    /**
     * We should define a description for users.
     * @return algorithm/parameter description
     */
    @Override
    public String description() {
        return "The active or otherwise specified table will be saved as a single sheet in an Excel workbook (.xlsx) file"
        		+ "under the specified name or the ImageJ table name otherwise.\n"
        		+ "Data can also be appended to existing data in a specified sheet if indicated with or without their headings.\n"
        		+ "If indicated the existing file will be overwritten or a new file will be created.";
    }


	/**
     * We should provide a user readable list of parameters
     * @return list of parameters
     */
    @Override
    public String parameters() {
        return "ijResultsTableTitle, filePathToExcelWorkbook, sheetNameOrIndex, appendDataToSheet, skipHeadingsWhenAppending";
    }
	
	
}
