/**
 * 
 */
package excel.functions.plugins;

import java.io.File;

import excel.functions.utils.ExcelUtils;
import ij.ImagePlus;
import ij.measure.ResultsTable;

/**
 * @author BioVoxxel
 *
 */
public class ExcelFunctions {
	
	/**
	 * Constructor to use the static Excel functions
	 */
	@SuppressWarnings("unused")
	public ExcelFunctions() {
		ExcelUtils utils = new ExcelUtils();
	}
	
	/**
	 * 
	 * @param ijResultsTable - instance of an ImageJ {@link ij.measure.ResultsTable}
	 * @param file - The file instance which holds the Excel workbook
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param appendData - set <i>false</i> to create a new worksheet for the current table or <i>true</i> to append data below existing ones on an already existing worksheet 
	 * @param skipHeadingsWhenAppending - set <i>false</i> to add table data headings or set <i>true</i> to skip headings when appending data to seamlessly fuse data to prior ones
	 */
	public static void saveTableToWorkbookSheet(ResultsTable ijResultsTable, File file, Object sheetNameOrIndex, Boolean appendData, Boolean skipHeadingsWhenAppending) {			
		SaveTableAsWorksheet.saveTableAsWorkbookSheet(ijResultsTable, file.getAbsolutePath(), sheetNameOrIndex, appendData, skipHeadingsWhenAppending);	
	}
	
	/**
	 * 
	 * @param ijResultsTable - instance of an ImageJ {@link ij.measure.ResultsTable}
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param appendData - set <i>false</i> to create a new worksheet for the current table or <i>true</i> to append data below existing ones on an already existing worksheet 
	 * @param skipHeadingsWhenAppending - set <i>false</i> to add table data headings or set <i>true</i> to skip headings when appending data to seamlessly fuse data to prior ones
	 */
	public static void saveTableToWorkbookSheet(ResultsTable ijResultsTable, String filePath, Object sheetNameOrIndex, Boolean appendData, Boolean skipHeadingsWhenAppending) {	
		SaveTableAsWorksheet.saveTableAsWorkbookSheet(ijResultsTable, filePath, sheetNameOrIndex, appendData, skipHeadingsWhenAppending);
	}
	
	/**
	 * 
	 * @param ijTableName - String name of the ImageJ results table 
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param appendData - set <i>false</i> to create a new worksheet for the current table or <i>true</i> to append data below existing ones on an already existing worksheet 
	 * @param skipHeadingsWhenAppending - set <i>false</i> to add table data headings or set <i>true</i> to skip headings when appending data to seamlessly fuse data to prior ones
	 */
	public static void saveTableToWorkbookSheet(String ijTableName, String filePath, Object sheetNameOrIndex, Boolean appendData, Boolean skipHeadingsWhenAppending) {	
		SaveTableAsWorksheet.saveTableAsWorkbookSheet(ResultsTable.getResultsTable(ijTableName), filePath, sheetNameOrIndex, appendData, skipHeadingsWhenAppending);	
	}
	
	/**
	 * 
	 * @param file - The file instance which holds the Excel workbook
	 * @param appendData
	 * @param skipHeadingsWhenAppending
	 */
	public static void saveAllOpenTablesToWorkbook(File file, Boolean appendData, Boolean skipHeadingsWhenAppending) {
		SaveAllTablesToWorkbook.saveAllOpenTablesAsWorkbookSheets(file.getAbsolutePath(), appendData, skipHeadingsWhenAppending);
	}

	/**
	 * 
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param appendData - set <i>false</i> to create a new worksheet for the current table or <i>true</i> to append data below existing ones on an already existing worksheet 
	 * @param skipHeadingsWhenAppending - set <i>false</i> to add table data headings or set <i>true</i> to skip headings when appending data to seamlessly fuse data to prior ones
	 */
	public static void saveAllOpenTablesToWorkbook(String filePath, Boolean appendData, Boolean skipHeadingsWhenAppending) {
		SaveAllTablesToWorkbook.saveAllOpenTablesAsWorkbookSheets(filePath, appendData, skipHeadingsWhenAppending);
	}
	
	/**
	 * This function adds an image open in ImageJ / Fiji as PNG to the defined worksheet in a defined Excel workbook
	 * <i>Note: the size of images added to Excel cannot be properly controlled via the Apache POI library. 
	 * Therefore, images normally appear distorted in Excel. Using the image properties in Excel and pressing "Reset" restores the correct aspect ratio.</i> 
	 * 
	 * @param imagePlus
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param column - 0-based column of the cell which should be the location of the upper left corner of the added image
	 * @param row - 0-based row of the cell which should be the location of the upper left corner of the added image
	 */
	public static void addImageToWorkbookSheet(ImagePlus imagePlus, String filePath, Object sheetNameOrIndex, int column, int row) {
		AddImageToWorksheet.addImageToWorksheet(imagePlus, filePath, sheetNameOrIndex, column, row);
	}
	
	
	/**
	 * 
	 * @param array - Number[] array which should be added to the specified Row
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param startingColumn - 0-based starting column from which onwards the array data are added to the cells of the first empty row in the sheet
	 */
	public static void appendArrayAsExcelRow(Number[] array, String filePath, Object sheetNameOrIndex, int startingColumn) {
				
			String[] stringArray = new String[array.length];
			
			for (int index = 0; index < stringArray.length; index++) {
				stringArray[index] = array[index].toString();
			}
			
			AppendArrayAsExcelRow.appendArrayAsTableRow(stringArray, filePath, sheetNameOrIndex, startingColumn);
	
	}
	
	
	/**
	 * 
	 * @param array - String[] array which should be added to the specified Row
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param startingColumn - 0-based starting column from which onwards the array data are added to the cells of the first empty row in the sheet
	 */
	public static void appendArrayAsExcelRow(String[] array, String filePath, Object sheetNameOrIndex, int startingColumn) {
		AppendArrayAsExcelRow.appendArrayAsTableRow(array, filePath, sheetNameOrIndex, startingColumn);
	}
	
	
	/**
	 * 
	 * @param array - Numbers[] array which should be added to the specified Row
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param startingRow - 0-based row at which the first entry of the provided array is added to the cells of the first empty column in the sheet
	 * @param columnDataFormat - optional data format for that column (options can be found under {@link org.apache.poi.ss.usermodel.BuiltinFormats}).
	 * This can be an empty String "" to use the setting "General", the default in Excel.
	 */
	public static void appendArrayAsExcelColumn(Number[] array, String filePath, Object sheetNameOrIndex, int startingRow, String columnDataFormat) {
		
		String[] stringArray = new String[array.length];
		
		for (int index = 0; index < stringArray.length; index++) {
			stringArray[index] = array[index].toString();
		}
		
		AppendArrayAsExcelColumn.appendArrayAsTableColumn(stringArray, filePath, sheetNameOrIndex, startingRow, columnDataFormat);
	}
	
	
	/**
	 * 
	 * @param array - String[] array which should be added to the specified Row
	 * @param filePath - the String file path including the extension ".xlsx" to the Excel workbook file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param startingRow - 0-based row at which the first entry of the provided array is added to the cells of the first empty column in the sheet
	 * @param columnDataFormat - optional data format for that column (options can be found under {@link org.apache.poi.ss.usermodel.BuiltinFormats}).
	 * This can be an empty String "" to use the setting "General", the default in Excel.
	 */
	public static void appendArrayAsExcelColumn(String[] array, String filePath, Object sheetNameOrIndex, int startingRow, String columnDataFormat) {
		AppendArrayAsExcelColumn.appendArrayAsTableColumn(array, filePath, sheetNameOrIndex, startingRow, columnDataFormat);
	}
	
	/**
	 * Allowed data formats can be found under:
	 * {@link org.apache.poi.ss.usermodel.BuiltinFormats}
	 * 
	 * 
	 * @param filePath - file path to an .xlsx worksheet file
	 * @param sheetNameOrIndex - name of the worksheet or the 0-based index of an existing sheet (the String keywords "FIRST" or "LAST" retrieve the corresponding sheets)
	 * @param columnNumber - the 0-based column number to which the format change should be applied
	 * @param dataFormatString - the actual format string as found in the BuiltinFormats
	 */
	public static void setColumnDataFormat(String filePath, Object sheetNameOrIndex, Integer columnNumber, String dataFormatString) {
		SetColumnDataFormat.setColumnDataFormat(filePath, sheetNameOrIndex, columnNumber, dataFormatString);
	}
	
}
