/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine.util;

import com.google.appengine.tools.admin.AppCfg;

import javax.annotation.Nonnull;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.util.List;

/**
 * Thread for Running the {@link com.google.appengine.tools.admin.AppCfg} command.
 */
class PasswordAppCfgMainThread extends Thread {
    /**
     * Handler for getting the actual maven logger.
     */
    @Nonnull
    private final LogHandler log;
    /**
     * Parent for all threads created by AppCfg.
     */
    @Nonnull
    private final ThreadGroup threads;
    /**
     * Password in plain text.
     */
    @Nonnull
    private final String password;
    /**
     * Arguments for {@link com.google.appengine.tools.admin.AppCfg}.
     */
    @Nonnull
    private final String[] args;

    /**
     * Constructor for the AppCfg-Thread.
     *
     * @param logHandler  maven logging accessor
     * @param threadGroup parent for all threads created by AppCfg
     * @param pw          password in plain text
     * @param arguments   arguments for {@link com.google.appengine.tools.admin.AppCfg}
     */
    PasswordAppCfgMainThread(@Nonnull final LogHandler logHandler, @Nonnull final ThreadGroup threadGroup, @Nonnull final String pw, @Nonnull final List<String> arguments) {
        super(threadGroup, "AppCfgMainThread");
        log = logHandler;
        threads = threadGroup;
        password = pw;
        args = arguments.toArray(new String[arguments.size()]);
    }

    @Override
    public final void run() {
        final PrintStream outOrig = System.out;
        final InputStream inOrig = System.in;

        final PipedInputStream inReplace = new PipedInputStream();
        final OutputStream stdin;
        try {
            stdin = new PipedOutputStream(inReplace);
        } catch (final IOException e) {
            log.getLog().error("Unable to redirect input", e);
            return;
        }
        System.setIn(inReplace);

        try {
            final BufferedWriter stdinWriter = new BufferedWriter(new OutputStreamWriter(stdin, "ASCII"));

            System.setOut(new PrintStream(new PasswordExpectOutputStream(threads, outOrig, new Runnable() {
                @Override
                public void run() {
                    try {
                        stdinWriter.write(password);
                        stdinWriter.newLine();
                        stdinWriter.flush();
                    } catch (final IOException e) {
                        log.getLog().error("Unable to enter password", e);
                    }
                }
            }), true, "ASCII"));

            AppCfg.main(args);
        } catch (UnsupportedEncodingException e) {
            log.getLog().error(e);
        } finally {
            System.setOut(outOrig);
            System.setIn(inOrig);
        }
    }
}
