/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine.util;

import com.google.common.collect.Iterables;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.ArtifactUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.UnArchiver;
import org.codehaus.plexus.archiver.manager.ArchiverManager;
import org.codehaus.plexus.archiver.manager.NoSuchArchiverException;
import org.codehaus.plexus.util.FileUtils;
import org.sonatype.aether.RepositorySystem;
import org.sonatype.aether.RepositorySystemSession;
import org.sonatype.aether.repository.RemoteRepository;
import org.sonatype.aether.resolution.ArtifactRequest;
import org.sonatype.aether.resolution.ArtifactResolutionException;
import org.sonatype.aether.resolution.ArtifactResult;
import org.sonatype.aether.resolution.VersionRangeRequest;
import org.sonatype.aether.resolution.VersionRangeResolutionException;
import org.sonatype.aether.resolution.VersionRangeResult;
import org.sonatype.aether.util.artifact.DefaultArtifact;
import org.sonatype.aether.version.Version;

import java.io.File;
import java.util.List;

/**
 * Helper class for finding the SDK directory when not set from plugin parameter. The right SDK version will be
 * resolved, downloaded and extracted.
 *
 * @author Michael Cramer
 * @version 1.7.5
 * @since 1.7.5
 */
public final class AppengineSdkFinder {
    /**
     * Maven artifact groupId for the appengine-sdk.
     */
    private static final String SDK_GROUP_ID = "com.google.appengine";
    /**
     * Maven artifact artifactId for the appengine-sdk.
     */
    private static final String SDK_ARTIFACT_ID = "appengine-java-sdk";
    /**
     * Maven artifact extension for the appengine-sdk.
     */
    private static final String SDK_EXTENSION = "zip";

    /**
     * Private constructor to prevent instantiation.
     */
    private AppengineSdkFinder() {
    }

    /**
     * Resolve the location of the Appengine SDK.
     *
     * @param project         Maven project
     * @param repoSystem      RepositorySystem
     * @param repoSession     RepositorySystemSession
     * @param repositories    Repositories to include in the lookup
     * @param archiverManager ArchiveManager for extracting the SDK
     * @return Path to the SDK
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    public static File getSdk(final MavenProject project, final RepositorySystem repoSystem, final RepositorySystemSession repoSession, final List<RemoteRepository> repositories, final ArchiverManager archiverManager) throws MojoExecutionException {
        final Artifact artifact = Iterables.find(project.getPluginArtifacts(), new AppengineMavenPluginPredicate());
        // remove the SNAPSHOT extension
        String sdkVersion = artifact.getVersion();
        if (ArtifactUtils.isSnapshot(sdkVersion)) {
            sdkVersion = sdkVersion.replace("-" + Artifact.SNAPSHOT_VERSION, "");
        }
        return getSdk(sdkVersion, repoSystem, repoSession, repositories, archiverManager);
    }

    /**
     * Lookup and find the latest existing Version from a VersionRangeRequest.
     *
     * @param version     maximum version to look for
     * @param repoSystem  RepositorySystem
     * @param repoSession RepositorySystemSession
     * @param allRepos    Repositories to include in the lookup
     * @return latest version as String
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    private static String getLatestVersion(final String version, final RepositorySystem repoSystem, final RepositorySystemSession repoSession, final List<RemoteRepository> allRepos) throws MojoExecutionException {
        final VersionRangeResult rangeResult;
        try {
            final VersionRangeRequest rangeRequest = new VersionRangeRequest();
            rangeRequest.setArtifact(new DefaultArtifact(SDK_GROUP_ID + ':' + SDK_ARTIFACT_ID + ":[0," + version + ']')).setRepositories(allRepos);
            rangeResult = repoSystem.resolveVersionRange(repoSession, rangeRequest);
        } catch (final VersionRangeResolutionException e) {
            throw new MojoExecutionException("Could not resolve versions of the App Engine Java SDK", e);
        }
        final Version highestVersion = rangeResult.getHighestVersion();
        if (highestVersion == null) {
            throw new MojoExecutionException("Could not find any Versions of the App Engine Java SDK");
        }
        return highestVersion.toString();
    }

    /**
     * Finds the latest SDK, extract it to the local repository and returns the path to this location.
     *
     * @param version         maximum version for the SDK
     * @param repoSystem      RepositorySystem
     * @param repoSession     RepositorySystemSession
     * @param allRepos        Repositories to include in the lookup
     * @param archiverManager ArchiveManager for extracting the SDK
     * @return Directory of the extracted SDK
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    public static File getSdk(final String version, final RepositorySystem repoSystem, final RepositorySystemSession repoSession, final List<RemoteRepository> allRepos, final ArchiverManager archiverManager) throws MojoExecutionException {
        final String latestVersion = getLatestVersion(version, repoSystem, repoSession, allRepos);

        final ArtifactResult result;
        try {
            final ArtifactRequest request = new ArtifactRequest(new DefaultArtifact(SDK_GROUP_ID, SDK_ARTIFACT_ID, SDK_EXTENSION, latestVersion), allRepos, null);
            result = repoSystem.resolveArtifact(repoSession, request);
        } catch (ArtifactResolutionException e) {
            throw new MojoExecutionException("Could not resolve SDK artifact in Maven.", e);
        }

        final File sdkArchive = result.getArtifact().getFile();
        final File sdkRepoDir = sdkArchive.getParentFile();
        final File sdkBaseDir = new File(sdkRepoDir, SDK_ARTIFACT_ID);

        if (sdkBaseDir.exists()) {
            if (!sdkBaseDir.isDirectory()) {
                throw new MojoExecutionException("Could not unpack the SDK because there is something located here: " + sdkRepoDir.getAbsolutePath());
            }
        } else {
            FileUtils.mkdir(sdkBaseDir.getAbsolutePath());
        }

        try {
            final UnArchiver unArchiver = archiverManager.getUnArchiver(sdkArchive);
            unArchiver.setSourceFile(sdkArchive);
            unArchiver.setDestDirectory(sdkBaseDir);
            unArchiver.extract();
            return new File(sdkBaseDir, result.getArtifact().getArtifactId() + "-" + result.getArtifact().getVersion());
        } catch (final NoSuchArchiverException e) {
            throw new MojoExecutionException("Unable to extract the SDK", e);
        }
    }
}
