/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine.devserver;

import com.google.appengine.tools.KickStart;
import com.google.appengine.tools.development.DevAppServerMain;
import com.google.common.base.Joiner;
import de.bigmichi1.appengine.AbstractBaseMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Mojo for running the DevAppServer.
 *
 * @author Michael Cramer
 * @version 1.7.5
 * @since 1.7.5
 */
public abstract class AbstractDevServerMojo extends AbstractBaseMojo {
    /**
     * The server to use to determine the latest SDK version.<br/><br/>
     * If not the value is not set the default {@link com.google.appengine.tools.info.SdkInfo#getDefaultServer()} will
     * be used by the Server.
     */
    @Parameter(property = "appengine.server")
    private String server;
    /**
     * The address of the interface on the local machine to bind to (or 0.0.0.0 for all interfaces).<br/><br/>
     * If not the value is not set the default
     * {@link com.google.appengine.tools.development.DevAppServer#DEFAULT_HTTP_ADDRESS} will be used by the Server.
     */
    @Parameter(property = "appengine.address")
    private String address;
    /**
     * The port number to bind to on the local machine.<br/><br/>
     * If not the value is not set the default
     * {@link com.google.appengine.tools.development.DevAppServer#DEFAULT_HTTP_PORT} will be used by the Server.
     */
    @Parameter(property = "appengine.port")
    private String port;
    /**
     * Additional flags to the JVM used to run the dev server.
     */
    @Parameter(property = "appengine.jvmFlags")
    private List<String> jvmFlags;
    /**
     * Disable the check for newer SDK versions.
     */
    @Parameter(property = "appengine.disableUpdateCheck")
    private boolean disableUpdateCheck;

    /**
     * {@inheritDoc}
     */
    @Override
    public final void execute() throws MojoExecutionException {
        final List<String> params = new ArrayList<String>();

        params.addAll(collectMojoParams());
        params.addAll(collectAdditionalMojoParams());

        final List<String> devAppServerCommand = buildDevAppServerCommand(params);

        getLog().info("");
        getLog().info("Running DevAppServer with KickStart");
        getLog().info("");
        getLog().debug("Running with following parameters:");
        getLog().debug(Joiner.on(" ").join(devAppServerCommand));
        getLog().debug("");

        KickStart.main(devAppServerCommand.toArray(new String[devAppServerCommand.size()]));
    }

    /**
     * Build a list with parameters from the plugin configuration.
     *
     * @return List with parameters from plugin configuration
     */
    private List<String> collectMojoParams() {
        final List<String> mojoParams = new ArrayList<String>(4);

        if (StringUtils.isNotBlank(server)) {
            mojoParams.add(String.format("--server=%s", server));
        }

        if (StringUtils.isNotBlank(address)) {
            mojoParams.add(String.format("--address=%s", address));
        }

        if (StringUtils.isNotBlank(port)) {
            mojoParams.add(String.format("--port=%s", port));
        }

        if (disableUpdateCheck) {
            mojoParams.add("--disable_update_check");
        }

        return mojoParams;
    }

    /**
     * Builds the entire command with all specified params to start the DevAppServer.
     *
     * @param params collected parameters from configuration
     * @return all parts of the command in the right order
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    private List<String> buildDevAppServerCommand(final List<String> params) throws MojoExecutionException {
        final String sdkBaseDir = getSdkFile();

        final List<String> devAppServerCommand = new ArrayList<String>(5);

        // hack for getting appengine-tools-api.jar on a runtime classpath
        // (KickStart checks java.class.path system property for classpath entries)
        final String classpath = System.getProperty("java.class.path");
        final String toolsJar = sdkBaseDir + File.separator + "lib" + File.separator + "appengine-tools-api.jar";
        if (!classpath.contains(toolsJar)) {
            System.setProperty("java.class.path", classpath + File.pathSeparator + toolsJar);
        }

        // add the main class from the DevAppServer
        devAppServerCommand.add(DevAppServerMain.class.getCanonicalName());

        // Add additional jvm flags
        if (jvmFlags != null) {
            for (final String jvmFlag : jvmFlags) {
                if (StringUtils.isNotBlank(jvmFlag)) {
                    devAppServerCommand.add("--jvm_flag=" + jvmFlag);
                }
            }
        }

        // add the sdkRootDir parameter from maven configuration
        devAppServerCommand.add("--sdk_root=" + sdkBaseDir);

        // add additional configuration params from maven plugin
        devAppServerCommand.addAll(params);

        devAppServerCommand.add(getApplicationDirectory());

        return devAppServerCommand;
    }

    /**
     * Get additional configuration params from the specific goals that extend this class.
     *
     * @return additional configuration parameters
     */
    protected abstract List<String> collectAdditionalMojoParams();
}
