/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine.util;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

/**
 * Read writes only from ThreadGroup, from other threads just redirect to PrintStream. This method is a copy and
 * slightly modified version of the one that is used in the "maven-gae-plugin".
 *
 * @author vlads
 * @author Michael Cramer
 * @version 1.7.5
 * @since 1.7.5
 */
class PasswordExpectOutputStream extends OutputStream {
    /**
     * OutputStream to which is written.
     */
    @Nonnull
    private final PrintStream out;
    /**
     * ThreadGroup for which the stream should do something.
     */
    @Nonnull
    private final ThreadGroup threads;
    /**
     * Runnable that writes the password to the OutputStream.
     */
    @Nonnull
    private final Runnable onExpected;
    /**
     * Expected password prompt text.
     */
    @Nonnull
    private final byte[] expect;
    /**
     * Matched bytes for finding the password prompt.
     */
    private int match = 0;

    /**
     * Constructor for the password output stream.
     *
     * @param threadGroup    ThreadGroup for which the stream should do something
     * @param printStreamOut OutputStream to which is written
     * @param runnable       Runnable that writes the password to the OutputStream
     */
    PasswordExpectOutputStream(@Nonnull final ThreadGroup threadGroup, @Nonnull final PrintStream printStreamOut, @Nonnull final Runnable runnable) {
        threads = threadGroup;
        out = printStreamOut;
        onExpected = runnable;
        try {
            expect = "Password for".getBytes("ASCII");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Find the right thread from that belongs to the ThreadGroup.
     *
     * @return current ThreadGroup is the expected one
     */
    private boolean isRedirectThread() {
        ThreadGroup tg = Thread.currentThread().getThreadGroup();
        while (!threads.equals(tg) && tg != null) {
            tg = tg.getParent();
        }
        return threads.equals(tg);
    }

    /**
     * {@inheritDoc}
     *
     * @param b a int.
     * @throws java.io.IOException if any
     */
    public final void write(final int b) throws IOException {
        if (isRedirectThread()) {
            expect((byte) (0xFF & b));
        }
        out.write(b);
    }

    /**
     * Writes the password when prompted to console in a separate Thread.
     *
     * @param b starting byte
     */
    private synchronized void expect(final byte b) {
        if (expect[match] == b) {
            match++;
            if (match == expect.length) {
                match = 0;
                final Thread t = new Thread(onExpected, "EnterPasswordThread");
                t.setDaemon(true);
                t.start();
            }
        } else {
            match = 0;
        }
    }
}
