/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine.appcfg;

import com.google.appengine.tools.admin.AppCfg;
import com.google.appengine.tools.info.SdkInfo;
import com.google.common.base.Joiner;
import de.bigmichi1.appengine.AbstractBaseMojo;
import de.bigmichi1.appengine.util.LogHandler;
import de.bigmichi1.appengine.util.PasswordUtil;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.settings.Server;
import org.codehaus.plexus.util.StringUtils;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcherException;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;

/**
 * Base for all AppCfg-Mojos.
 *
 * @author Michael Cramer
 * @version 1.7.5
 * @since 1.7.5
 */
public abstract class AbstractBaseAppCfgMojo extends AbstractBaseMojo implements LogHandler {
    /**
     * Component for decrypting a password supplied in password property in settings.xml.
     */
    @Component
    private SecDispatcher secDispatcher;
    /**
     * The server to connect to.
     */
    @Parameter
    private String server;
    /**
     * The username to use. Will prompt if omitted.
     */
    @Parameter
    private String email;
    /**
     * Overrides the Host header sent with all RPCs.
     */
    @Parameter
    private String host;
    /**
     * Proxies requests through the given proxy server. If {@link #proxyHttps} is also set, only HTTP will be proxied here, otherwise both HTTP and HTTPS will.
     */
    @Parameter
    private String proxyHttp;
    /**
     * Proxies HTTPS requests through the given proxy server.
     */
    @Parameter
    private String proxyHttps;
    /**
     * Do not save/load access credentials to/from disk.
     */
    @Parameter(defaultValue = "false")
    private boolean noCookies;
    /**
     * Always read the login password from stdin.
     */
    @Parameter(defaultValue = "false")
    private boolean passin;
    /**
     * Do not use HTTPS to communicate with the Admin Console.
     */
    @Parameter(defaultValue = "false")
    private boolean insecure;
    /**
     * Do not verify the SSL certificate matches the hostname of the Admin Console.
     */
    @Parameter(defaultValue = "false")
    private boolean ignoreBadCert;
    /**
     * Override application id from appengine-web.xml.
     */
    @Parameter
    private String appId;
    /**
     * Override (major) version from appengine-web.xml.
     */
    @Parameter
    private String appVersion;
    /**
     * Use OAuth2 instead of password auth.
     */
    @Parameter(defaultValue = "true")
    private boolean oauth2;
    /**
     * Use the App Engine Java 7 runtime for this app.
     */
    @Parameter(defaultValue = "true")
    private boolean useJava7;
    /**
     * Id of the server configuration in the settings.xml.
     */
    @Parameter
    private String serverId;

    /**
     * Constructor.
     */
    protected AbstractBaseAppCfgMojo() {
    }

    /**
     * Build a list with parameters from the plugin configuration.
     *
     * @return List with parameters from plugin configuration
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    @Nonnull
    private List<String> collectMojoParams() throws MojoExecutionException {
        final List<String> mojoParams = new ArrayList<String>(14);

        if (StringUtils.isNotBlank(server)) {
            mojoParams.add(String.format("--server=%s", server));
        }

        if (StringUtils.isNotBlank(host)) {
            mojoParams.add(String.format("--host=%s", host));
        }

        addProxyParameter(mojoParams, "http", "--proxy", proxyHttp);

        addProxyParameter(mojoParams, "https", "--proxy_https", proxyHttps);

        if (noCookies) {
            mojoParams.add("--no_cookies");
        }

        final String sdkRootDirectory = getSdkFile();
        mojoParams.add(String.format("--sdk_root=%s", sdkRootDirectory));
        // hack to work around the automatic discovery
        System.setProperty(SdkInfo.SDK_ROOT_PROPERTY, sdkRootDirectory);


        if (passin) {
            mojoParams.add("--passin");
        }

        if (insecure) {
            mojoParams.add("--insecure");
        }

        if (ignoreBadCert) {
            mojoParams.add("--ignore_bad_cert");
        }

        if (StringUtils.isNotBlank(appId)) {
            mojoParams.add(String.format("--application=%s", appId));
        }

        if (StringUtils.isNotBlank(appVersion)) {
            mojoParams.add(String.format("--version=%s", appVersion));
        }

        if (useJava7) {
            mojoParams.add("--use_java7");
        }
        return mojoParams;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final void execute() throws MojoExecutionException {
        addLogMessages();

        final List<String> params = new ArrayList<String>(2);

        params.addAll(collectMojoParams());
        params.addAll(collectActionParams());

        executeAppCfg(params);
    }

    /**
     * executing the {@link AppCfg} command with either the standard wa or in a separate Thread to provide fill in the
     * provided password automatically to the password prompt. Also some missing parameters are added if not set in the
     * plugin configuration to get the expected behavior.
     *
     * @param arguments arguments for {@link AppCfg}
     * @see PasswordUtil#forkPasswordExpectThread(java.util.List, String)
     */
    private void executeAppCfg(@Nonnull final List<String> arguments) {
        if (oauth2) {
            arguments.add("--oauth2");
        } else {
            final Server serverFromSettings = getServerFromSettings(serverId);
            if (serverFromSettings != null && StringUtils.isNotBlank(serverFromSettings.getPassword()) && StringUtils.isNotBlank(serverFromSettings.getUsername())) {
                getLog().info("Use Settings configuration from server id {" + serverId + "}");
                arguments.add(String.format("--email=%s", serverFromSettings.getUsername()));
                if (!passin) {
                    arguments.add("--passin");
                }
                final PasswordUtil thread = new PasswordUtil(this);
                runAppCfg(arguments, new AppCfgRunner() {
                    @Override
                    public void run(@Nonnull final List<String> arguments) {
                        try {
                            final String decrypt = secDispatcher.decrypt(serverFromSettings.getPassword());
                            thread.forkPasswordExpectThread(arguments, decrypt);
                        } catch (SecDispatcherException e) {
                            getLog().error("Decryption of password failed.", e);
                        }
                    }
                });
                return;
            } else {
                if (StringUtils.isNotBlank(email)) {
                    arguments.add(String.format("--email=%s", email));
                }
            }
        }
        runAppCfg(arguments, new AppCfgRunner() {
            @Override
            public void run(@Nonnull final List<String> arguments) {
                AppCfg.main(arguments.toArray(new String[arguments.size()]));
            }
        });

    }

    /**
     * Running the AppCfg in a separate {@link AppCfgRunner}. Also add the action name and application directory as the
     * last parameters to the argument list.
     *
     * @param arguments arguments for {@link AppCfg}
     * @param runner    runner for the {@link AppCfg} command
     */
    private void runAppCfg(@Nonnull final List<String> arguments, @Nonnull final AppCfgRunner runner) {
        arguments.add(getActionName());
        arguments.add(getApplicationDirectory());

        getLog().debug("");
        getLog().debug("Running AppCfg with following parameters:");
        getLog().debug(Joiner.on(" ").join(arguments));
        getLog().debug("");

        runner.run(arguments);
    }

    /**
     * Return the name of the action for the AppCfg command.
     *
     * @return name of the action
     */
    @Nonnull
    protected abstract String getActionName();

    /**
     * Prints additional messages for the specific command during execution.
     *
     * @see #getLog()
     */
    protected abstract void addLogMessages();

    /**
     * Build a list with special parameters for the action.
     *
     * @return List with parameters from plugin configuration
     */
    @Nonnull
    protected abstract List<String> collectActionParams();
}
