/*
 * #%L
 * A Maven Plugin for the Google App Engine
 * %%
 * Copyright (C) 2013 None
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package de.bigmichi1.appengine;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import de.bigmichi1.appengine.util.AppengineSdkFinder;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.codehaus.plexus.archiver.manager.ArchiverManager;
import org.codehaus.plexus.util.StringUtils;
import org.sonatype.aether.RepositorySystem;
import org.sonatype.aether.RepositorySystemSession;
import org.sonatype.aether.repository.RemoteRepository;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.File;
import java.util.List;

/**
 * Base for all Mojos.
 *
 * @author Michael Cramer
 * @version 1.7.5
 * @since 1.7.5
 */
public abstract class AbstractBaseMojo extends AbstractMojo {
    /**
     * Component for reading settings from settings.xml.
     */
    @Component
    private Settings settings;
    /**
     * The current repository/network configuration of Maven.
     */
    @Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
    private RepositorySystemSession repoSession;
    /**
     * The maven project.
     */
    @Parameter(property = "project", readonly = true, required = true)
    private MavenProject project;
    /**
     * To look up Archiver/UnArchiver implementations.
     */
    @Component
    private ArchiverManager archiverManager;
    /**
     * The entry point to Aether, i.e. the component doing all the work.
     */
    @Component
    private RepositorySystem repoSystem;
    /**
     * The project's remote repositories to use for the resolution of project dependencies.
     */
    @Parameter(defaultValue = "${project.remoteProjectRepositories}", readonly = true)
    private List<RemoteRepository> projectRepos;
    /**
     * The project's remote repositories to use for the resolution of plugins and their dependencies.
     */
    @Parameter(defaultValue = "${project.remotePluginRepositories}", readonly = true)
    private List<RemoteRepository> pluginRepos;
    /**
     * Override the appengine sdk version. When not set sdkVersion it will be determined from the plugin version.
     * If a version specified higher than a released version of the sdk the latest available version is used, but not
     * higher than the specified version.
     */
    @Parameter
    private String sdkVersion;
    /**
     * Overrides where the SDK is located.<br/><br/>
     * If the value is not set, the SDK is downloaded from maven, extracted to the local repository and the
     * sdkRootDir parameter set to this location.
     */
    @Parameter
    private String sdkRootDir;
    /**
     * Path to the appengine application directory.<br/><br/>
     * If not set the default of build-directory + final-name is used.
     */
    @Parameter
    private String appDir;

    /**
     * Determine the path to the SDK, either the specified in {@link #sdkRootDir} Parameter or the downloaded and
     * extracted one based on plugin version or {@link #sdkVersion}.
     *
     * @return absolute path to the SDK
     * @throws org.apache.maven.plugin.MojoExecutionException
     *          if any
     */
    protected final String getSdkFile() throws MojoExecutionException {
        final File sdkBaseDir;
        if (StringUtils.isNotBlank(sdkRootDir)) {
            sdkBaseDir = new File(sdkRootDir);
        } else {
            final List<RemoteRepository> repositories = ImmutableList.copyOf(Iterables.concat(pluginRepos, projectRepos));
            if (StringUtils.isNotBlank(sdkVersion)) {
                sdkBaseDir = AppengineSdkFinder.getSdk(sdkVersion, repoSystem, repoSession, repositories, archiverManager);
            } else {
                sdkBaseDir = AppengineSdkFinder.getSdk(project, repoSystem, repoSession, repositories, archiverManager);
            }
        }
        return sdkBaseDir.getAbsolutePath();
    }

    /**
     * Determine the application directory.
     *
     * @return path to the application
     */
    protected final String getApplicationDirectory() {
        if (StringUtils.isNotBlank(appDir)) {
            return appDir;
        } else {
            return project.getBuild().getDirectory() + File.separator + project.getBuild().getFinalName();
        }
    }

    /**
     * Adds a proxy parameter that are passed to the AppCfg options.
     *
     * @param mojoParams             parameter collection to add the proxy parameter
     * @param protocolName           name of the protocol
     * @param appCfgParameterName    name of the AppCfg parameter
     * @param configurationParameter plugin configuration parameter
     */
    protected final void addProxyParameter(final List<String> mojoParams, final String protocolName, final String appCfgParameterName, final String configurationParameter) {
        if (configurationParameter != null && !configurationParameter.isEmpty()) {
            mojoParams.add(String.format("%s=%s", appCfgParameterName, configurationParameter));
        } else {
            final Proxy activeProxy = settings.getActiveProxy();
            if (activeProxy != null && StringUtils.isNotBlank(activeProxy.getProtocol()) && protocolName.equalsIgnoreCase(activeProxy.getProtocol())) {
                mojoParams.add(String.format("%s=%s:%d", appCfgParameterName, activeProxy.getHost(), activeProxy.getPort()));
            }
        }
    }

    /**
     * Get a server configuration from the settings.xml file for the given serverId.
     *
     * @param serverId Id for the server configuration setting
     * @return server configuration if found
     */
    @Nullable
    protected final Server getServerFromSettings(@Nonnull final String serverId) {
        Preconditions.checkNotNull(serverId, "serverId is empty");
        return settings.getServer(serverId);
    }
}
