package de.alx_development.preferences;

/*-
 * #%L
 * Application base feature library
 * %%
 * Copyright (C) 2013 - 2021 ALX-Development
 * %%
 * This file is part of the de.alx-development.application library.
 * 
 * The application library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 * 
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Diese Datei ist Teil der de.alx-development.application Bibliothek.
 * 
 * Die Application-Bibliothek ist Freie Software: Sie können sie unter den
 * Bedingungen der GNU Lesser General Public License, wie von der
 * Free Software Foundation, Version 3 der Lizenz oder (nach Ihrer Wahl)
 * jeder neueren veröffentlichten Version, weiter verteilen und/oder modifizieren.
 * 
 * Diese Bibliothek wird in der Hoffnung, dass sie nützlich sein wird, aber
 * OHNE JEDE GEWÄHRLEISTUNG, bereitgestellt; sogar ohne die implizite
 * Gewährleistung der MARKTFÄHIGKEIT oder EIGNUNG FÜR EINEN BESTIMMTEN ZWECK.
 * Siehe die GNU Lesser General Public License für weitere Details.
 * 
 * Sie sollten eine Kopie der GNU Lesser General Public License zusammen mit diesem
 * Programm erhalten haben. Wenn nicht, siehe <https://www.gnu.org/licenses/>.
 * 
 * Copyright (c) 2019 ALX-Development
 * https://www.alx-development.de/
 * #L%
 */

import de.alx_development.application.languages.Translator;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.text.MessageFormat;
import java.util.logging.Logger;
import java.util.prefs.Preferences;

public class PreferenceDialog extends JDialog {

    /**
     * Internal logger instance to handle log messages in a unique way
     */
    private static final Logger logger = Logger.getLogger(PreferenceTreeModel.class.getName());

    /**
     * Creates a modeless dialog
     * with the specified owner frame.  If {@code owner}
     * is {@code null}, a shared, hidden frame will be set as the
     * owner of the dialog.
     *
     * @param owner the {@code Frame} from which the dialog is displayed
     * @throws HeadlessException If instantiated in headless mode
     * @see java.awt.GraphicsEnvironment#isHeadless
     * @see JComponent#getDefaultLocale
     */
    public PreferenceDialog(Frame owner) throws HeadlessException {
        super(owner, Translator.getInstance().getLocalizedString("CONFIG"), true);

        // Preference manager for the components persistent preferences
        Preferences preferences = Preferences.userNodeForPackage(getClass());

        // Loading look and feel definitions and other preferences
        setSize(preferences.getInt("SIZE_WIDTH", 400), preferences.getInt("SIZE_HEIGHT", 300));
        setLocation(preferences.getInt("POSITION_X", 50), preferences.getInt("POSITION_Y", 50));

        /*
         * Adding additional component listener to be informed about
         * window size and location changes which should be stored
         * in the properties.
         */
        addComponentListener(new ComponentListener() {

            /**
             * Storing the new dimension in the preferences
             * @param event The components change event
             */
            public void componentResized(ComponentEvent event) {
                preferences.putInt("SIZE_HEIGHT", event.getComponent().getHeight());
                preferences.putInt("SIZE_WIDTH", event.getComponent().getWidth());
            }

            public void componentMoved(ComponentEvent event) {
                preferences.putInt("POSITION_X", event.getComponent().getX());
                preferences.putInt("POSITION_Y", event.getComponent().getY());
            }

            /*
             * These events are ignored, cause the application window
             * should be displayed every time.
             */
            public void componentShown(ComponentEvent event) {
            }

            public void componentHidden(ComponentEvent event) {
            }
        });

        // Setting up the desired layoutmanager
        setLayout(new BorderLayout());

        JButton closeButton = new JButton("Close");
        JTree preferencesTree = new PreferenceTree();
        JTable preferencesEditor = new JTable();

        preferencesTree.setCellRenderer(new PreferenceTreeCellRenderer());

        JSplitPane jSplitPane;
        jSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, preferencesTree, add(new JScrollPane(preferencesEditor)));
        jSplitPane.setOneTouchExpandable(true);

        closeButton.addActionListener(e -> setVisible(false));

        // Set up the divider position
        jSplitPane.setDividerLocation(preferences.getInt("DIVIDER_POS", 200));
        jSplitPane.addPropertyChangeListener(evt -> {
            if (evt.getPropertyName().equals("dividerLocation")) {
                preferences.putInt("DIVIDER_POS", (Integer) evt.getNewValue());
                logger.config(MessageFormat.format("Preference changed: \"{0}\" {1}->{2}", evt.getPropertyName(), evt.getOldValue(), evt.getNewValue()));
            }
        });

        // Set up tree selection handler
        preferencesTree.addTreeSelectionListener(e -> {
            Preferences preferences1 = (Preferences) e.getNewLeadSelectionPath().getLastPathComponent();
            preferencesEditor.setModel(new PreferenceTableModel(preferences1));
        });

        add(jSplitPane, BorderLayout.CENTER);
        add(closeButton, BorderLayout.SOUTH);
    }
}
