/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.alpharogroup.model.util;

import java.io.Serializable;
import java.util.Objects;

import de.alpharogroup.model.IDetachable;
import de.alpharogroup.model.IObjectClassAwareModel;

/**
 * Base class for models that contain instances that do not appear to be serializable and cannot
 * thus use <code>Model</code> directly. Makes sure that the object set into the model can be
 * serialized.
 *
 * @author Timo Rantalaiho
 * @param <T>
 *            type of model object
 */
public abstract class GenericBaseModel<T> implements IObjectClassAwareModel<T>
{
	private static final long serialVersionUID = 1L;
	/** model object */
	private T object;

	/**
	 * @see de.alpharogroup.model.IModel#getObject()
	 */
	@Override
	public T getObject()
	{
		return object;
	}

	/**
	 * Set the model object. The contents must be serializable, as they are stored in the session
	 *
	 * @param object
	 *            the model object
	 * @see de.alpharogroup.model.IModel#setObject(Object)
	 */
	@Override
	public void setObject(T object)
	{
		if (!(object instanceof Serializable))
		{
			object = createSerializableVersionOf(object);
		}
		this.object = object;
	}

	/**
	 * Creates a serializable version of the object. The object is usually a collection.
	 *
	 * @param object the object
	 * @return serializable version of <code>object</code>
	 */
	protected abstract T createSerializableVersionOf(T object);

	/**
	 * @see de.alpharogroup.model.IDetachable#detach()
	 */
	@Override
	public void detach()
	{
		if (object instanceof IDetachable)
		{
			((IDetachable)object).detach();
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public Class<T> getObjectClass()
	{
		return object != null ? (Class<T>) object.getClass() : null;
	}

	/**
	 * @see Object#toString()
	 */
	@Override
	public String toString()
	{
	 final StringBuilder sb = new StringBuilder("Model:classname=[");
		sb.append(getClass().getName()).append(']');
		sb.append(":object=[").append(object).append(']');
		return sb.toString();
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode()
	{
		return Objects.hashCode(object);
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(final Object obj)
	{
		if (this == obj)
		{
			return true;

		}
		if (!(obj instanceof GenericBaseModel))
		{
			return false;
		}
		final GenericBaseModel<?> that = (GenericBaseModel<?>)obj;
		return Objects.equals(object, that.object);
	}
}
