/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.alpharogroup.auth;

import java.util.Date;
import java.util.Locale;
import java.util.Map;

import de.alpharogroup.random.RandomObjectsUtils;
import de.alpharogroup.auth.interfaces.Permission;
import de.alpharogroup.auth.interfaces.Role;
import de.alpharogroup.auth.interfaces.Session;
import de.alpharogroup.auth.interfaces.User;
import de.alpharogroup.collections.InsertionOrderMap;

/**
 * The Class UserSession.
 *
 * @version 1.0
 * @author Asterios Raptis
 */
public class UserSession implements Session<String, String>
{

	/**
	 * The serialVersionUID.
	 */
	private static final long serialVersionUID = 2858590853976767758L;

	/** The id. */
	private String id;

	/** The locale. */
	private Locale locale;

	/** The start time. */
	private Date startTime;

	/** The user. */
	private User<Permission, Role<Permission>> user;

	/** The last access. */
	private Date lastAccess;

	/** The max inactive time. */
	private int maxInactiveTime;

	/** The attributtes. */
	private Map<String, String> attributtes;

	/**
	 * Default constructor.
	 *
	 * @param user
	 *            The user
	 */
	public UserSession(final User<Permission, Role<Permission>> user)
	{
		if (null == user)
		{
			throw new IllegalArgumentException("User can't be null.");
		}
		this.user = user;
		this.initialize();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getAttributtes()
	 */
	@Override
	public Map<String, String> getAttributtes()
	{
		if (null == this.attributtes)
		{
			this.attributtes = new InsertionOrderMap<>();
		}
		return this.attributtes;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getId()
	 */
	@Override
	public String getId()
	{
		return this.id;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getLastAccess()
	 */
	@Override
	public Date getLastAccess()
	{
		return (Date)this.lastAccess.clone();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getLocale()
	 */
	@Override
	public Locale getLocale()
	{
		return this.locale;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getMaxInactiveTime()
	 */
	@Override
	public int getMaxInactiveTime()
	{
		return this.maxInactiveTime;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getStartTime()
	 */
	@Override
	public Date getStartTime()
	{
		return (Date)this.startTime.clone();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#getUser()
	 */
	@Override
	public User<Permission, Role<Permission>> getUser()
	{
		return this.user;
	}

	/**
	 * Initialie the UserSession object.
	 */
	private void initialize()
	{
		this.id = RandomObjectsUtils.newRandomId();
		this.startTime = new Date(System.currentTimeMillis());
		this.lastAccess = (Date)this.startTime.clone();
		this.maxInactiveTime = 180000;
		this.attributtes = new InsertionOrderMap<>();
		this.locale = Locale.getDefault();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setAttributtes(java.util.Map)
	 */
	@Override
	public void setAttributtes(final Map<String, String> attributtes)
	{
		this.attributtes = attributtes;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setId(java.lang.String)
	 */
	@Override
	public void setId(final String id)
	{
		this.id = id;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setLastAccess(java.util.Date)
	 */
	@Override
	public void setLastAccess(final Date lastAccess)
	{
		this.lastAccess = (Date)lastAccess.clone();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setLocale(java.util.Locale)
	 */
	@Override
	public void setLocale(final Locale locale)
	{
		this.locale = locale;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setMaxInactiveTime(int)
	 */
	@Override
	public void setMaxInactiveTime(final int maxInactiveTime)
	{
		this.maxInactiveTime = maxInactiveTime;
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setStartTime(java.util.Date)
	 */
	@Override
	public void setStartTime(final Date startTime)
	{
		this.startTime = (Date)startTime.clone();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @see de.alpharogroup.auth.interfaces.Session#setUser(de.alpharogroup.auth.interfaces.User)
	 */
	@Override
	public void setUser(final User<Permission, Role<Permission>> user)
	{
		this.user = user;
	}

	@Override
	public String getAttributte(String key)
	{
		return attributtes.get(key);
	}

	@Override
	public String setAttribute(String key, String value)
	{
		return attributtes.put(key, value);
	}

}
