/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.net.proxy;

import java.io.IOException;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * The Class BaseProxySelector holds a map with proxy addresses. It can replace
 * the default ProxySelector. Example:<br/>
 * <code> <br/> BaseProxySelector proxySelector = <br/>
 * new BaseProxySelector(ProxySelector.getDefault(), proxies);<br/>
 * ProxySelector.setDefault(proxySelector); <br/> </code><br/>
 * The argument proxies is the map with the map of the addresses to be used as
 * proxy.
 */
public class BaseProxySelector extends ProxySelector {

	/** A reference to the default ProxySelector. */
	private ProxySelector defaultSelector = null;

	/** A map of proxies. */
	private Map<SocketAddress, ProxyDecorator> proxies;

	/**
	 * Instantiates a new base proxy selector.
	 * 
	 * @param defaultSelector
	 *            the default selector
	 * @param proxies
	 *            the proxies
	 */
	public BaseProxySelector(final ProxySelector defaultSelector,
			final Map<SocketAddress, ProxyDecorator> proxies) {
		this.defaultSelector = defaultSelector;
		this.proxies = proxies;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void connectFailed(final URI uri, final SocketAddress sa,
			final IOException ioe) {
		if (uri == null || sa == null || ioe == null) {
			throw new IllegalArgumentException("Arguments can not be null.");
		}
		ProxyDecorator p = proxies.get(sa);
		if (p != null) {
			if (p.failed() >= 3) {
				proxies.remove(sa);
			}
		} else {
			if (defaultSelector != null) {
				defaultSelector.connectFailed(uri, sa, ioe);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Proxy> select(final URI uri) {
		if (uri == null) {
			throw new IllegalArgumentException("Arguments can not be null.");
		}
		String protocol = uri.getScheme();
		if ("http".equalsIgnoreCase(protocol)
				|| "https".equalsIgnoreCase(protocol)) {
			ArrayList<Proxy> proxyList = new ArrayList<Proxy>();
			for (ProxyDecorator p : proxies.values()) {
				proxyList.add(p.getProxy());
			}
			return proxyList;
		}
		if (defaultSelector != null) {
			return defaultSelector.select(uri);
		} else {
			ArrayList<Proxy> proxyList = new ArrayList<Proxy>();
			proxyList.add(Proxy.NO_PROXY);
			return proxyList;
		}
	}
}