package de.alpharogroup.locale;

import java.io.File;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import de.alpharogroup.file.namefilter.PropertiesResourceBundleFilenameFilter;
import lombok.experimental.UtilityClass;

/**
 * The Class {@link LocaleResolver} helps to resolve locale objects and languages.
 *
 * @deprecated use instead {@link LocaleResolver} from project resourcebundle-core.
 */
@UtilityClass
public class LocaleResolver
{

	/**
	 * Resolves all the available languages for the given resource bundle name in the given bundle
	 * package. Note the default resource bundle is excluded.
	 *
	 * @param bundlepackage
	 *            The package that contains the properties files.
	 * @param bundlename
	 *            The name of the resource bundle.
	 * @return a Set of String objects with the available languages excluding the default.
	 */
	public static Set<String> resolveAvailableLanguages(final String bundlepackage,
		final String bundlename)
	{
		final ClassLoader loader = Thread.currentThread().getContextClassLoader();
		final File root = new File(loader.getResource(bundlepackage.replace('.', '/')).getFile());
		final File[] files = root.listFiles(new PropertiesResourceBundleFilenameFilter(bundlename));

		final Set<String> languages = new TreeSet<>();
		for (final File file : files)
		{
			final String language = file.getName()
				.replaceAll("^" + bundlename + "(_)?|\\.properties$", "");
			if ((language != null) && !language.isEmpty())
			{
				languages.add(language);
			}
		}
		return languages;
	}

	/**
	 * Resolves the bundle name from the given properties file.
	 *
	 * @param propertiesFile
	 *            the properties file
	 * @return the bundle name
	 */
	public static String resolveBundlename(final File propertiesFile)
	{
		final String filename = propertiesFile.getName();
		final int indexOfUnderscore = filename.indexOf("_");
		String bundlename = filename;
		if (0 < indexOfUnderscore)
		{
			bundlename = propertiesFile.getName().substring(0, filename.indexOf("_"));
		}
		return bundlename;
	}

	/**
	 * Resolves the locale from the given properties file.
	 *
	 * @param propertiesFile
	 *            the properties file
	 * @return the locale
	 */
	public static Locale resolveLocale(final File propertiesFile)
	{
		final String localeString = propertiesFile.getName()
			.replaceAll("^" + resolveBundlename(propertiesFile) + "(_)?|\\.properties$", "");
		return LocaleResolver.resolveLocale(localeString);
	}

	/**
	 * Resolves the {@link Locale} object from the given properties file.
	 *
	 * @param localeCode
	 *            the locale code
	 * @return the {@link Locale} object.
	 */
	public static Locale resolveLocale(final String localeCode)
	{
		Locale current = null;
		if ((localeCode != null) && !localeCode.isEmpty())
		{
			final String[] splitted = localeCode.split("_");
			if (1 < splitted.length)
			{
				current = new Locale(splitted[0], splitted[1]);
			}
			else
			{
				current = new Locale(localeCode);
			}
		}
		else
		{
			current = Locale.getDefault();
		}
		return current;
	}

	/**
	 * Resolves all the available Locales to the given resource bundle name in the given bundle
	 * package. For now it is only for properties files.
	 *
	 * @param bundlepackage
	 *            The package that contains the properties files.
	 * @param bundlename
	 *            The name of the resource bundle.
	 * @return a Map of File objects with the corresponding Locales to it.
	 */
	public static Map<File, Locale> resolveLocales(final String bundlepackage, final String bundlename)
	{
		final ClassLoader loader = Thread.currentThread().getContextClassLoader();
		final File root = new File(loader.getResource(bundlepackage.replace('.', '/')).getFile());
		final File[] files = root.listFiles(new PropertiesResourceBundleFilenameFilter(bundlename));
		final Map<File, Locale> locales = new HashMap<>();
		for (final File file : files)
		{
			final Locale current = LocaleResolver.resolveLocale(file);
			locales.put(file, current);
		}
		return locales;
	}

}
