/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.tree;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import net.sourceforge.jaulp.tree.ifaces.ITreeNode;

/**
 * The generic class TreeNode.
 * 
 * @param <T>
 *            the generic type
 */
public class TreeNode<T> implements ITreeNode<T> {

	/**
	 * The serialVersionUID.
	 */
	private static final long serialVersionUID = 1L;

	/** The children. */
	private List<ITreeNode<T>> children;

	/** The parent from this node. If this is null it is the root. */
	private ITreeNode<T> parent;

	/** The value. */
	private T value;
	
	/** The optional display value. */
	private String displayValue;

	/**
	 * Instantiates a new tree node.
	 */
	public TreeNode() {
	}

	/**
	 * Instantiates a new tree node.
	 * 
	 * @param value
	 *            the value
	 */
	public TreeNode(final T value) {
		this();
		setValue(value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addChild(final ITreeNode<T> child) {
		if (children != null) {
			children.add(child);
		} else {
			children = new ArrayList<ITreeNode<T>>();
			children.add(child);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addChildAt(final int index, final ITreeNode<T> child)
			throws IndexOutOfBoundsException {
		if (children != null && children.size() < index) {
			children.add(index, child);
		} else {
			addChild(child);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(final ITreeNode<T> treeNode) {
		return treeNode.getValue().equals(treeNode);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getChildCount() {
		if (children == null) {
			return 0;
		}
		return children.size();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ITreeNode<T>> getChildren() {
		if (this.children == null) {
			return new ArrayList<ITreeNode<T>>();
		}
		return this.children;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ITreeNode<T> getParent() {
		return parent;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public T getValue() {
		return value;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasChildren() {
		return getChildren() != null && !getChildren().isEmpty();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((children == null) ? 0 : children.hashCode());
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasParent() {
		return parent != null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isLeaf() {
		return !isNode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isNode() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isRoot() {
		return !hasParent();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeChild(final ITreeNode<T> child) {
		if (children != null) {
			children.remove(child);
		} else {
			children = new ArrayList<ITreeNode<T>>();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeChildAt(final int index) throws IndexOutOfBoundsException {
		children.remove(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setChildren(final List<ITreeNode<T>> children) {
		this.children = children;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParent(final ITreeNode<T> parent) {
		this.parent = parent;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValue(final T value) {
		this.value = value;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ITreeNode<T>> toList() {
		final List<ITreeNode<T>> list = new ArrayList<ITreeNode<T>>();
		traverse(this, list);
		return list;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void traverse(final ITreeNode<T> node, final List<ITreeNode<T>> list) {
		list.add(node);
		for (final ITreeNode<T> data : node.getChildren()) {
			traverse(data, list);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ITreeNode<T> getNextSibling() {
		ITreeNode<T> parent = getParent();
		if (parent == null) {
			return null;
		}
		int index = parent.getChildren().indexOf(this) + 1;
		if (index == parent.getChildCount()) {
			return null;
		}
		return parent.getChildren().get(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ITreeNode<T> getPreviousSibling() {
		ITreeNode<T> parent = getParent();
		if (parent == null) {
			return null;
		}
		int index = parent.getChildren().indexOf(this) - 1;
		if (index < 0) {
			return null;
		}
		return parent.getChildren().get(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ITreeNode<T>> getAllSiblings() {
		ITreeNode<T> parent = getParent();
		if (parent == null) {
			return null;
		}
		List<ITreeNode<T>> siblings = new ArrayList<ITreeNode<T>>(
				parent.getChildren());
		siblings.remove(this);
		return siblings;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getLevel() {
		ITreeNode<T> current = this;
		int count = 0;
		while ((current = current.getParent()) != null) {
			count++;
		}
		return count;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getDepth() {
		if (isLeaf() || getChildCount() == 0) {
			return 0;
		}
		Stack<Integer> stack = new Stack<Integer>();
		stack.push(new Integer(0));
		ITreeNode<T> node = getChildren().get(0);
		int depth = 0;
		int current = 1;
		while (!stack.empty()) {
			if (node.getChildCount() != 0) {
				node = getChildren().get(0);
				stack.push(new Integer(0));
				current++;
			} else {
				if (current > depth) {
					depth = current;
				}
				int size;
				int index;
				do {
					node = node.getParent();
					size = node.getChildCount();
					index = ((Integer) stack.pop()).intValue() + 1;
					current--;
				} while (index >= size && node != this);

				if (index < size) {
					node = getChildren().get(index);
					stack.push(new Integer(index));
					current++;
				}
			}
		}
		return depth;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDisplayValue() {
		return displayValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDisplayValue(String displayValue) {
		this.displayValue = displayValue;
	}

}
