/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.zip;

import java.io.File;

import net.lingala.zip4j.core.ZipFile;
import net.lingala.zip4j.exception.ZipException;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.util.Zip4jConstants;

/**
 * The Class Zip4jUtils.
 */
public final class Zip4jUtils
{

	/**
	 * Adds a file or several files to the given zip file.
	 *
	 * @param zipFile4j
	 *            the zip file4j
	 * @param parameters
	 *            the parameters
	 * @param toAdd
	 *            the to add
	 * @throws ZipException
	 *             the zip exception
	 */
	public static void zipFiles(ZipFile zipFile4j, ZipParameters parameters, File... toAdd)
		throws ZipException
	{
		for (File element : toAdd)
		{
			zipFile4j.addFile(element, parameters);
		}
	}

	/**
	 * Adds a file or several files to the given zip file with the parameters
	 * Zip4jConstants.COMP_DEFLATE for the compression method and
	 * Zip4jConstants.DEFLATE_LEVEL_NORMAL as the compression level.
	 *
	 * @param zipFile4j
	 *            the zip file4j
	 * @param toAdd
	 *            the to add
	 * @throws ZipException
	 *             the zip exception
	 */
	public static void zipFiles(ZipFile zipFile4j, File... toAdd) throws ZipException
	{
		zipFiles(zipFile4j, Zip4jConstants.COMP_DEFLATE, Zip4jConstants.DEFLATE_LEVEL_NORMAL, toAdd);
	}

	/**
	 * Adds a file or several files to the given zip file with the given parameters for the
	 * compression method and the compression level.
	 *
	 * @param zipFile4j
	 *            the zip file4j
	 * @param compressionMethod
	 *            The compression method
	 * @param compressionLevel
	 *            the compression level
	 * @param toAdd
	 *            the to add
	 * @throws ZipException
	 *             the zip exception
	 */
	public static void zipFiles(ZipFile zipFile4j, int compressionMethod, int compressionLevel,
		File... toAdd) throws ZipException
	{
		// Initiate Zip Parameters which define various properties such
		// as compression method, etc.
		ZipParameters parameters = new ZipParameters();

		// set compression method to store compression
		// Zip4jConstants.COMP_STORE is for no compression
		// Zip4jConstants.COMP_DEFLATE is for compression
		parameters.setCompressionMethod(compressionMethod);

		// Set the compression level
		// DEFLATE_LEVEL_ULTRA = ultra maximum compression
		// DEFLATE_LEVEL_MAXIMUM = maximum compression
		// DEFLATE_LEVEL_NORMAL = normal compression
		// DEFLATE_LEVEL_FAST = fast compression
		// DEFLATE_LEVEL_FASTEST = fastest compression
		parameters.setCompressionLevel(compressionLevel);
		zipFiles(zipFile4j, parameters, toAdd);
	}

	/**
	 * Extract.
	 *
	 * @param zipFile4j
	 *            the zip file4j
	 * @param destination
	 *            the destination
	 * @param password
	 *            the password
	 * @throws ZipException
	 *             the zip exception
	 */
	public static void extract(ZipFile zipFile4j, File destination, String password)
		throws ZipException
	{
		if (zipFile4j.isEncrypted())
		{
			zipFile4j.setPassword(password);
		}
		zipFile4j.extractAll(destination.getAbsolutePath());
	}

}
